unit syscloneunit;

{$mode objfpc}{$H+}

interface

uses
  sysutils, ctypes, syscall;

type
  TThreadFunc = function(Arg: Pointer): PtrInt; cdecl;

  TSysClone = class
  private
    FThreadFunc: TThreadFunc;
    FArg: Pointer;
    FThreadID: clong;
    FStack: Pointer;
    FStackSize: csize_t;
    function CreateStack: Boolean;
    procedure FreeStack;
  public
    constructor Create(ThreadFunc: TThreadFunc; Arg: Pointer; StackSize: csize_t = 1024 * 1024);
    destructor Destroy; override;
    function Start: Boolean;
    function Join: PtrInt;
  end;

implementation

uses
  Unix, BaseUnix;

const
  CLONE_VM       = $00000100;
  CLONE_FS       = $00000200;
  CLONE_FILES    = $00000400;
  CLONE_SIGHAND  = $00000800;
  CLONE_PARENT   = $00008000;
  CLONE_THREAD   = $00010000;
  CLONE_IO       = $80000000;

  THREAD_FLAGS   = CLONE_VM or CLONE_FS or CLONE_FILES or CLONE_SIGHAND or CLONE_PARENT or CLONE_THREAD or CLONE_IO;

  PROT_READ      = $1;
  PROT_WRITE     = $2;

  MAP_ANONYMOUS  = $20;
  MAP_PRIVATE    = $2;
  MAP_GROWSDOWN  = $100;

  SYS_CLONE      = 56;

type
  TThreadData = record
    Func: TThreadFunc;
    Arg: Pointer;
  end;
  PThreadData = ^TThreadData;

function ThreadWrapper(Data: Pointer): PtrInt; cdecl;
var
  ThreadData: PThreadData;
begin
  WriteLn('ThreadWrapper: Starting thread...');
  ThreadData := PThreadData(Data);
  Result := ThreadData^.Func(ThreadData^.Arg);
  WriteLn('ThreadWrapper: Thread finished with result: ', Result);
end;

constructor TSysClone.Create(ThreadFunc: TThreadFunc; Arg: Pointer; StackSize: csize_t);
begin
  WriteLn('TSysClone.Create: Initializing thread...');
  inherited Create;
  FThreadFunc := ThreadFunc;
  FArg := Arg;
  FStackSize := StackSize;
  if not CreateStack then
    raise Exception.Create('Failed to allocate stack');
  WriteLn('TSysClone.Create: Stack allocated successfully');
end;

destructor TSysClone.Destroy;
begin
  WriteLn('TSysClone.Destroy: Freeing resources...');
  FreeStack;
  inherited Destroy;
  WriteLn('TSysClone.Destroy: Resources freed');
end;

function TSysClone.CreateStack: Boolean;
begin
  WriteLn('TSysClone.CreateStack: Allocating stack of size ', FStackSize, ' bytes...');
  FStack := fpmmap(nil, FStackSize, PROT_READ or PROT_WRITE, MAP_ANONYMOUS or MAP_PRIVATE or MAP_GROWSDOWN, -1, 0);
  Result := FStack <> MAP_FAILED;
  if Result then
    WriteLn('TSysClone.CreateStack: Stack allocated at address ', HexStr(@PtrUInt(FStack)))
  else
    WriteLn('TSysClone.CreateStack: Failed to allocate stack');
end;

procedure TSysClone.FreeStack;
begin
  if FStack <> nil then
  begin
    WriteLn('TSysClone.FreeStack: Freeing stack at address ', HexStr(@PtrUInt(FStack)));
    fpmunmap(FStack, FStackSize);
    FStack := nil;
  end;
end;

function TSysClone.Start: Boolean;
var
  StackTop: Pointer;
  ThreadData: PThreadData;
begin
  WriteLn('TSysClone.Start: Preparing to start thread...');
  // Устанавливаем указатель на верх стека (стек растет вниз)
  StackTop := Pointer(PtrUInt(FStack) + FStackSize);
  // Выравниваем стек по 16 байт
  StackTop := Pointer((PtrUInt(StackTop) and not $F));
  WriteLn('TSysClone.Start: Stack top at address ', HexStr(@PtrUInt(StackTop)));

  // Передаем аргументы через структуру
  ThreadData := StackTop - SizeOf(TThreadData);
  ThreadData^.Func := FThreadFunc;
  ThreadData^.Arg := FArg;
  WriteLn('TSysClone.Start: Thread data at address ', HexStr(@PtrUInt(ThreadData)));

  // Запускаем поток
  WriteLn('TSysClone.Start: Calling clone syscall...');
  FThreadID := do_SysCall(SYS_CLONE, THREAD_FLAGS, Int64(StackTop - SizeOf(TThreadData)), 0, 0, 0);
  Result := FThreadID <> -1;
  if Result then
    WriteLn('TSysClone.Start: Thread started with ID ', FThreadID)
  else
    WriteLn('TSysClone.Start: Failed to start thread');
end;

function TSysClone.Join: PtrInt;
var
  Status: cint;
begin
  WriteLn('TSysClone.Join: Waiting for thread ', FThreadID, ' to finish...');
  fpwaitpid(FThreadID, @Status, 0);
  Result := WEXITSTATUS(Status);
  WriteLn('TSysClone.Join: Thread finished with status ', Result);
end;

end.