unit syscloneunit;

{$mode objfpc}{$H+}

interface

uses
  sysutils, ctypes, syscall, Unix, BaseUnix;

type
  TThreadFunc = function(Arg: Pointer): PtrInt; cdecl;

  TSysClone = class
  private
    FThreadFunc: TThreadFunc;
    FArg: Pointer;
    FThreadID: clong;
    FStack: Pointer;
    FStackSize: csize_t;
    FThreadData: Pointer;
    function CreateStack: Boolean;
    procedure FreeStack;
  public
    constructor Create(ThreadFunc: TThreadFunc; Arg: Pointer; StackSize: csize_t = 1024 * 1024);
    destructor Destroy; override;
    function Start: Boolean;
    function Join: PtrInt;
  end;

implementation

const
  CLONE_VM       = $00000100;
  CLONE_FS       = $00000200;
  CLONE_FILES    = $00000400;
  CLONE_SIGHAND  = $00000800;
  CLONE_PARENT   = $00008000;
  CLONE_THREAD   = $00010000;
  CLONE_IO       = $80000000;

  THREAD_FLAGS   = CLONE_VM or CLONE_FS or CLONE_FILES or CLONE_SIGHAND or CLONE_PARENT or CLONE_THREAD or CLONE_IO;

  PROT_READ      = $1;
  PROT_WRITE     = $2;

  MAP_ANONYMOUS  = $20;
  MAP_PRIVATE    = $2;

  SYS_CLONE      = 56;

type
  PThreadData = ^TThreadData;
  TThreadData = record
    Func: TThreadFunc;
    Arg: Pointer;
  end;

function ThreadWrapper(Data: Pointer): PtrInt; cdecl;
var
  ThreadData: PThreadData;
begin
  WriteLn('ThreadWrapper: Starting thread...');
  ThreadData := PThreadData(Data);
  Result := ThreadData^.Func(ThreadData^.Arg);
  WriteLn('ThreadWrapper: Thread finished with result: ', Result);
end;

constructor TSysClone.Create(ThreadFunc: TThreadFunc; Arg: Pointer; StackSize: csize_t);
begin
  WriteLn('TSysClone.Create: Initializing thread...');
  inherited Create;
  FThreadFunc := ThreadFunc;
  FArg := Arg;
  FStackSize := StackSize;
  FThreadData := nil;
  if not CreateStack then
    raise Exception.Create('Failed to allocate stack');
  WriteLn('TSysClone.Create: Stack allocated successfully');
end;

destructor TSysClone.Destroy;
begin
  WriteLn('TSysClone.Destroy: Freeing resources...');
  FreeStack;
  if FThreadData <> nil then
  begin
    FreeMem(FThreadData);
    FThreadData := nil;
  end;
  inherited Destroy;
  WriteLn('TSysClone.Destroy: Resources freed');
end;

function TSysClone.CreateStack: Boolean;
begin
  WriteLn('TSysClone.CreateStack: Allocating stack of size ', FStackSize, ' bytes...');
  FStack := fpmmap(nil, FStackSize, PROT_READ or PROT_WRITE, MAP_ANONYMOUS or MAP_PRIVATE, -1, 0);
  Result := FStack <> MAP_FAILED;
  if Result then
    WriteLn('TSysClone.CreateStack: Stack allocated at address ', HexStr(@PtrUInt(FStack)))
  else
    WriteLn('TSysClone.CreateStack: Failed to allocate stack');
end;

procedure TSysClone.FreeStack;
begin
  if FStack <> nil then
  begin
    WriteLn('TSysClone.FreeStack: Freeing stack at address ', HexStr(@PtrUInt(FStack)));
    fpmunmap(FStack, FStackSize);
    FStack := nil;
  end;
end;

function TSysClone.Start: Boolean;
var
  StackTop: Pointer;
  ThreadData: PThreadData;
begin
  WriteLn('TSysClone.Start: Preparing to start thread...');
  StackTop := Pointer(PtrUInt(FStack) + FStackSize);
  StackTop := Pointer(PtrUInt(StackTop) and not $F);
  WriteLn('TSysClone.Start: Stack top at address ', HexStr(@PtrUInt(StackTop)));

  // Выделяем память для ThreadData
  WriteLn('TSysClone.Start: Allocating memory for thread data...');
  GetMem(FThreadData, SizeOf(TThreadData));
  ThreadData := PThreadData(FThreadData);
  ThreadData^.Func := FThreadFunc;
  ThreadData^.Arg := FArg;
  WriteLn('TSysClone.Start: Thread data stored at ', HexStr(@PtrUInt(ThreadData)));

  // Запускаем поток через clone()
  WriteLn('TSysClone.Start: Calling clone syscall...');
  FThreadID := do_SysCall(SYS_CLONE, THREAD_FLAGS, Int64(StackTop), Int64(@ThreadWrapper), Int64(ThreadData), 0);
  Result := FThreadID <> -1;

  if Result then
    WriteLn('TSysClone.Start: Thread started with ID ', FThreadID)
  else
    WriteLn('TSysClone.Start: Failed to start thread');
end;

function TSysClone.Join: PtrInt;
var
  Status: cint;
begin
  WriteLn('TSysClone.Join: Waiting for thread ', FThreadID, ' to finish...');
  fpwaitpid(FThreadID, @Status, 0);
  Result := WEXITSTATUS(Status);
  WriteLn('TSysClone.Join: Thread finished with status ', Result);
end;

end.