#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <math.h>
#include <time.h>
#include <gdk/x11/gdkx.h>
#include <X11/Xatom.h>

#define CLOCK_RADIUS 100
#define WINDOW_SIZE (CLOCK_RADIUS * 2 + 20)

typedef struct {
    GtkWidget *window;
    GtkWidget *drawing_area;
    gboolean maximized;
} AppWidgets;

void set_keep_above(GtkWidget *widget) {
    GdkSurface *surface = gtk_native_get_surface(GTK_NATIVE(widget));
    if (GDK_IS_X11_SURFACE(surface)) {
        Display *dpy = GDK_SURFACE_XDISPLAY(surface);
        Window win = GDK_SURFACE_XID(surface);

        Atom wm_state = XInternAtom(dpy, "_NET_WM_STATE", False);
        Atom above = XInternAtom(dpy, "_NET_WM_STATE_ABOVE", False);

        XEvent e = {0};
        e.xclient.type = ClientMessage;
        e.xclient.message_type = wm_state;
        e.xclient.display = dpy;
        e.xclient.window = win;
        e.xclient.format = 32;
        e.xclient.data.l[0] = 1;  // _NET_WM_STATE_ADD
        e.xclient.data.l[1] = above;
        e.xclient.data.l[2] = 0;
        e.xclient.data.l[3] = 0;
        e.xclient.data.l[4] = 0;

        XSendEvent(dpy, DefaultRootWindow(dpy), False,
                   SubstructureRedirectMask | SubstructureNotifyMask, &e);
    }
}

static void on_draw(GtkDrawingArea *area, cairo_t *cr, int width, int height, gpointer user_data) {
    AppWidgets *app = (AppWidgets *)user_data;

    int center_x = width / 2;
    int center_y = height / 2;

    cairo_set_source_rgba(cr, 0, 0, 0, 0);
    cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
    cairo_paint(cr);

    cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
    cairo_set_source_rgb(cr, 0.8, 0.8, 0.8);
    cairo_arc(cr, center_x, center_y, CLOCK_RADIUS, 0, 2 * M_PI);
    cairo_fill(cr);

    cairo_set_source_rgb(cr, 0, 0, 0);
    cairo_select_font_face(cr, "Sans", CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_BOLD);
    cairo_set_font_size(cr, 14);
    for (int i = 1; i <= 12; i++) {
        double angle = (i - 3) * M_PI / 6.0;
        int x = center_x + cos(angle) * (CLOCK_RADIUS - 20);
        int y = center_y + sin(angle) * (CLOCK_RADIUS - 20);
        char buf[3];
        snprintf(buf, sizeof(buf), "%d", i);
        cairo_text_extents_t extents;
        cairo_text_extents(cr, buf, &extents);
        cairo_move_to(cr, x - extents.width / 2, y + extents.height / 2);
        cairo_show_text(cr, buf);
    }

    time_t t = time(NULL);
    struct tm *tm = localtime(&t);

    // Minute hand (blue)
    double minute_angle = (tm->tm_min + tm->tm_sec / 60.0) * M_PI / 30.0 - M_PI / 2.0;
    cairo_set_source_rgb(cr, 0.0, 0.0, 1.0);
    cairo_set_line_width(cr, 3);
    cairo_move_to(cr, center_x, center_y);
    cairo_line_to(cr, center_x + cos(minute_angle) * (CLOCK_RADIUS - 30), center_y + sin(minute_angle) * (CLOCK_RADIUS - 30));
    cairo_stroke(cr);

    // Hour hand (black)
    double hour_angle = ((tm->tm_hour % 12) + tm->tm_min / 60.0) * M_PI / 6.0 - M_PI / 2.0;
    cairo_set_source_rgb(cr, 0.0, 0.0, 0.0);
    cairo_set_line_width(cr, 5);
    cairo_move_to(cr, center_x, center_y);
    cairo_line_to(cr, center_x + cos(hour_angle) * (CLOCK_RADIUS - 50), center_y + sin(hour_angle) * (CLOCK_RADIUS - 50));
    cairo_stroke(cr);

    // Second hand (red)
    double second_angle = tm->tm_sec * M_PI / 30.0 - M_PI / 2.0;
    cairo_set_source_rgb(cr, 1.0, 0.0, 0.0);
    cairo_set_line_width(cr, 1);
    cairo_move_to(cr, center_x, center_y);
    cairo_line_to(cr, center_x + cos(second_angle) * (CLOCK_RADIUS - 20), center_y + sin(second_angle) * (CLOCK_RADIUS - 20));
    cairo_stroke(cr);
}

static void on_click(GtkGestureClick *gesture, int n_press, double x, double y, gpointer user_data) {
    AppWidgets *app = user_data;
    if (n_press == 2) {
        if (!app->maximized) {
            gtk_window_fullscreen(GTK_WINDOW(app->window));
            app->maximized = TRUE;
        } else {
            gtk_window_unfullscreen(GTK_WINDOW(app->window));
            app->maximized = FALSE;
        }
    }
}

static void on_right_click(GtkGestureClick *gesture, int n_press, double x, double y, gpointer user_data) {
    if (gtk_gesture_single_get_current_button(GTK_GESTURE_SINGLE(gesture)) == GDK_BUTTON_SECONDARY) {
        gtk_window_close(GTK_WINDOW(((AppWidgets *)user_data)->window));
    }
}

static void on_drag_drag_begin(GtkGestureDrag *gesture, double start_x, double start_y, gpointer user_data) {
    AppWidgets *app = user_data;
    GdkSurface *surface = gtk_native_get_surface(GTK_NATIVE(app->window));
    gdk_toplevel_begin_move(GDK_TOPLEVEL(surface), NULL, 1, (int)start_x, (int)start_y, GDK_BUTTON_PRIMARY);
}

static gboolean on_tick(gpointer user_data) {
    AppWidgets *app = (AppWidgets *)user_data;
    gtk_widget_queue_draw(app->drawing_area);
    return G_SOURCE_CONTINUE;
}

static void activate(GtkApplication *application, gpointer user_data) {
    AppWidgets *app = (AppWidgets *)user_data;

    app->window = gtk_application_window_new(application);
    gtk_window_set_decorated(GTK_WINDOW(app->window), FALSE);
    gtk_window_set_default_size(GTK_WINDOW(app->window), WINDOW_SIZE, WINDOW_SIZE);
    gtk_window_set_resizable(GTK_WINDOW(app->window), FALSE);
    gtk_window_set_title(GTK_WINDOW(app->window), "glock");

//    gtk_window_set_keep_above(GTK_WINDOW(app->window), TRUE);
set_keep_above(app->window);

    GtkCssProvider *provider = gtk_css_provider_new();
    gtk_css_provider_load_from_string(provider,
        "* { background-color: transparent; }");
    gtk_style_context_add_provider_for_display(gdk_display_get_default(),
        GTK_STYLE_PROVIDER(provider),
        GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
    g_object_unref(provider);

    GtkWidget *area = gtk_drawing_area_new();
    gtk_widget_set_size_request(area, WINDOW_SIZE, WINDOW_SIZE);
    gtk_widget_set_focusable(area, TRUE);
    gtk_drawing_area_set_draw_func(GTK_DRAWING_AREA(area), on_draw, app, NULL);

    GtkGesture *click_gesture = gtk_gesture_click_new();
    gtk_gesture_single_set_button(GTK_GESTURE_SINGLE(click_gesture), 0);
    g_signal_connect(click_gesture, "pressed", G_CALLBACK(on_click), app);
    g_signal_connect(click_gesture, "pressed", G_CALLBACK(on_right_click), app);
    gtk_widget_add_controller(area, GTK_EVENT_CONTROLLER(click_gesture));

    GtkGesture *drag_gesture = gtk_gesture_drag_new();
    g_signal_connect(drag_gesture, "drag-begin", G_CALLBACK(on_drag_drag_begin), app);
    gtk_widget_add_controller(area, GTK_EVENT_CONTROLLER(drag_gesture));

    gtk_window_set_child(GTK_WINDOW(app->window), area);
    app->drawing_area = area;

    gtk_widget_set_visible(app->window, TRUE);
    g_timeout_add_seconds(1, on_tick, app);
}

int main(int argc, char **argv) {
    GtkApplication *app_instance = gtk_application_new("com.example.glock", G_APPLICATION_DEFAULT_FLAGS);
    AppWidgets app = {0};
    g_signal_connect(app_instance, "activate", G_CALLBACK(activate), &app);
    int status = g_application_run(G_APPLICATION(app_instance), argc, argv);
    g_object_unref(app_instance);
    return status;
}