#ifndef TTA_CODEC_TTA_H
#define TTA_CODEC_TTA_H
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// codec/tta.h                                                              //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// Copyright (C) 2007, Aleksander Djuric                                    //
// Copyright (C) 2023-2024, Shane Seelig                                    //
// SPDX-License-Identifier: GPL-3.0-or-later                                //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include <assert.h>
#include <stddef.h>	// size_t

#include "../bits.h"

#include "common.h"
#include "filter.h"

//////////////////////////////////////////////////////////////////////////////

enum LibTTAr_RetVal {
	// frame finished
	LIBTTAr_RET_DONE	 =  0,

	// frame not finished
	LIBTTAr_RET_AGAIN	 =  1,

	// frame finished, but (nbytes_tta_total != nbytes_tta_perframe)
	//||
	// frame not finished, but (nbytes_tta_total > nbytes_tta_perframe)
	LIBTTAr_RET_DECFAIL      =  2,

	// (ni32_target % nchan != 0) or other bad parameter
	// used as the base value, can return greater values
	LIBTTAr_RET_INVAL,

	// library was misconfigured; see libttaR_test_nchan
	LIBTTAr_RET_MISCONFIG	 = -1
};

// max unary r/w size:		read		write
//	8/16-bit:		   8194u	   8200u
//	  24-bit:		2097154uL	2097160uL
// max binary r/w size:		      3u	      0u
// max cacheflush w size: 		  	      8u
#define TTABUF_SAFETY_MARGIN_1_2	((size_t)    8208u)
#define TTABUF_SAFETY_MARGIN_3		((size_t) 2097168uL)

//////////////////////////////////////////////////////////////////////////////

INLINE CONST size_t get_safety_margin(enum TTASampleBytes, uint) /*@*/;
INLINE CONST bitcnt get_predict_k(enum TTASampleBytes) /*@*/;
INLINE CONST i32 get_filter_round(enum TTASampleBytes) /*@*/;
INLINE CONST bitcnt get_filter_k(enum TTASampleBytes) /*@*/;

//--------------------------------------------------------------------------//

ALWAYS_INLINE CONST i32 asr32(i32, bitcnt) /*@*/;

//--------------------------------------------------------------------------//

ALWAYS_INLINE CONST i32 tta_predict1(i32, bitcnt) /*@*/;
ALWAYS_INLINE CONST u32 tta_postfilter_enc(i32) /*@*/;
ALWAYS_INLINE CONST i32 tta_prefilter_dec(u32) /*@*/;

//////////////////////////////////////////////////////////////////////////////

/**@fn get_safety_margin
 * @brief safety margin for the TTA buffer
 *
 * @param samplebytes number of bytes per PCM sample
 *
 * @return safety margin
**/
INLINE CONST size_t
get_safety_margin(const enum TTASampleBytes samplebytes, const uint nchan)
/*@*/
{
	size_t r;
	switch ( samplebytes ){
	case TTASAMPLEBYTES_1:
	case TTASAMPLEBYTES_2:
		r = TTABUF_SAFETY_MARGIN_1_2;
		break;
	case TTASAMPLEBYTES_3:
		r = TTABUF_SAFETY_MARGIN_3;
		break;
	}
	return (size_t) (r * nchan);
}

/**@fn get_predict_k
 * @brief arg for tta_predict1
 *
 * @param samplebytes number of bytes per PCM sample
 *
 * @return arg 'k' for tta_predict1
**/
INLINE CONST bitcnt
get_predict_k(const enum TTASampleBytes samplebytes)
/*@*/
{
	bitcnt r;
	switch ( samplebytes ){
	case TTASAMPLEBYTES_1:
		r = (bitcnt) 4u;
		break;
	case TTASAMPLEBYTES_2:
	case TTASAMPLEBYTES_3:
		r = (bitcnt) 5u;
		break;
	}
	return r;
}

/**@fn get_filter_round
 * @brief arg for tta_filter
 *
 * @param samplebytes number of bytes per PCM sample
 *
 * @return arg 'round' for tta_filter
**/
INLINE CONST i32
get_filter_round(const enum TTASampleBytes samplebytes)
/*@*/
{
	i32 r;
	switch ( samplebytes ){
	case TTASAMPLEBYTES_1:
	case TTASAMPLEBYTES_3:
		r = (i32) 0x00000200;	// binexp32(filter_k - 1u)
		break;
	case TTASAMPLEBYTES_2:
		r = (i32) 0x00000100;	// ~
		break;
	}
	return r;
}

/**@fn get_filter_k
 * @brief arg for tta_filter
 *
 * @param samplebytes number of bytes per PCM sample
 *
 * @return arg 'k' for tta_filter
**/
INLINE CONST bitcnt
get_filter_k(const enum TTASampleBytes samplebytes)
/*@*/
{
	bitcnt r;
	switch ( samplebytes ){
	case TTASAMPLEBYTES_1:
	case TTASAMPLEBYTES_3:
		r = (bitcnt) 10u;
		break;
	case TTASAMPLEBYTES_2:
		r = (bitcnt)  9u;
		break;
	}
	return r;
}

//==========================================================================//

/**@fn asr32
 * @brief arithmetic shift right 32-bit
 *
 * @param x value to shift
 * @param k amount to shift
 *
 * @return shifted value
 *
 * @pre k <= (bitcnt) 31u
**/
ALWAYS_INLINE CONST i32
asr32(const i32 x, const bitcnt k)
/*@*/
{
	assert(k <= (bitcnt) 31u);

	if ( HAS_ASR(i32) ){
		/*@-shiftimplementation@*/
		return (i32) (x >> k);
		/*@=shiftimplementation@*/
	}
	else {	return (UNPREDICTABLE (x < 0)
			? (i32) ~((~((u32) x)) >> k) : (i32) (((u32) x) >> k)
		);
	}
}

//==========================================================================//

/**@fn tta_predict1
 * @brief fixed order 1 prediction
 *
 * @param x input value
 * @param k how much to shift it by
 *
 * @return predicted value
 *
 * @pre k <= (bitcnt) 32u
**/
ALWAYS_INLINE CONST i32
tta_predict1(const i32 x, const bitcnt k)
/*@*/
{
	assert(k <= (bitcnt) 32u);

	return (i32) (((((u64f) x) << k) - x) >> k);
}

/**@fn tta_postfilter_enc
 * @brief interleave value for coding
 *        0 => 0
 *        1 => 1
 *       -1 => 2
 *        2 => 3
 *       -2 => 4
 *
 * @param x input value
 *
 * @return interleaved value
 *
 * @note https://en.wikipedia.org/wiki/Golomb_coding#Overview#\
 *     Use%20with%20signed%20integers
 * @note affected by LIBTTAr_OPT_PREFER_CONDITIONAL_MOVES
**/
ALWAYS_INLINE CONST u32
tta_postfilter_enc(const i32 x)
/*@*/
{
#ifndef LIBTTAr_OPT_PREFER_CONDITIONAL_MOVES
	const u32 y     = (u32) -x;
	const u32 xsign = (u32) asr32((i32) y, (bitcnt) 31u);
	return (u32) ((y << 1u) ^ xsign);
#else
	const u32 yp = (u32) x, yn = (u32) -x;
	return (UNPREDICTABLE (x > 0)
		? (u32) ((yp << 1u) - 1u) : (u32) (yn << 1u)
	);
#endif
}

/**@fn tta_prefilter_dec
 * @brief deinterleave value for filtering
 *        0 =>  0
 *        1 =>  1
 *        2 => -1
 *        3 =>  2
 *        4 => -2
 *
 * @param x input value
 *
 * @return deinterleaved value
 *
 * @see tta_postfilter_enc
 * @note affected by LIBTTAr_OPT_PREFER_CONDITIONAL_MOVES
**/
ALWAYS_INLINE CONST i32
tta_prefilter_dec(const u32 x)
/*@*/
{
#ifndef LIBTTAr_OPT_PREFER_CONDITIONAL_MOVES
	const u32 xsign = (u32) -((i32) (x & 0x1u));
	return -((i32) ((x >> 1u) ^ xsign));
#else
	const i32 y = (i32) x;
	return (UNPREDICTABLE ((((u32) x) & 0x1u) != 0)
		? asr32(y + 1, (bitcnt) 1u) : asr32(-y, (bitcnt) 1u)
	);
#endif
}

// EOF ///////////////////////////////////////////////////////////////////////
#endif
