unit syscloneunit;

{$mode objfpc}{$H+}

interface

uses
  sysutils,ctypes;

type
  TThreadFunc = function(Arg: Pointer): Pointer;

  TSysClone = class
  private
    FThreadFunc: TThreadFunc;
    FArg: Pointer;
    FThreadID: clong;
    FStack: Pointer;
    FStackSize: csize_t;
    function CreateStack: Boolean;
    procedure FreeStack;
  public
    constructor Create(ThreadFunc: TThreadFunc; Arg: Pointer; StackSize: csize_t = 1024 * 1024);
    destructor Destroy; override;
    function Start: Boolean;
    function Join: Pointer;
  end;

implementation

uses
  syscall, BaseUnix;

const
  CLONE_VM       = $00000100;
  CLONE_FS       = $00000200;
  CLONE_FILES    = $00000400;
  CLONE_SIGHAND  = $00000800;
  CLONE_PARENT   = $00008000;
  CLONE_THREAD   = $00010000;
  CLONE_IO       = $80000000;

  THREAD_FLAGS   = CLONE_VM or CLONE_FS or CLONE_FILES or CLONE_SIGHAND or CLONE_PARENT or CLONE_THREAD or CLONE_IO;

  PROT_READ      = $1;
  PROT_WRITE     = $2;
  PROT_EXEC      = $4;

  MAP_ANONYMOUS  = $20;
  MAP_PRIVATE    = $2;
  MAP_GROWSDOWN  = $100;

constructor TSysClone.Create(ThreadFunc: TThreadFunc; Arg: Pointer; StackSize: csize_t);
begin
  inherited Create;
  FThreadFunc := ThreadFunc;
  FArg := Arg;
  FStackSize := StackSize;
  if not CreateStack then
    raise Exception.Create('Failed to allocate stack');
end;

destructor TSysClone.Destroy;
begin
  FreeStack;
  inherited Destroy;
end;

function TSysClone.CreateStack: Boolean;
begin
  FStack := fpmmap(nil, FStackSize, PROT_READ or PROT_WRITE, MAP_ANONYMOUS or MAP_PRIVATE or MAP_GROWSDOWN, -1, 0);
  Result := FStack <> MAP_FAILED;
end;

procedure TSysClone.FreeStack;
begin
  if FStack <> nil then
    fpmunmap(FStack, FStackSize);
end;

function TSysClone.Start: Boolean;
var
  StackTop: Pointer;
begin
  // Устанавливаем указатель на верх стека (стек растет вниз)
  StackTop := FStack + FStackSize;
  // Выравниваем стек по 16 байт
  StackTop := Pointer((PtrUInt(StackTop) and not $F));
  // Передаем аргументы через стек
  PPointer(StackTop - SizeOf(Pointer))^ := FArg;
  PPointer(StackTop - 2 * SizeOf(Pointer))^ := FThreadFunc;
  // Запускаем поток
  FThreadID := syscall(syscall_nr_clone, THREAD_FLAGS, StackTop - 2 * SizeOf(Pointer), nil, nil, nil);
  Result := FThreadID <> -1;
end;

function TSysClone.Join: Pointer;
begin
  fpwaitpid(FThreadID, @Result, 0);
end;

end.