program AdvancedChatAI;
{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}{$CODEPAGE UTF8}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

uses
  SysUtils, Classes, PostProcessor, Unix, BaseUnix,
  Transformer, MatrixOps, TextEmbeddings,LazUTF8, ucs4unit,
  ucs4opunit, ucs4functionsunit, Word2Vec,DataUtils,Math,
  TrainerUnit,DateUtils, {DataAugmentation,} HyperparameterOptimizer,
  ContextManagerUnit, DecoderUnit, SymbolicEngineUnit, ContextProcessing;

const
  TEMP_DIR = 'temp_attachments/';
  POST_FILE = 'post.txt';
  POST_FILES_LIST = 'postfiles.txt';
  WORD2VEC_MODEL = 'word2vec.model';

var
  UserInput: TUserInput;
  ShouldExit: Boolean = False;

  IsClassifierInitialized: Boolean = False;

  LastMessage: string;

  ResponseTemplates: array of record
    Patterns: array of string;
    Response: string;
  end;

  CommonWordsCache: TStringList;

  ConversationHistory: TStringList;
  LastUserMessage: string = '';
  ContextWindow: Integer = 3; // Количество сообщений в контексте

  LastAIResponse: string = '';
  LastTrainingTime: TDateTime = 0;

procedure QuickTraining;
var
  TrainingData: TTrainingDataset;
  startTime: TDateTime;
  initialLoss, finalLoss: Double;

 valLoss:Double;
 trainData, valData: TTrainingDataset;
begin
  if not FileExists('training_data.txt') then
  begin
    WriteLn('Файл training_data.txt не найден, пропускаем обучение');
    WriteLn('Создайте файл с примерами диалогов для обучения');
    Exit;
  end;

  WriteLn('=== БЫСТРОЕ ОБУЧЕНИЕ ТРАНСФОРМЕРА ===');
  startTime := Now;
  
  // Загружаем данные
  LoadTrainingData(TrainingData, 'training_data.txt');
  FilterEmptyExamples(TrainingData);
  
  if Length(TrainingData) = 0 then
  begin
    WriteLn('Нет данных для обучения');
    Exit;
  end;

  WriteLn('Загружено примеров: ', Length(TrainingData));
  
  try
    // Оцениваем начальные потери
    initialLoss := EvaluateModel(TransformerModel, TrainingData);
    WriteLn('Начальные потери: ', initialLoss:0:4);
    
    // Обучаем с индексацией
    WriteLn('Запуск обучения...');
{
//    TrainTransformerWithIndexing(TransformerModel, TrainingData, 5, 0.0001); // 5 эпох
TrainTransformerWithIndexing(TransformerModel, TrainingData, 10, 0.0001); // 10 эпох вместо 5
    
    // Оцениваем конечные потери
    finalLoss := EvaluateModel(TransformerModel, TrainingData);
    WriteLn('Конечные потери: ', finalLoss:0:4);
    WriteLn('Улучшение: ', (initialLoss - finalLoss):0:4);
}
SplitDataset(TrainingData, trainData, valData, 0.8); // 80% обучение, 20% валидация
WriteLn('Данные разделены: ', Length(trainData), ' обучение, ', Length(valData), ' валидация');
// Обучаем на trainData
TrainTransformerWithIndexing(TransformerModel, trainData, 15, 0.0001);
// Проверяем на valData
valLoss := EvaluateModel(TransformerModel, valData);
WriteLn('Потери на валидации: ', valLoss:0:4);
    
    WriteLn('Обучение завершено за ', MilliSecondsBetween(Now, startTime), ' мс');
    
    // Сохраняем обученную модель
    SaveModel(TransformerModel, 'trained_model.bin');
    WriteLn('Модель сохранена в trained_model.bin');
    
  except
    on E: Exception do
    begin
      WriteLn('Ошибка обучения: ', E.Message);
      WriteLn('Продолжаем без обученной модели');
    end;
  end;
end;

procedure LoadTrainedModel;
var sampleLoss:Double;
begin
  if FileExists('trained_model.bin') then
  begin
    WriteLn('Обнаружена обученная модель, загружаем...');
    try
      LoadModel(TransformerModel, 'trained_model.bin');
      WriteLn('Обученная модель загружена успешно!');
      
      // Проверяем качество загруженной модели
      sampleLoss := EvaluateModel(TransformerModel, []);
      WriteLn('Модель готова к работе');
      
    except
      on E: Exception do
      begin
        WriteLn('Ошибка загрузки модели: ', E.Message);
        WriteLn('Продолжаем с случайными весами');
      end;
    end;
  end
  else
  begin
    WriteLn('Обученная модель не найдена, используем случайные веса');
  end;
end;

procedure TestTrainedModel;
var
  testInputs: TUC4Array;
  testOutputs,inputMatrix: TDoubleMatrix;
  i: Integer;
  response: string;
begin
  if not IsTransformerInitialized then
  begin
    WriteLn('Модель не инициализирована для тестирования');
    Exit;
  end;

  WriteLn('=== ТЕСТИРОВАНИЕ ОБУЧЕННОЙ МОДЕЛИ ===');
  
  SetLength(testInputs, 5);
  testInputs[0] := 'привет';
  testInputs[1] := 'как дела?';
  testInputs[2] := 'что ты умеешь?';
  testInputs[3] := 'сколько будет 2+2?';
  testInputs[4] := 'пока';

  for i := 0 to High(testInputs) do
  begin
    try
      WriteLn('Тест ', i+1, ': "', testInputs[i].ToUTF8, '"');
      
      inputMatrix := CreateContextEnrichedInput(testInputs[i], '');
      ForwardTransformer(TransformerModel, inputMatrix, testOutputs);
      
      if Assigned(ResponseGenerator) then
        response := ResponseGenerator.GenerateResponse(testOutputs, '', testInputs[i].ToUTF8)
      else
        response := 'Ответ не сгенерирован';
        
      WriteLn('   Ответ: ', response);
      WriteLn('   ---');
      
    except
      on E: Exception do
        WriteLn('   Ошибка: ', E.Message);
    end;
  end;
end;

procedure IncrementalTraining;
var
  TrainingData: TTrainingDataset;
begin
  if not FileExists('auto_learning_data.txt') then Exit;

  try
    LoadTrainingData(TrainingData, 'auto_learning_data.txt');
    FilterEmptyExamples(TrainingData);
    
    if Length(TrainingData) > 5 then // Только если есть новые данные
    begin
      WriteLn('Дообучение на ', Length(TrainingData), ' новых примерах...');
      TrainTransformerWithIndexing(TransformerModel, TrainingData, 1, 0.00001);
      WriteLn('Дообучение завершено');
    end;
  except
    on E: Exception do
      WriteLn('Ошибка инкрементального обучения: ', E.Message);
  end;
end;

procedure PreloadFrequentWords(Embeddings: TWordEmbeddings);
const FrequentWords: array[0..102] of string = (
    'привет', 'здравствуйте', 'пока', 'спасибо', 'да', 'нет', 'как', 'что',
    'где', 'когда', 'почему', 'какой', 'какая', 'какое', 'какие', 'кто',
    'чем', 'чего', 'ему', 'его', 'её', 'им', 'их', 'нам', 'вам', 'им',
    'меня', 'тебя', 'его', 'её', 'нас', 'вас', 'их', 'мне', 'тебе', 'ему',
    'ей', 'нам', 'вам', 'им', 'мной', 'тобой', 'им', 'ей', 'нами', 'вами',
    'ими', 'себя', 'себе', 'собой', 'что', 'кто', 'как', 'где', 'куда',
    'откуда', 'когда', 'почему', 'зачем', 'какой', 'какая', 'какое', 'какие',
    'чей', 'чья', 'чьё', 'чьи', 'сколько', 'насколько', 'столько', 'который',
    'которая', 'которое', 'которые', 'каков', 'какова', 'каково', 'каковы',
    'отчего', 'зачем', 'ли', 'не', 'ни', 'бы', 'же', 'ведь', 'вот', 'вон',
    'вот', 'даже', 'именно', 'только', 'лишь', 'исключительно', 'почти',
    'очень', 'самый', 'более', 'менее', 'почти', 'совсем', 'чуть', 'едва'
  );
var
  i: Integer;
  startTime: TDateTime;
emb:TDoubleArray;
begin
  if Embeddings = nil then
  begin
    WriteLn('Предупреждение: WordEmbeddings не инициализирован');
    Exit;
  end;

  startTime := Now;
  WriteLn('Начало предзагрузки частых слов...');

  for i := 0 to High(FrequentWords) do begin
    try
      // ✅ БЕЗОПАСНАЯ ПРЕДЗАГРУЗКА
      Emb := Embeddings.GetEmbedding(FrequentWords[i]); // Простой метод без кэша
      if Length(Emb) > 0 then
      begin
        // Добавляем в кэш вручную если нужно
        // Embeddings.PutToCache(FrequentWords[i], Emb);
      end;

    except
      on E: Exception do
      begin
        WriteLn('  Ошибка предзагрузки слова "', FrequentWords[i], '": ', E.Message);
        // Продолжаем со следующими словами
      end;
    end;
  end;

  WriteLn('Предзагрузка завершена за ', MilliSecondsBetween(Now, startTime), ' мс');
end;

procedure HandleSigInt(sig: cint); cdecl;
begin
Halt;
end;

procedure ShowWelcome;
begin
  WriteLn('=== Advanced Chat AI ===');
  WriteLn('Версия с Transformer и классификатором текста');
  WriteLn('Нажмите Ctrl+C для выхода');
  WriteLn;
end;

procedure SaveBestConfig(const Config: TBestConfig; const Filename: string);
var
  F: TextFile;
begin
  AssignFile(F, Filename);
  Rewrite(F);
  WriteLn(F, 'learning_rate=', Config.LearningRate:0:6);
  WriteLn(F, 'batch_size=', Config.BatchSize);
  WriteLn(F, 'num_heads=', Config.NumHeads);
  WriteLn(F, 'ffn_dim=', Config.FFNDim);
  WriteLn(F, 'num_layers=', Config.NumLayers);
  WriteLn(F, 'dropout_rate=', Config.DropoutRate:0:2);
  WriteLn(F, 'weight_decay=', Config.WeightDecay:0:6);
  CloseFile(F);
end;

procedure OptimizeHyperparameters;
var
  TrainingData: TTrainingDataset;
  HyperConfig: THyperparameterConfig;
  BestConfig: TBestConfig;
  FinalConfig: TTransformerConfig;
begin
  WriteLn('=== HYPERPARAMETER OPTIMIZATION ===');

  // Загружаем данные
  LoadTrainingData(TrainingData, 'training_data.txt');
  WriteLn('Loaded ', Length(TrainingData), ' training examples');

  // Настраиваем конфигурацию поиска
  HyperConfig := CreateDefaultHyperparameterConfig;
  HyperConfig.MaxEpochs := 5; // Меньше эпох для быстрого поиска

  // Запускаем оптимизацию
  BestConfig := FindBestHyperparameters(TrainingData, HyperConfig);

  // Выводим результаты
  WriteLn('');
  WriteLn('=== BEST CONFIGURATION ===');
  WriteLn('Learning Rate: ', BestConfig.LearningRate:0:6);
  WriteLn('Batch Size: ', BestConfig.BatchSize);
  WriteLn('Number of Heads: ', BestConfig.NumHeads);
  WriteLn('FFN Dimension: ', BestConfig.FFNDim);
  WriteLn('Number of Layers: ', BestConfig.NumLayers);
  WriteLn('Dropout Rate: ', BestConfig.DropoutRate:0:2);
  WriteLn('Weight Decay: ', BestConfig.WeightDecay:0:6);
  WriteLn('Validation Loss: ', BestConfig.ValidationLoss:0:4);

  // Создаем финальную конфигурацию
with FinalConfig do begin
  InputSize := 300;
  NumLayers := BestConfig.NumLayers;
  NumHeads := BestConfig.NumHeads;
  FFNDim := BestConfig.FFNDim;
  MaxSeqLength := 1000; // Увеличиваем до 1000
  DropoutRate := BestConfig.DropoutRate;
  WeightDecay := BestConfig.WeightDecay;
  GradientClipValue := 1.0;
  UseLayerNorm := True;
end;

  // Переинициализируем модель с лучшими параметрами
  FreeTransformer(TransformerModel);
  InitTransformer(TransformerModel, FinalConfig);

  // Обучаем окончательно на всех данных
  WriteLn('Training final model with optimal parameters...');
//  TrainTransformerWithRegularization(TransformerModel, TrainingData, 50, BestConfig.LearningRate);

  // Сохраняем оптимальную конфигурацию
  SaveBestConfig(BestConfig, 'best_config.cfg');
end;

procedure SaveTrainingTime(Time: TDateTime);
var
  F: TextFile;
begin
  AssignFile(F, 'last_training.time');
  try
    Rewrite(F);
    WriteLn(F, DateTimeToStr(Time));
  finally
    CloseFile(F);
  end;
end;

function LoadTrainingTime: TDateTime;
var
  F: TextFile;
  TimeStr: string;
begin
  if not FileExists('last_training.time') then
    Exit(0); // Возвращаем дату по умолчанию

  AssignFile(F, 'last_training.time');
  try
    Reset(F);
    ReadLn(F, TimeStr);
    Result := StrToDateTimeDef(TimeStr, 0);
  except
    Result := 0;
  end;
  CloseFile(F);
end;

function GetConversationContext: string;
var
  i: Integer;
begin
  Result := '';
  WriteLn('GetConversationContext: история содержит ', ConversationHistory.Count, ' сообщений');
  
  for i := 0 to ConversationHistory.Count - 1 do
  begin
    WriteLn('  Сообщение ', i, ': ', Copy(ConversationHistory[i], 1, 50));
    Result := Result + ConversationHistory[i] + #10;
  end;
  
  Result := Result.Trim;
  WriteLn('GetConversationContext: итоговый контекст: "', Copy(Result, 1, 100), '"');
end;

function CalculateResponseQuality(const UserInput, AIResponse: string): Double;
var
  score: Double;
  inputLower, responseLower: string;
begin
  inputLower := UTF8LowerCase(UserInput);
  responseLower := UTF8LowerCase(AIResponse);

  score := 0.5; // Базовый балл

  // Награда за релевантность
  if responseLower.Contains('помощь') and inputLower.Contains('помощ') then score := score + 0.2;
  if responseLower.Contains('вопрос') and inputLower.Contains('что') then score := score + 0.2;
  if responseLower.Contains('обсуд') and inputLower.Contains('тема') then score := score + 0.2;

  // Награда за длину (не слишком коротко, не слишком длинно)
  if (Length(AIResponse) > 20) and (Length(AIResponse) < 100) then score := score + 0.1;

  // Штраф за повторения
  if responseLower = 'я вас слушаю' then score := score - 0.1;

  Result := Max(0.1, Min(1.0, score)); // Ограничиваем от 0.1 до 1.0
end;

procedure AutoLearnFromGoodResponses;
var
  LogFile: TStringList;
  i: Integer;
  UserInput, AIResponse: string;
  Quality: Double;
  TrainingFile: TextFile;
begin
  if not FileExists('conversation_log.txt') then Exit;

  LogFile := TStringList.Create;
  try
    LogFile.LoadFromFile('conversation_log.txt');

    AssignFile(TrainingFile, 'auto_learning_data.txt');
    Append(TrainingFile);

    for i := 0 to LogFile.Count - 1 do begin
      if LogFile[i].StartsWith('USER: ') then
        UserInput := Copy(LogFile[i], 7, MaxInt)
      else if LogFile[i].StartsWith('AI: ') then
      begin
        AIResponse := Copy(LogFile[i], 5, MaxInt);

        // Оцениваем качество ответа
        Quality := CalculateResponseQuality(UserInput, AIResponse);

        if Quality > 0.7 then // Хорошие ответы
        begin
          WriteLn(TrainingFile, 'В: ', UserInput);
          WriteLn(TrainingFile, 'О: ', AIResponse);
          WriteLn(TrainingFile, 'К: ');
          WriteLn(TrainingFile, '---');
          WriteLn('Авто-обучение: добавлен хороший пример (качество: ', Quality:0:2, ')');
        end;
      end;
    end;

    CloseFile(TrainingFile);

  finally
    LogFile.Free;
  end;
end;

type
  TMemoryInfo = record
    Total: Int64;
    Available: Int64;
  end;

// Если GetMemoryInfo не доступен, используем простую версию
function GetMemoryInfo(var MemInfo: TMemoryInfo): Boolean;
begin
  try
    // Читаем информацию из /proc/meminfo
    with TStringList.Create do
    try
      LoadFromFile('/proc/meminfo');
      MemInfo.Total := StrToInt64Def(Values['MemTotal:'], 0) * 1024;
      MemInfo.Available := StrToInt64Def(Values['MemAvailable:'], 0) * 1024;
      Result := (MemInfo.Total > 0) and (MemInfo.Available > 0);
    finally
      Free;
    end;
  except
    Result := False;
  end;
end;

function GetSystemMemory: Integer;
var
  MemInfo: TMemoryInfo;
begin
  // Получаем информацию о памяти системы
  if GetMemoryInfo(MemInfo) then
  begin
    Result := MemInfo.Available; // Доступная память в байтах
    WriteLn('Доступно памяти: ', Result div (1024*1024), ' MB');
  end
  else
  begin
    // Fallback значение если не удалось получить информацию
    Result := 512 * 1024 * 1024; // 512 MB по умолчанию
    WriteLn('Не удалось получить информацию о памяти, используем ', 
            Result div (1024*1024), ' MB по умолчанию');
  end;
end;

procedure ScheduledRetraining(var Model: TTransformer);
var
  AutoLearningData: TTrainingDataset;
  optimalBatchSize: Integer;
  systemMemory: Integer;
begin
  // Загружаем время последнего обучения
  LastTrainingTime := LoadTrainingTime;

  // Если никогда не обучались или прошло больше 24 часов
  if (LastTrainingTime = 0) or (HoursBetween(Now, LastTrainingTime) > 24) then begin
    WriteLn('Запуск ежедневного переобучения...');

    // Загружаем авто-собранные данные
    if FileExists('auto_learning_data.txt') then begin
      LoadTrainingData(AutoLearningData, 'auto_learning_data.txt');

      if Length(AutoLearningData) > 0 then begin
        WriteLn('Обучение на ', Length(AutoLearningData), ' новых примерах...');

        // Определяем оптимальный размер пакета
        systemMemory := GetSystemMemory;
        optimalBatchSize := CalculateOptimalBatchSize(systemMemory, Model.Config.InputSize);

        // Обучаем с пакетной обработкой
        TrainEpochWithBatches(Model, AutoLearningData, 0.001, optimalBatchSize);

        WriteLn('Переобучение завершено!');
      end
      else
      begin
        WriteLn('Нет новых данных для переобучения');
      end;
    end
    else
    begin
      WriteLn('Файл auto_learning_data.txt не найден');
    end;

    // Сохраняем время обучения
    SaveTrainingTime(Now);
  end
  else
  begin
    WriteLn('До следующего переобучения: ', 
            24 - HoursBetween(Now, LastTrainingTime), ' часов');
  end;
end;

procedure LearnFromConversation(const UserInput, AIResponse: string);
var
  F: TextFile;
  Context: string;
begin
  Context := GetConversationContext;

  // Сохраняем в основной файл обучения
  AssignFile(F, 'training_data.txt');
  Append(F);
  try
    WriteLn(F, 'В: ', UserInput);
    WriteLn(F, 'О: ', AIResponse);
    WriteLn(F, 'К: ', Context);
    WriteLn(F, '---');
    WriteLn(F, '');
  finally
    CloseFile(F);
  end;

  // Также сохраняем в авто-обучение
  AssignFile(F, 'auto_learning_data.txt');
  Append(F);
  try
    WriteLn(F, 'В: ', UserInput);
    WriteLn(F, 'О: ', AIResponse);
    WriteLn(F, 'К: ', Context);
    WriteLn(F, '---');
    WriteLn(F, '');
  finally
    CloseFile(F);
  end;

  WriteLn('Диалог сохранен для обучения');
end;

procedure InteractiveLearning(var Model: TTransformer);
var
  UserFeedback: string;
  ShouldSave: Boolean;
begin
  WriteLn('');
  WriteLn('=== ОБРАТНАЯ СВЯЗЬ ===');
  WriteLn('Понравился ли вам мой ответ?');
  WriteLn('1 - Да, сохранить как хороший пример');
  WriteLn('2 - Нет, предложить лучший ответ');
  WriteLn('3 - Пропустить');
  Write('Ваш выбор (1/2/3): ');

  ReadLn(UserFeedback);

  case UserFeedback of
    '1': 
      begin
        LearnFromConversation(LastUserMessage, LastAIResponse);
        WriteLn('Спасибо! Ответ сохранен для обучения.');
      end;

    '2':
      begin
        WriteLn('Как бы вы ответили лучше?');
        Write('Ваш вариант: ');
        ReadLn(UserFeedback);

        if UserFeedback <> '' then
        begin
          LearnFromConversation(LastUserMessage, UserFeedback);
          WriteLn('Спасибо! Я научился новому ответу.');
        end;
      end;

    '3':
      WriteLn('Обратная связь пропущена.');

  else
    WriteLn('Неверный выбор. Обратная связь не сохранена.');
  end;
end;

procedure MonitorTrainingProgress(var Model: TTransformer);
var
  TestInputs: TUC4Array;
  TestOutputs: TDoubleMatrix;
  i: Integer;
  AvgResponseLength: Double;
begin
  SetLength(TestInputs, 5);
  TestInputs[0] := 'привет';
  TestInputs[1] := 'как дела?';
  TestInputs[2] := 'что ты умеешь?';
  TestInputs[3] := 'кто ты?';
  TestInputs[4] := 'расскажи о себе';

  WriteLn('Тестирование прогресса обучения:');

  for i := 0 to High(TestInputs) do
  begin
    try
      ForwardTransformer(Model, TextsToMatrix([TestInputs[i]], 300), TestOutputs);
      WriteLn('  ', TestInputs[i].ToUTF8, ' -> OK');
    except
      on E: Exception do
        WriteLn('  ', TestInputs[i].ToUTF8, ' -> Error: ', E.Message);
    end;
  end;
end;

procedure UpdateConversationContext(const UserMessage, AIResponse: string);
begin
  // Добавляем сообщение пользователя
  if UserMessage <> '' then
    ConversationHistory.Add('USER: ' + UserMessage);

  // Добавляем ответ AI
  if AIResponse <> '' then
    ConversationHistory.Add('AI: ' + AIResponse);

  // Ограничиваем размер истории
  while ConversationHistory.Count > ContextWindow * 2 do // *2 потому что пары user+ai
    ConversationHistory.Delete(0);

  LastUserMessage := UserMessage;
end;

function ParseEntitiesFromOutput(const OutputMatrix: TDoubleMatrix): TStringArray;
begin
  // Используем функцию из ContextProcessing
  Result := DetectEntitiesInOutput(OutputMatrix, WordEmbeddings.FVocab);
end;

function BuildSymbolicContext(const History: TStringList): string;
begin
  // Используем функцию из ContextProcessing
  Result := BuildEnhancedContext(History, LastUserMessage);
end;

function GenerateTransformerResponse(const inputText: ucs4; 
  const outputMatrix: TDoubleMatrix): string;
var
  questionStr, context: string;
begin
  questionStr := UTF8LowerCase(inputText.ToUTF8);
  context := GetConversationContext; // Получаем контекст здесь

  WriteLn('Debug: Контекст для генерации: ', context);

  // Анализируем контекст
  if context.Contains('как дела') and questionStr.Contains('хорош') then
    Result := 'Отлично! Рад, что у вас всё хорошо. '
  else if context.Contains('погод') then
    Result := 'Продолжая тему погоды: '
  else if context.Contains('имя') or context.Contains('зовут') then
    Result := 'Как я уже говорил, я - чат-бот. '
  else
    Result := '';

  // Базовые ответы
  if Length(outputMatrix) > 0 then
  begin
    if Result = '' then
      Result := 'Интересно! ';

    case Random(4) of
      0: Result := Result + 'Что вы думаете об этом?';
      1: Result := Result + 'Хотите обсудить это подробнее?';
      2: Result := Result + 'Есть ли у вас еще вопросы?';
      3: Result := Result + 'Чем еще могу помочь?';
    end;
  end
  else
  begin
    Result := 'Не совсем понял. Можете уточнить?';
  end;
end;

procedure LoadResponseTemplates;
begin
  SetLength(ResponseTemplates, 6);

  // Приветствие
  ResponseTemplates[0].Patterns := ['привет', 'здравствуй', 'добрый день', 'хай', 'hello', 'hi'];
  ResponseTemplates[0].Response := 'Здравствуйте! Чем могу помочь?';

  // Вопрос о делах
  ResponseTemplates[1].Patterns := ['как дела', 'как жизнь', 'как сам', 'как ты', 'how are you'];
  ResponseTemplates[1].Response := 'У меня всё отлично! А у вас как дела?';

  // Прощание
  ResponseTemplates[2].Patterns := ['пока', 'до свидания', 'выход', 'закончить', 'bye', 'goodbye'];
  ResponseTemplates[2].Response := 'До свидания! Буду рад пообщаться снова.';

  // Благодарность
  ResponseTemplates[3].Patterns := ['спасибо', 'благодарю', 'мерси', 'thanks', 'thank you'];
  ResponseTemplates[3].Response := 'Пожалуйста! Обращайтесь ещё.';

  // Вопрос о возможностях
  ResponseTemplates[4].Patterns := ['что ты умеешь', 'твои возможности', 'какие функции'];
  ResponseTemplates[4].Response := 'Я могу общаться на разные темы, отвечать на вопросы и поддерживать беседу!';

  // Вопрос о имени
  ResponseTemplates[5].Patterns := ['как тебя зовут', 'твое имя', 'who are you'];
  ResponseTemplates[5].Response := 'Я - чат-бот с искусственным интеллектом!';
end;

function ShouldSaveForTraining(const userMessage, aiResponse: string): Boolean;
begin
  // Сохраняем для обучения если:
  // 1. Сообщение не слишком короткое
  // 2. Ответ не шаблонный
  // 3. Не приветствие/прощание
  Result := (Length(userMessage) > 5) and
            (Length(aiResponse) > 10) and
            (not userMessage.Contains('привет')) and
            (not userMessage.Contains('пока')) and
            (not aiResponse.Contains('Здравствуйте')) and
            (not aiResponse.Contains('До свидания'));
end;

function GetFallbackResponse(const userMessage: string): string;
const
  FallbackResponses: array[0..5] of string = (
    'Извините, я не совсем понял ваш вопрос. Можете переформулировать?',
    'Интересный вопрос! К сожалению, я need больше контекста.',
    'Благодарю за сообщение. Чем еще могу помочь?',
    'Я все еще учусь. Можете объяснить по-другому?',
    'Это выходит за рамки моих текущих возможностей.',
    'Давайте обсудим что-то другое. Что вас интересует?'
  );
begin
  // Анализируем сообщение для выбора подходящего fallback
  if userMessage.Contains('?') then
    Result := FallbackResponses[0] // Для вопросов
  else if userMessage.Contains('!') then
    Result := FallbackResponses[2] // Для восклицаний
  else
    Result := FallbackResponses[Random(Length(FallbackResponses))];
end;

procedure FullHyperparameterOptimization;
begin
  if not FileExists('training_data.txt') then
  begin
    WriteLn('Файл training_data.txt не найден');
    WriteLn('Создайте файл с данными обучения в формате:');
    WriteLn('В: вопрос');
    WriteLn('О: ответ');
    WriteLn('К: контекст');
    WriteLn('---');
    Exit;
  end;

  WriteLn('=== ПОЛНАЯ ОПТИМИЗАЦИЯ ГИПЕРПАРАМЕТРОВ ===');
  WriteLn('Это может занять несколько минут...');
  
  try
    OptimizeHyperparameters;
    WriteLn('Оптимизация завершена!');
  except
    on E: Exception do
      WriteLn('Ошибка оптимизации: ', E.Message);
  end;
end;

procedure Initialize;
var 
  defaultConfig: TTransformerConfig;
  TrainingData: TTrainingDataset;
  i: Integer;
begin
  try
    if FileExists(WORD2VEC_MODEL) then
    begin
      WriteLn('Loading Word2Vec model...');
      WordEmbeddings := TWordEmbeddings.Create(WORD2VEC_MODEL, 5000);
      
      // ✅ ПРОВЕРКА УСПЕШНОСТИ ЗАГРУЗКИ
      if (WordEmbeddings.FVocab = nil) or (WordEmbeddings.FVocab.Count = 0) then
      begin
        WriteLn('ОШИБКА: Word2Vec модель загружена, но словарь пуст!');
        WordEmbeddings.Free;
        WordEmbeddings := nil;
      end
      else
      begin
        Word2Vec.WordEmbeddings := WordEmbeddings;
        WriteLn('Word2Vec model successfully loaded');
        WriteLn('Vocabulary size: ', WordEmbeddings.FVocab.Count);
        WriteLn('Embedding size: ', WordEmbeddings.EmbeddingSize);

        // Предзагрузка частых слов
        PreloadFrequentWords(WordEmbeddings);
      end;
    end
    else
    begin
      WriteLn('Warning: Word2Vec model file not found: ', WORD2VEC_MODEL);
      WordEmbeddings := nil;
      Word2Vec.WordEmbeddings := nil;
    end;
  except
    on E: Exception do
    begin
      WriteLn('Error loading Word2Vec: ', E.Message);
      WordEmbeddings := nil;
      Word2Vec.WordEmbeddings := nil;
    end;
  end;

// После загрузки Word2Vec
if Assigned(WordEmbeddings) then
begin
  ResponseGenerator := TResponseGenerator.Create(WordEmbeddings);
  WriteLn('Декодер ответов инициализирован');
end
else
begin
  WriteLn('Внимание: Word2Vec не загружен, декодер будет ограничен');
  ResponseGenerator := TResponseGenerator.Create(nil);
end;

  // ✅ ПРОДОЛЖАЕМ ДАЖЕ ЕСЛИ Word2Vec НЕ ЗАГРУЗИЛСЯ
  WriteLn('WordEmbeddings status: ', IfThen(Assigned(WordEmbeddings), 'LOADED', 'NOT AVAILABLE'));

  LoadResponseTemplates;

  // Создаем временную директорию для вложений
  if not DirectoryExists(TEMP_DIR) then
    ForceDirectories(TEMP_DIR);

  // Инициализация истории диалога
  ConversationHistory := TStringList.Create;
  ConversationHistory.Delimiter := '|';
  ConversationHistory.StrictDelimiter := True;
  LastUserMessage := '';

  // Загружаем и фильтруем данные
  LoadTrainingData(TrainingData, 'training_data.txt');
  WriteLn('Loaded ', Length(TrainingData), ' examples before filtering');
  FilterEmptyExamples(TrainingData);
  WriteLn('After filtering: ', Length(TrainingData), ' examples');

  // ✅ СОЗДАЕМ КОНФИГУРАЦИЮ
  with defaultConfig do
  begin
    InputSize := 300;
    NumLayers := 2;
    NumHeads := 4;
    FFNDim := 512;
    MaxSeqLength := 1000;
    DropoutRate := 0.1;
    WeightDecay := 0.0001;
    GradientClipValue := 1.0;
    UseLayerNorm := True;
  end;

  WriteLn('Инициализация модели трансформера...');
  InitTransformer(TransformerModel, defaultConfig);

  // ✅ ПРОВЕРЯЕМ ИНИЦИАЛИЗАЦИЮ
  WriteLn('Модель инициализирована: InputSize=', TransformerModel.Config.InputSize);

  if not ValidateModelStructure(TransformerModel) then
  begin
    WriteLn('КРИТИЧЕСКАЯ ОШИБКА: Модель не прошла валидацию!');
    // Но продолжаем работу - возможно, это ложная тревога
  end;

// После инициализации модели
WriteLn('Модель инициализирована: InputSize=', TransformerModel.Config.InputSize);

// ✅ БЫСТРОЕ ОБУЧЕНИЕ ПРИ СТАРТЕ
QuickTraining;

LoadTrainedModel;

TestTrainedModel;

if not ValidateModelStructure(TransformerModel) then
begin
  WriteLn('КРИТИЧЕСКАЯ ОШИБКА: Модель не прошла валидацию!');
end;

//SimpleHyperparameterTuning;
//FullHyperparameterOptimization; // долго

  // Добавляем символьный движок
  SymbolicEngine := TSymbolicEngine.Create;
  
  // Загружаем базу знаний если есть
  if FileExists('knowledge_base.txt') then
    SymbolicEngine.LoadFromFile('knowledge_base.txt');

end;

function ExtractFactsWithTransformer(const Text: string): TFactArray;
var
  tokens: TUC4Array;
  i, factCount: Integer;
  lowerText: string;
begin
  SetLength(Result, 0);
  if Text = '' then Exit;
  
  lowerText := UTF8LowerCase(Text);
  tokens := TokenizeForNLP(Text);
  
  factCount := 0;
  SetLength(Result, 2); // Максимум 2 факта из короткого сообщения
  
  // Определенные паттерны для извлечения фактов
  if ContainsAny(lowerText, ['москв', 'питер', 'город']) then
  begin
    Result[factCount].Subject := 'мы';
    Result[factCount].Relation := 'находится_в';
    
    if lowerText.Contains('москв') then
      Result[factCount]._Object := 'Москва'
    else if lowerText.Contains('питер') then
      Result[factCount]._Object := 'Санкт-Петербург'
    else
      Result[factCount]._Object := 'городе';
      
    Result[factCount].Confidence := 0.8;
    Inc(factCount);
  end;
  
  if lowerText.Contains('нормальн') or lowerText.Contains('хорош') then
  begin
    Result[factCount].Subject := 'состояние';
    Result[factCount].Relation := 'является';
    Result[factCount]._Object := 'хорошим';
    Result[factCount].Confidence := 0.7;
    Inc(factCount);
  end;
  
  SetLength(Result, factCount);
  WriteLn('Извлечено фактов: ', factCount);
end;

function GenerateHybridResponse(const UserMessage, RuleResult, Context: string): string;
var
outputMatrix: TDoubleMatrix;
begin
  // Комбинируем подходы
  
  // 1. Если символьная система уверена - используем её
  if RuleResult = 'Greeting' then
    Result := 'Здравствуйте! Чем могу помочь?'
  else if RuleResult = 'Farewell' then
    Result := 'До свидания! Рад был пообщаться.'
  else if RuleResult = 'Question' then
  begin
    // Для вопросов используем комбинацию
    if Context.Contains('погод') then
      Result := 'К сожалению, у меня нет доступа к актуальным данным о погоде. '
    else
      Result := 'Интересный вопрос! ';
    
    // Добавляем нейросетевую вариативность
    Result := Result + GenerateTransformerResponse(UserMessage,outputMatrix);
  end
  else
    // По умолчанию - нейросеть
    Result := GenerateTransformerResponse(UserMessage,outputMatrix);
end;

function GenerateResponseFromSymbolicSystem(const SymbolicResult, UserMessage, Context: string): string;
var
outputMatrix: TDoubleMatrix;
begin
  // Генерируем ответ на основе символьного результата и контекста
  case SymbolicResult of
    'IdentityQuery': 
      begin
        if Context.Contains('ранее') or Context.Contains('уже') then
          Result := 'Как я уже говорил, я - чат-бот с искусственным интеллектом. ' +
                   'Мои ответы генерируются на основе нейросетевой модели.'
        else
          Result := 'Я - чат-бот с искусственным интеллектом, созданный для общения и помощи. ' +
                   'Можно называть меня "Помощник". А как вас зовут?';
      end;
    'CapabilityQuery':
      Result := 'Я могу: отвечать на вопросы, поддерживать беседу, помогать с идеями. ' +
               'Чем конкретно могу помочь?';
    'WhoQuestion':
      Result := 'Интересный вопрос о личности! ' + GenerateTransformerResponse(UserMessage, outputMatrix);
    'WhatQuestion':
      Result := 'Хороший вопрос о сущностях! ' + GenerateTransformerResponse(UserMessage, outputMatrix);
    'Greeting': 
      begin
        if Context.Contains('ранее') then
          Result := 'Снова здравствуйте! Чем могу помочь?'
        else
          Result := 'Здравствуйте! Рад вас видеть. Чем могу помочь?';
      end;
    'Farewell': 
      Result := 'До свидания! Буду рад пообщаться снова.';
    'Thanks':
      Result := 'Пожалуйста! Обращайтесь, если понадобится помощь.';
    'Question':
      begin
        if Context.Contains('погод') then
          Result := 'Относительно погоды: ' + GenerateTransformerResponse(UserMessage, outputMatrix)
        else if Context.Contains('математик') then
          Result := 'Для математического вопроса: ' + GenerateTransformerResponse(UserMessage, outputMatrix)
        else
          Result := 'Интересный вопрос! ' + GenerateTransformerResponse(UserMessage, outputMatrix);
      end;
    'WeatherQuery':
      Result := 'К сожалению, у меня нет доступа к актуальным данным о погоде. ' +
                'Но могу обсудить погоду в общем смысле!';
    'NameQuery':
      Result := 'Я - чат-бот с искусственным интеллектом. ' +
                'Можно просто называть меня "помощник".';
    'MathQuery':
      Result := 'Я могу помочь с математическими вопросами! ' +
                'Задайте конкретный пример для расчета.';
    else
      Result := GenerateTransformerResponse(UserMessage, outputMatrix);
  end;
end;

procedure EnhancedProcessUserInput;
var
  inputText: ucs4;
  userMessage, symbolicResult, finalResponse, enhancedContext: string;
  filesList: TStringList;
  inputMatrix, outputMatrix: TDoubleMatrix;
  extractedFacts: TFactArray;
  i: Integer;
begin
  filesList := TStringList.Create;
  try
    // 1. Получаем пользовательский ввод
    UserInput := GetUserInput(POST_FILE, filesList);
    inputText := UserInput.Message;
    userMessage := inputText.ToUTF8.Trim;

    if inputText.Length = 0 then begin
      WriteLn('Пустое сообщение, пропускаем');
      Exit;
    end;

    WriteLn('Пользователь: ', userMessage);
    LastUserMessage := userMessage;

    // 2. Добавляем сообщение в историю
    UpdateConversationContext(userMessage, '');

    // 3. Строим расширенный контекст
    enhancedContext := BuildSymbolicContext(ConversationHistory);
    WriteLn('Расширенный контекст: ', Copy(enhancedContext, 1, 100), '...');

    // 4. Символьный анализ
    SymbolicEngine.DebugRuleMatching(userMessage); 
    symbolicResult := SymbolicEngine.ExecuteRule(userMessage);

  WriteLn('=== ОТЛАДКА СИМВОЛЬНОЙ СИСТЕМЫ ===');
  WriteLn('Пользователь: ', userMessage);
  WriteLn('Символьный результат: ', symbolicResult);
  WriteLn('Контекст: ', Copy(enhancedContext, 1, 80));
  WriteLn('=== КОНЕЦ ОТЛАДКИ ===');
    
    if symbolicResult <> '' then
    begin
      WriteLn('Символьная система: ', symbolicResult);
      finalResponse := GenerateResponseFromSymbolicSystem(symbolicResult, userMessage, enhancedContext);
    end
    else
    begin
      // 5. Если символьная система не нашла правила, используем трансформер
      WriteLn('Символьная система: правило не найдено, используем трансформер');
      inputMatrix := CreateContextEnrichedInput(inputText, enhancedContext);
      ForwardTransformer(TransformerModel, inputMatrix, outputMatrix);
      finalResponse := ResponseGenerator.GenerateResponse(outputMatrix, enhancedContext, userMessage);
    end;

    // 6. Извлекаем и сохраняем факты
    extractedFacts := ExtractFactsWithTransformer(userMessage);
    for i := 0 to High(extractedFacts) do
    begin
      if extractedFacts[i].Confidence > 0.5 then // Сохраняем только уверенные факты
      begin
        SymbolicEngine.AddFact(extractedFacts[i].Subject, 
                              extractedFacts[i].Relation, 
                              extractedFacts[i]._Object,
                              extractedFacts[i].Confidence);
      end;
    end;

    // 7. Выводим ответ и добавляем в историю
    WriteLn('AI: ', finalResponse);
    UpdateConversationContext('', finalResponse);

    // 8. Обучение на успешном диалоге
    if ShouldSaveForTraining(userMessage, finalResponse) then
    begin
      LearnFromConversation(userMessage, finalResponse);
      SymbolicEngine.LearnFromDialogue(userMessage, finalResponse);
      
      // Сохраняем базу знаний
      SymbolicEngine.SaveToFile('symbolic_knowledge.txt');
      WriteLn('Символьные знания сохранены');
    end;

  except
    on E: Exception do
    begin
      WriteLn('Ошибка в EnhancedProcessUserInput: ', E.Message);
      finalResponse := 'Извините, произошла техническая ошибка. Попробуйте еще раз.';
      WriteLn('AI: ', finalResponse);
    end;
  end;
  filesList.Free;
end;

function FindBestResponse(const question: ucs4): string;
var
  i, j: Integer;
  questionStr, pattern, context: string;
  scores: array of Double;
  maxScore: Double;
  bestIndex: Integer;
  foundDirectMatch: Boolean;
begin
  questionStr := UTF8LowerCase(question.ToUTF8);
  questionStr := questionStr.Trim;
  WriteLn('Debug: Анализируем вопрос: "', questionStr, '"');

  // Получаем контекст диалога
  context := GetConversationContext;

  WriteLn('Debug: Контекст: ', context);

//Exit('');

  // Проверяем продолжение диалога
  if context.Contains('как дела') and questionStr.Contains('норм') then
  begin
    Result := 'Рад слышать! Чем еще могу помочь?';
    Exit;
  end;

  if context.Contains('погод') and questionStr.Contains('солн') then
  begin
    Result := 'Солнечная погода - это прекрасно! Хотите обсудить планы на день?';
    Exit;
  end;

  // Сначала проверяем прямые совпадения
  foundDirectMatch := False;
  for i := 0 to High(ResponseTemplates) do
  begin
    for j := 0 to High(ResponseTemplates[i].Patterns) do
    begin
      pattern := UTF8LowerCase(ResponseTemplates[i].Patterns[j]);

      // Прямое вхождение или очень похожая фраза
      if (questionStr = pattern) or 
         (Pos(pattern, questionStr) > 0) or
         (questionStr.Contains(pattern)) then
      begin
        WriteLn('Debug: Прямое совпадение с шаблоном: ', pattern);
        Result := ResponseTemplates[i].Response;
        Exit;
      end;
    end;
  end;

  // Если нет прямых совпадений, используем Word2Vec (с проверкой!)
  if (WordEmbeddings <> nil) and (questionStr <> '') and 
     (Assigned(WordEmbeddings.FVocab)) and (WordEmbeddings.FVocab.Count > 0) then 
  begin
    try
      SetLength(scores, Length(ResponseTemplates));
      maxScore := 0;
      bestIndex := -1;

      for i := 0 to High(ResponseTemplates) do begin
        scores[i] := 0;
        for j := 0 to High(ResponseTemplates[i].Patterns) do begin
          try
            pattern := UTF8LowerCase(ResponseTemplates[i].Patterns[j]);
            scores[i] := Max(scores[i], WordEmbeddings.FastSimilarity(questionStr, pattern));
          except
            on E: Exception do
            begin
              WriteLn('Ошибка в FastSimilarity: ', E.Message);
              scores[i] := 0;
            end;
          end;
        end;

        if scores[i] > maxScore then begin
          maxScore := scores[i];
          bestIndex := i;
        end;
      end;

      if (bestIndex >= 0) and (maxScore > 0.4) then // Понижаем порог
      begin
        WriteLn('Debug: Найден шаблонный ответ [', 
          ResponseTemplates[bestIndex].Patterns[0], 
          '] score=', maxScore:0:2);
        Result := ResponseTemplates[bestIndex].Response;
        Exit;
      end;
    except
      on E: Exception do
      begin
        WriteLn('Ошибка в Word2Vec поиске: ', E.Message);
        // Продолжаем без Word2Vec
      end;
    end;
  end
  else
  begin
    WriteLn('Debug: WordEmbeddings не доступен, используем только шаблоны');
  end;

  // Если ничего не найдено
  WriteLn('Debug: Шаблон не найден');
  Result := '';
end;

function WaitForUserInput: Boolean;
var
  input: string;
begin
  Write('Введите сообщение (Enter - файл, Ctrl+C - выход): ');
  try
    ReadLn(input);
  except
    on E: EInOutError do
      if ShouldExit then Exit(False);
  end;

  if ShouldExit then Exit(False);

  if input = '' then
  begin
    // Пользователь хочет ввести многострочное сообщение через файл
    WriteLn('Пожалуйста, введите сообщение в файл ', POST_FILE, 
            ' и список вложений в ', POST_FILES_LIST);
    WriteLn('После завершения нажмите Enter...');
    ReadLn; // Ждем подтверждения
    Result := True;
  end
  else
  begin
    // Сохраняем однострочное сообщение во временный файл
    with TStringList.Create do
    try
      Text := input;
      SaveToFile(POST_FILE);
    finally
      Free;
    end;
    Result := True;
  end;
end;

function GenerateContextAwareResponse(const inputText: ucs4): string;
var
  context: string;
  inputMatrix, outputMatrix: TDoubleMatrix;
begin
  context := GetConversationContext;

  WriteLn('Debug: Генерация ответа с учетом контекста');
  WriteLn('Debug: Контекст: ', context);

  // Создаем обогащенный вход с контекстом
  inputMatrix := CreateContextEnrichedInput(inputText, context);

  try
    ForwardTransformer(TransformerModel, inputMatrix, outputMatrix);
    Result := GenerateTransformerResponse(inputText, outputMatrix);
  except
    on E: Exception do
    begin
      WriteLn('Ошибка трансформера: ', E.Message);
      Result := 'Извините, возникла техническая ошибка';
    end;
  end;
end;

procedure ProcessUserInput;
var
  inputText: ucs4;
  response, userMessage: string;
  filesList: TStringList;
  inputMatrix, outputMatrix: TDoubleMatrix;
begin
  filesList := TStringList.Create;
  try
    // 1. Получаем пользовательский ввод
    UserInput := GetUserInput(POST_FILE, filesList);
    inputText := UserInput.Message;
    userMessage := inputText.ToUTF8.Trim;

    if inputText.Length = 0 then begin
      WriteLn('Пустое сообщение, пропускаем');
      Exit;
    end;

    WriteLn('Пользователь: ', userMessage);
    LastUserMessage := userMessage;

    // 2. Добавляем сообщение в историю
    UpdateConversationContext(userMessage, '');

    // 3. Создаем входные данные
    inputMatrix := CreateContextEnrichedInput(inputText, GetConversationContext);

    // 4. Прямой проход через модель (без подробного вывода)
    ForwardTransformer(TransformerModel, inputMatrix, outputMatrix);

    // 5. Генерируем ответ через шаблоны
    response := FindBestResponse(inputText);

    // 6. Если шаблонный ответ не найден, используем вывод трансформера
if response = '' then 
begin
    response := ResponseGenerator.GenerateResponse(outputMatrix, GetConversationContext, userMessage)
end;

    // 7. Выводим ответ и добавляем в историю
    WriteLn('AI: ', response);
    UpdateConversationContext('', response);

// После успешного ответа
if ShouldSaveForTraining(userMessage, response) then
begin
  LearnFromConversation(userMessage, response);
  
  // Переобучение каждые 5 новых диалогов
  if (ConversationHistory.Count div 2) mod 5 = 0 then
  begin
    WriteLn('=== ИНКРЕМЕНТАЛЬНОЕ ОБУЧЕНИЕ ===');
    IncrementalTraining;
  end;
end;

  except
    on E: Exception do
    begin
      WriteLn('Ошибка: ', E.Message);
      response := 'Извините, произошла техническая ошибка. Попробуйте еще раз.';
      WriteLn('AI: ', response);
    end;
  end;
  filesList.Free;
end;

procedure TrainTransformerOnChatHistory;
var
  samples: TUC4Array;
  inputs, targets: TDoubleMatrix;
  i: Integer;
begin
  if not IsTransformerInitialized then
    Exit;

  // Простые примеры для обучения
  SetLength(samples, 4);
  samples[0] := 'привет как дела';
  samples[1] := 'что ты умеешь'; 
  samples[2] := 'расскажи о себе';
  samples[3] := 'спасибо пока';

  // Создаем входные данные
  inputs := TextsToMatrix(samples, TransformerModel.Config.InputSize);

  // Целевые выходы (простая имитация)
  SetLength(targets, Length(samples), TransformerModel.Config.InputSize);
  for i := 0 to High(samples) do
    targets[i] := RandomArray(TransformerModel.Config.InputSize, -0.1, 0.1);

  // Простой forward pass (без обратного распространения)
  try
    ForwardTransformer(TransformerModel, inputs, targets);
    WriteLn('Transformer прошел обучение на ', Length(samples), ' примерах');
  except
    on E: Exception do
      WriteLn('Ошибка обучения Transformer: ', E.Message);
  end;
end;

procedure Cleanup;
begin
  // Очистка временных файлов
  if FileExists(POST_FILE) then
    DeleteFile(POST_FILE);
  if FileExists(POST_FILES_LIST) then
    DeleteFile(POST_FILES_LIST);
end;

begin
FpSignal(SIGINT, @HandleSigInt);
ShowWelcome;
Initialize;
  try
    WriteLn('Инициализация моделей...');
    TrainTransformerOnChatHistory;
    WriteLn('Готов к работе!');
    WriteLn;

    while not ShouldExit do begin
      try
        if WaitForUserInput then
//          ProcessUserInput;
          EnhancedProcessUserInput;
        Cleanup; // Очищаем файлы после каждой итерации
      except
        on E: Exception do
          Writeln('Ошибка: ', E.ClassName, ': ', E.Message);
      end;
    end;
  finally
    Cleanup;
if IsTransformerInitialized then FreeTransformer(TransformerModel);
if Assigned(WordEmbeddings) then WordEmbeddings.Free;
if Assigned(CommonWordsCache) then CommonWordsCache.Free;
if Assigned(ConversationHistory) then ConversationHistory.Free;
if Assigned(ResponseGenerator) then ResponseGenerator.Free;
if Assigned(SymbolicEngine) then SymbolicEngine.Free;
    WriteLn('Все ресурсы освобождены. До свидания!');
  end;
end.