unit ContextManagerUnit;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}{$CODEPAGE UTF8}

interface

uses Classes,SysUtils,DataUtils,MatrixOps,Word2Vec,Math,LazUTF8,TextEmbeddings,DateUtils,ucs4unit,ucs4opunit,ucs4functionsunit;

type
  TContextManager = class
  private
    FHistory: TStringList;
    FMaxContextLength: Integer;
    FContextEmbeddings: TDoubleMatrix;
    FLastUpdate: TDateTime;
    FEmbeddingsValid: Boolean;
    // Добавляем кэш релевантности
    FRelevanceCache: TStringList;
    FLastCurrentMessage: string;
    FLastRelevanceCheck: TDateTime;
    function IsRelevantMessage(const Message: string; const CurrentMessage: string): Boolean;
    function FastRelevanceCheck(const msgLower, currentLower: string): Boolean;
    function DeepRelevanceCheck(const Message, CurrentMessage: string): Boolean;
    procedure CacheRelevance(const Message, CurrentMessage: string; IsRelevant: Boolean);
    function CompressMessage(const Message: string): string;
    function ExtractKeywords(const Text: string): string;
  public
    constructor Create(MaxContextLength: Integer = 5);
    destructor Destroy; override;
    procedure AddMessage(const Message: string; IsUser: Boolean);
    function GetOptimizedContext(const CurrentMessage: string): string;
    function GetContextEmbeddings(const CurrentMessage: string): TDoubleMatrix;
    procedure ClearContext;
    property History: TStringList read FHistory;
    function GetFastContextEmbeddings(const CurrentMessage: string): TDoubleMatrix;
  end;

var ContextManager : TContextManager = nil;

implementation

constructor TContextManager.Create(MaxContextLength: Integer = 5);
begin
inherited Create;
FHistory := TStringList.Create;
FRelevanceCache := TStringList.Create;
FMaxContextLength := MaxContextLength;
FEmbeddingsValid := False;
FLastUpdate := Now;
FLastCurrentMessage := '';
end;

destructor TContextManager.Destroy;
begin
FHistory.Free;
FRelevanceCache.Free;
SetLength(FContextEmbeddings, 0);
inherited Destroy;
end;

procedure TContextManager.AddMessage(const Message: string; IsUser: Boolean);
var
  prefix: string;
begin
  if IsUser then
    prefix := 'USER: '
  else
    prefix := 'AI: ';
  FHistory.Add(prefix + Message);
  FEmbeddingsValid := False; // Помечаем, что эмбеддинги устарели
  // Ограничиваем размер истории
  while FHistory.Count > FMaxContextLength * 3 do
    FHistory.Delete(0);
end;

operator in (s:string;sa:Array of String)r:bytebool; overload;
var f:Integer;
begin
for f := 0 to High(sa) do if sa[f]=s then Exit(true);
Exit(false);
end;

function TContextManager.ExtractKeywords(const Text: string): string;
const
  StopWords: array[0..51] of string = (
    'и', 'в', 'во', 'не', 'что', 'он', 'на', 'я', 'с', 'со', 'как', 'а',
    'то', 'все', 'она', 'так', 'его', 'но', 'да', 'ты', 'к', 'у', 'же',
    'вы', 'за', 'бы', 'по', 'только', 'ее', 'мне', 'было', 'вот', 'от',
    'меня', 'еще', 'нет', 'о', 'из', 'ему', 'теперь', 'когда', 'даже',
    'ну', 'вдруг', 'ли', 'если', 'уже', 'или', 'ни', 'быть', 'был', 'него');
var
  words: TStringArray;
  i, j: Integer;
  word: string;
  isStopWord: Boolean;
  keywordCount: Integer;
begin
  Result := '';
  keywordCount := 0;

  // Оптимизированное разбиение на слова
  words := Text.Split([' ', ',', '.', '!', '?', ';', ':', '"', ''''], 
                     TStringSplitOptions.ExcludeEmpty);

  for i := 0 to High(words) do begin
    word := UTF8LowerCase(words[i].Trim);

    // Быстрая проверка длины
    if (word = '') or (UTF8Length(word) <= 2) then Continue;

    // Быстрая проверка стоп-слов
    isStopWord := False;
    for j := 0 to High(StopWords) do begin
      if word = StopWords[j] then
      begin
        isStopWord := True;
        Break;
      end;
    end;

    if not isStopWord then begin
      if Result <> '' then Result := Result + ' ';
      Result := Result + word;
      Inc(keywordCount);

      // Ограничиваем количество ключевых слов для производительности
      if keywordCount >= 10 then Break;
    end;
  end;
end;

function TContextManager.IsRelevantMessage(const Message: string; const CurrentMessage: string): Boolean;
var
  cacheKey: string;
  cacheIndex: Integer;
  msgLower, currentLower: string;
begin
  // Кэширование результатов проверки релевантности
  if (FLastCurrentMessage = CurrentMessage) and 
     (MinutesBetween(Now, FLastRelevanceCheck) < 2) then
  begin
    cacheKey := Message + '|' + CurrentMessage;
    cacheIndex := FRelevanceCache.IndexOfName(cacheKey);
    if cacheIndex >= 0 then
      Exit(FRelevanceCache.ValueFromIndex[cacheIndex] = '1');
  end
  else
  begin
    // Сбрасываем кэш если текущее сообщение изменилось
    FRelevanceCache.Clear;
    FLastCurrentMessage := CurrentMessage;
    FLastRelevanceCheck := Now;
  end;

  // Быстрая проверка для коротких сообщений
  if (Length(Message) < 3) or (UTF8Length(CurrentMessage) < 3) then begin
    Result := False;
    Exit;
  end;

  msgLower := UTF8LowerCase(Message);
  currentLower := UTF8LowerCase(CurrentMessage);

  // 1. Быстрая проверка по ключевым словам
  if FastRelevanceCheck(msgLower, currentLower) then begin
    CacheRelevance(Message, CurrentMessage, True);
    Exit(True);
  end;

  // 2. Более глубокая проверка только если необходимо
  Result := DeepRelevanceCheck(Message, CurrentMessage);
  CacheRelevance(Message, CurrentMessage, Result);
end;

function TContextManager.FastRelevanceCheck(const msgLower, currentLower: string): Boolean;
const QuickKeywords: array[0..7] of string = ('это', 'тот', 'предыдущ', 'ранее', 'вопрос', 'ответ', 'тема', 'обсужда');
var i: Integer;
begin
  // Быстрая проверка по очевидным ключевым словам
  for i := 0 to High(QuickKeywords) do begin
    if msgLower.Contains(QuickKeywords[i]) then
      Exit(True);
  end;
  // Проверка прямого упоминания
  if (msgLower.Contains('ты') and currentLower.Contains('я')) or
     (msgLower.Contains('я') and currentLower.Contains('ты')) then
    Exit(True);
Exit(False);
end;

function TContextManager.DeepRelevanceCheck(const Message, CurrentMessage: string): Boolean;
var
  words1, words2: TStringArray;
  commonWords, i, j: Integer;
  word1, word2: string;
  similarity: Double;
begin
  // Оптимизированная проверка общих слов
  words1 := UTF8LowerCase(Message).Split([' ', ',', '.', '!', '?'], TStringSplitOptions.ExcludeEmpty);
  words2 := UTF8LowerCase(CurrentMessage).Split([' ', ',', '.', '!', '?'], TStringSplitOptions.ExcludeEmpty);

  // Быстрый подсчет общих слов (только слова длиной > 3)
  commonWords := 0;
  for i := 0 to High(words1) do begin
    if UTF8Length(words1[i]) <= 3 then Continue;

    for j := 0 to High(words2) do begin
      if UTF8Length(words2[j]) <= 3 then Continue;

      if words1[i] = words2[j] then
      begin
        Inc(commonWords);
        if commonWords >= 2 then
          Exit(True);
        Break; // Переходим к следующему слову
      end;
    end;
  end;

  // Word2Vec проверка только если есть достаточное пересечение
  if (commonWords >= 1) and Assigned(WordEmbeddings) then begin
    try
      similarity := WordEmbeddings.FastSimilarity(
        ExtractKeywords(Message),
        ExtractKeywords(CurrentMessage)
      );
      Result := similarity > 0.3; // Понижаем порог для скорости
    except
      Result := False;
    end;
  end
  else
    Result := commonWords >= 1;
end;

procedure TContextManager.CacheRelevance(const Message, CurrentMessage: string; IsRelevant: Boolean);
begin
  if FRelevanceCache.Count > 100 then // Ограничиваем размер кэша
    FRelevanceCache.Delete(0);

  FRelevanceCache.Add(Message + '|' + CurrentMessage + '=' + 
                     IfThen(IsRelevant, '1', '0'));
end;

function TContextManager.CompressMessage(const Message: string): string;
begin
// Упрощенная компрессия - удаляем стоп-слова и ограничиваем длину
Result := ExtractKeywords(Message);
// Ограничиваем длину
if UTF8Length(Result) > 100 then Result := UTF8Copy(Result, 1, 100) + '...';
end;

function TContextManager.GetContextEmbeddings(const CurrentMessage: string): TDoubleMatrix;
var
  contextText: string;
  s, tmp: ucs4;
  a: TUC4Array;
begin
  // Если эмбеддинги актуальны, возвращаем их
  if FEmbeddingsValid and (MinutesBetween(Now, FLastUpdate) < 5) then
    Exit(FContextEmbeddings);

  // Обновляем эмбеддинги
  contextText := GetOptimizedContext(CurrentMessage);

  if contextText <> '' then begin
    try
      s.Init;
      tmp.Init;
      tmp := #10;
      s := contextText;
      a := Split(s, tmp[0]);

      // ✅ ПЕРЕКЛЮЧАЕМ: Используем оптимизированную версию
      if Assigned(WordEmbeddings) then begin
        WriteLn('GetContextEmbeddings: используем оптимизированную версию с индексами');
        FContextEmbeddings := TextsToMatrixIndices(a, WordEmbeddings, 300);
      end else begin
        WriteLn('GetContextEmbeddings: используем стандартную версию');
Halt;
        FContextEmbeddings := TextsToMatrix(a, 300);
      end;

      tmp.Clear;
      s.Clear;

      FEmbeddingsValid := True;
      FLastUpdate := Now;

    except
      on E: Exception do
      begin
        WriteLn('Ошибка создания эмбеддингов контекста: ', E.Message);
        SetLength(FContextEmbeddings, 0, 0);
      end;
    end;
  end
  else
  begin
    SetLength(FContextEmbeddings, 0, 0);
  end;

Exit(FContextEmbeddings);
end;

procedure TContextManager.ClearContext;
begin
FHistory.Clear;
SetLength(FContextEmbeddings, 0, 0);
FEmbeddingsValid := False;
end;

{
function TContextManager.GetOptimizedContext(const CurrentMessage: string): string;
var
  i, count: Integer;
  relevantMessages: TStringList;
  maxMessages: Integer;
begin
  // Быстрый выход если история пуста
  if FHistory.Count = 0 then
    Exit('');
WriteLn(1);
  // Динамическое ограничение количества сообщений
  maxMessages := Min(FMaxContextLength, 3); // Уменьшаем для производительности
WriteLn(2);
  relevantMessages := TStringList.Create;
  try
    // Проверяем только последние N сообщений для производительности
    for i := FHistory.Count - 1 downto Max(0, FHistory.Count - maxMessages * 3) do
    begin
      if IsRelevantMessage(FHistory[i], CurrentMessage) then
      begin
        relevantMessages.Add(CompressMessage(FHistory[i]));
        if relevantMessages.Count >= maxMessages then
          Break;
      end;
    end;
WriteLn(3);
    // Собираем результат
    if relevantMessages.Count > 0 then
    begin
      Result := '';
      for i := relevantMessages.Count - 1 downto 0 do
      begin
        if Result <> '' then 
          Result := Result + #10;
        Result := Result + relevantMessages[i];

        // Ограничиваем общую длину контекста
        if UTF8Length(Result) > 500 then
        begin
          Result := UTF8Copy(Result, 1, 500) + '...';
          Break;
        end;
      end;
    end
    else
    begin
      // Возвращаем только последнее сообщение если нет релевантных
      Result := CompressMessage(FHistory[FHistory.Count - 1]);
    end;

  finally
    relevantMessages.Free;
  end;
end;
}
// Временно закомментируем вызовы, которые могут вызывать проблемы
function TContextManager.GetOptimizedContext(const CurrentMessage: string): string;
var
  i, count: Integer;
  relevantMessages: TStringList;
  maxMessages: Integer;
begin
  // Быстрый выход если история пуста
  if FHistory.Count = 0 then
    Exit('');

  WriteLn('ContextManager: получение контекста из ', FHistory.Count, ' сообщений');
  
  // ✅ ВРЕМЕННО УПРОЩАЕМ - ВОЗВРАЩАЕМ ТОЛЬКО ПОСЛЕДНИЕ СООБЩЕНИЯ
  maxMessages := Min(FMaxContextLength, FHistory.Count);
  Result := '';
  
  for i := FHistory.Count - maxMessages to FHistory.Count - 1 do
  begin
    if i >= 0 then
    begin
      if Result <> '' then Result := Result + #10;
      Result := Result + FHistory[i];
      
      // Ограничиваем длину
      if UTF8Length(Result) > 500 then
      begin
        Result := UTF8Copy(Result, 1, 500) + '...';
        Break;
      end;
    end;
  end;
  
  WriteLn('ContextManager: возвращаем контекст длиной ', UTF8Length(Result));
end;

function TContextManager.GetFastContextEmbeddings(const CurrentMessage: string): TDoubleMatrix;
var
  contextText: string;
  s, tmp: ucs4;
  texts: TUC4Array;
begin
  // Используем быструю версию с индексами
  contextText := GetOptimizedContext(CurrentMessage);

  if contextText <> '' then
  begin
    s.Init;
    tmp.Init;
    tmp := #10;
    s := contextText;
    texts := Split(s, tmp[0]);

    // ИСПОЛЬЗУЕМ БЫСТРУЮ ВЕРСИЮ С ИНДЕКСАМИ
    Result := TextsToMatrixIndices(texts, WordEmbeddings, 300);

    tmp.Clear;
    s.Clear;
  end
  else
  begin
    SetLength(Result, 0, 0);
  end;
end;

end.