unit ContextProcessing;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


{$MODE OBJFPC}{$H+}
{$CODEPAGE UTF8}

interface

uses
  SysUtils, Classes, DataUtils, MatrixOps, ucs4unit, ucs4opunit, ucs4functionsunit, 
  TextEmbeddings, SymbolicEngineUnit,LazUTF8,Math;

// Функции для обработки контекста
function BuildEnhancedContext(const History: TStringList; const CurrentInput: string): string;
function ExtractEntitiesFromText(const Text: string): TStringArray;
function AnalyzeConversationTopic(const History: TStringList): string;
function ShouldUpdateSymbolicFacts(const UserMessage, AIResponse: string): Boolean;

// Функции для работы с матрицами вывода
function DetectEntitiesInOutput(const OutputMatrix: TDoubleMatrix; const Vocabulary: TStringList): TStringArray;
function ExtractRelationsFromOutput(const OutputMatrix: TDoubleMatrix): TStringArray;

implementation

function ExtractFirstWords(const Text: string; WordCount: Integer): string;
var
  tokens: TUC4Array;
  i, count: Integer;
begin
  Result := '';
  tokens := TokenizeForNLP(Text);

  count := 0;
  for i := 0 to High(tokens) do
  begin
    if count >= WordCount then Break;

    if tokens[i].Length > 0 then
    begin
      if Result <> '' then Result := Result + ' ';
      Result := Result + tokens[i].ToUTF8;
      Inc(count);
    end;
  end;
end;

function CleanContext(const Text: string): string;
var
  words: TStringArray;
  i: Integer;
  cleanText: string;
begin
  // Удаляем служебные слова
  words := Text.Split([' ']);
  cleanText := '';

  for i := 0 to High(words) do
  begin
    if (words[i] <> 'USER:') and (words[i] <> 'AI:') and 
       (words[i] <> 'привет') and (words[i] <> 'пока') and
       (Length(words[i]) > 2) then
    begin
      if cleanText <> '' then cleanText := cleanText + ' ';
      cleanText := cleanText + words[i];
    end;
  end;

  Result := cleanText;
end;

function BuildEnhancedContext(const History: TStringList; const CurrentInput: string): string;
var
  i: Integer;
  simpleContext: string;
begin
  // СУПЕР-ПРОСТОЙ контекст - только ключевые слова
  simpleContext := '';

  // Берем только ключевые слова из последних 2 сообщений
  for i := Max(0, History.Count - 4) to History.Count - 1 do
  begin
    if (i >= 0) and (i < History.Count) then
    begin
      // Берем только первые 2 слова и удаляем служебные
      simpleContext := simpleContext + ' ' + ExtractFirstWords(History[i], 2);
    end;
  end;

  // Очищаем от служебных слов
  simpleContext := CleanContext(simpleContext);

  // Формируем очень компактный контекст
  Result := 'C:' + Copy(simpleContext.Trim, 1, 40);
end;

function ExtractEntitiesFromText(const Text: string): TStringArray;
var
  tokens: TUC4Array;
  i, count: Integer;
  tokenStr, cleanToken: string;
begin
  SetLength(Result, 0);
  if Text = '' then Exit;

  // Токенизируем оригинальный текст, а не обработанный
  tokens := TokenizeForNLP(Text); // Убрали NormalizeForAI и RemovePunctuation

  if Length(tokens) = 0 then Exit;

  count := 0;
  SetLength(Result, Length(tokens));

  for i := 0 to High(tokens) do
  begin
    tokenStr := tokens[i].ToUTF8;
    cleanToken := UTF8LowerCase(tokenStr.Trim);

    // Более качественная идентификация сущностей
    if (UTF8Length(cleanToken) >= 2) then
    begin
      // Географические названия (эвристика)
      if (UTF8Length(cleanToken) >= 4) and 
         (ContainsAny(cleanToken, ['москв', 'санкт', 'петербург', 'росси', 'город'])) then
      begin
        Result[count] := 'LOC:' + tokenStr;
        Inc(count);
      end
      // Вопросы
      else if ContainsAny(cleanToken, ['где', 'когда', 'почему', 'как', 'что']) then
      begin
        Result[count] := 'QUESTION:' + tokenStr;
        Inc(count);
      end
      // Местоимения и указатели
      else if ContainsAny(cleanToken, ['мы', 'вы', 'они', 'я', 'ты', 'это', 'тот']) then
      begin
        Result[count] := 'PRON:' + tokenStr;
        Inc(count);
      end
      // Длинные слова - вероятно концепции
      else if (UTF8Length(cleanToken) >= 5) and 
              (not ContainsAny(cleanToken, ['нормальн', 'интересн', 'продолжайте'])) then
      begin
        Result[count] := 'CONCEPT:' + tokenStr;
        Inc(count);
      end;
    end;
  end;

  SetLength(Result, count);
end;

function AnalyzeConversationTopic(const History: TStringList): string;
var
  i: Integer;
  allText, lowerText: string;
  topicScores: array of record
    Topic: string;
    Score: Integer;
  end;
begin
  if History.Count = 0 then
    Exit('general');

  // Собираем весь текст истории
  allText := '';
  for i := 0 to History.Count - 1 do
    allText := allText + ' ' + History[i];

  lowerText := UTF8LowerCase(allText);

  // Простой анализ темы по ключевым словам
  SetLength(topicScores, 5);
  topicScores[0].Topic := 'погода';
  topicScores[0].Score := 0;
  topicScores[1].Topic := 'имя';
  topicScores[1].Score := 0;
  topicScores[2].Topic := 'математика';
  topicScores[2].Score := 0;
  topicScores[3].Topic := 'технологии';
  topicScores[3].Score := 0;
  topicScores[4].Topic := 'общение';
  topicScores[4].Score := 0;

  // Подсчитываем упоминания тем
  if lowerText.Contains('погод') then Inc(topicScores[0].Score, 2);
  if lowerText.Contains('дожд') or lowerText.Contains('солн') then Inc(topicScores[0].Score);

  if lowerText.Contains('имя') or lowerText.Contains('зовут') then Inc(topicScores[1].Score, 3);

  if lowerText.Contains('счет') or lowerText.Contains('числ') then Inc(topicScores[2].Score, 2);
  if lowerText.Contains('плюс') or lowerText.Contains('минус') then Inc(topicScores[2].Score);

  if lowerText.Contains('компьютер') or lowerText.Contains('програм') then Inc(topicScores[3].Score, 2);

  // Базовая тема для общих разговоров
  topicScores[4].Score := 1;

  // Находим тему с максимальным счетом
  Result := topicScores[0].Topic;
  for i := 1 to High(topicScores) do
  begin
    if topicScores[i].Score > topicScores[0].Score then
    begin
      Result := topicScores[i].Topic;
      topicScores[0].Score := topicScores[i].Score;
    end;
  end;
end;

function ShouldUpdateSymbolicFacts(const UserMessage, AIResponse: string): Boolean;
begin
  // Решаем, нужно ли обновлять символьные факты на основе диалога
  Result := (Length(UserMessage) > 10) and 
            (not UserMessage.Contains('?')) and
            (AIResponse.Length > 5) and
            (not ContainsAny(UserMessage, ['привет', 'пока', 'спасибо']));
end;

function DetectEntitiesInOutput(const OutputMatrix: TDoubleMatrix; const Vocabulary: TStringList): TStringArray;
var
  i, j: Integer;
  maxScore: Double;
  bestIndex: Integer;
begin
  SetLength(Result, 0);

  if (Length(OutputMatrix) = 0) or (Vocabulary = nil) then Exit;

  // Упрощенная детекция сущностей - находим наиболее вероятные токены
  for i := 0 to High(OutputMatrix) do
  begin
    if Length(OutputMatrix[i]) > 0 then
    begin
      maxScore := OutputMatrix[i][0];
      bestIndex := 0;

      for j := 1 to High(OutputMatrix[i]) do
      begin
        if OutputMatrix[i][j] > maxScore then
        begin
          maxScore := OutputMatrix[i][j];
          bestIndex := j;
        end;
      end;

      // Если уверенность достаточно высока, добавляем сущность
      if (maxScore > 0.3) and (bestIndex < Vocabulary.Count) then
      begin
        SetLength(Result, Length(Result) + 1);
        Result[High(Result)] := Vocabulary[bestIndex];
      end;
    end;
  end;
end;

function ExtractRelationsFromOutput(const OutputMatrix: TDoubleMatrix): TStringArray;
begin
  SetLength(Result, 0);
  // Заглушка для извлечения отношений
  // В реальной реализации здесь будет анализ структуры выходной матрицы
  // для обнаружения отношений между сущностями
end;

end.