unit DecoderUnit;
{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}{$CODEPAGE UTF8}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


interface

uses
  SysUtils, Classes, DataUtils, MatrixOps, TextEmbeddings, Word2Vec, ucs4unit, LazUTF8,Math;

type
  TResponseGenerator = class
  private
    FWordEmbeddings: TWordEmbeddings;
    FResponseTemplates: array of record
      Category: string;
      Templates: array of string;
    end;

{    
    procedure LoadResponseTemplates;
    function FindBestTemplate(const outputFeatures: TDoubleArray): string;
    function GenerateFromVocabulary(const outputMatrix: TDoubleMatrix): string;
    function CreateContextualResponse(const features: TDoubleArray; const context: string): string;
  public
    constructor Create(AWordEmbeddings: TWordEmbeddings);
//    function GenerateResponse(const outputMatrix: TDoubleMatrix; const context: string = ''): string;
    function GenerateResponse(const outputMatrix: TDoubleMatrix; const context: string = ''; const userMessage: string = ''): string;
    function MatrixToFeatures(const outputMatrix: TDoubleMatrix): TDoubleArray;
}
  procedure LoadResponseTemplates;
  function FindBestTemplate(const outputFeatures: TDoubleArray; const userMessage: string): string;
  function GenerateFromVocabulary(const outputMatrix: TDoubleMatrix; const userMessage: string): string;
  function CreateContextualResponse(const features: TDoubleArray; const context: string; const userMessage: string): string;
public
  constructor Create(AWordEmbeddings: TWordEmbeddings);
  function GenerateResponse(const outputMatrix: TDoubleMatrix; const context: string = ''; const userMessage: string = ''): string;
  function MatrixToFeatures(const outputMatrix: TDoubleMatrix): TDoubleArray;
  end;

// Вспомогательные функции
function FindClosestWords(const embedding: TDoubleArray; WordEmbeddings: TWordEmbeddings; topK: Integer = 5): TStringArray;
function SampleFromDistribution(const probabilities: TDoubleArray): Integer;
function CreateResponseFromKeywords(const keywords: TStringArray): string;
function CosineSimilarity(const A, B: TDoubleArray): Double;

var ResponseGenerator: TResponseGenerator;

implementation

constructor TResponseGenerator.Create(AWordEmbeddings: TWordEmbeddings);
begin
  FWordEmbeddings := AWordEmbeddings;
  LoadResponseTemplates;
end;

{
procedure TResponseGenerator.LoadResponseTemplates;
begin
  SetLength(FResponseTemplates, 8);

  // 1. Приветственные ответы
  FResponseTemplates[0].Category := 'greeting';
  SetLength(FResponseTemplates[0].Templates, 4);
  FResponseTemplates[0].Templates[0] := 'Здравствуйте! Рад вас видеть.';
  FResponseTemplates[0].Templates[1] := 'Привет! Как ваши дела?';
  FResponseTemplates[0].Templates[2] := 'Добрый день! Чем могу помочь?';
  FResponseTemplates[0].Templates[3] := 'Приветствую! Что вас интересует?';

  // 2. Вопросные ответы
  FResponseTemplates[1].Category := 'question';
  SetLength(FResponseTemplates[1].Templates, 4);
  FResponseTemplates[1].Templates[0] := 'Интересный вопрос! Давайте обсудим это.';
  FResponseTemplates[1].Templates[1] := 'Хороший вопрос. Что вы сами об этом думаете?';
  FResponseTemplates[1].Templates[2] := 'Понял ваш вопрос. Могу рассказать подробнее.';
  FResponseTemplates[1].Templates[3] := 'Это действительно важная тема.';

  // 3. Информационные ответы
  FResponseTemplates[2].Category := 'informative';
  SetLength(FResponseTemplates[2].Templates, 4);
  FResponseTemplates[2].Templates[0] := 'По моей информации, это связано с ';
  FResponseTemplates[2].Templates[1] := 'Я считаю, что важно учитывать ';
  FResponseTemplates[2].Templates[2] := 'Интересно, что вы упомянули об этом. ';
  FResponseTemplates[2].Templates[3] := 'В контексте нашего разговора, ';

  // 4. Поддерживающие ответы
  FResponseTemplates[3].Category := 'supportive';
  SetLength(FResponseTemplates[3].Templates, 4);
  FResponseTemplates[3].Templates[0] := 'Понимаю вашу точку зрения.';
  FResponseTemplates[3].Templates[1] := 'Это действительно интересная мысль!';
  FResponseTemplates[3].Templates[2] := 'Спасибо, что поделились этим.';
  FResponseTemplates[3].Templates[3] := 'Я вас понимаю.';

  // 5. Уточняющие ответы
  FResponseTemplates[4].Category := 'clarifying';
  SetLength(FResponseTemplates[4].Templates, 4);
  FResponseTemplates[4].Templates[0] := 'Не могли бы вы уточнить?';
  FResponseTemplates[4].Templates[1] := 'Что именно вас интересует?';
  FResponseTemplates[4].Templates[2] := 'Можете рассказать подробнее?';
  FResponseTemplates[4].Templates[3] := 'Я не совсем понял. Объясните, пожалуйста.';

  // 6. Продолжающие ответы
  FResponseTemplates[5].Category := 'continuing';
  SetLength(FResponseTemplates[5].Templates, 4);
  FResponseTemplates[5].Templates[0] := 'Продолжайте, пожалуйста.';
  FResponseTemplates[5].Templates[1] := 'Интересно! Что еще?';
  FResponseTemplates[5].Templates[2] := 'Расскажите больше об этом.';
  FResponseTemplates[5].Templates[3] := 'И что было дальше?';

  // 7. Заключительные ответы
  FResponseTemplates[6].Category := 'closing';
  SetLength(FResponseTemplates[6].Templates, 4);
  FResponseTemplates[6].Templates[0] := 'Было приятно пообщаться!';
  FResponseTemplates[6].Templates[1] := 'Надеюсь, я был полезен.';
  FResponseTemplates[6].Templates[2] := 'Обращайтесь, если будут вопросы!';
  FResponseTemplates[6].Templates[3] := 'До новых встреч!';

  // 8. Нейтральные ответы
  FResponseTemplates[7].Category := 'neutral';
  SetLength(FResponseTemplates[7].Templates, 4);
  FResponseTemplates[7].Templates[0] := 'Понятно.';
  FResponseTemplates[7].Templates[1] := 'Интересно.';
  FResponseTemplates[7].Templates[2] := 'Я вас слушаю.';
  FResponseTemplates[7].Templates[3] := 'Продолжайте.';
end;
}
procedure TResponseGenerator.LoadResponseTemplates;
begin
  SetLength(FResponseTemplates, 8);

  // 1. Приветственные ответы
  FResponseTemplates[0].Category := 'greeting';
  SetLength(FResponseTemplates[0].Templates, 4);
  FResponseTemplates[0].Templates[0] := 'Здравствуйте! Рад вас видеть.';
  FResponseTemplates[0].Templates[1] := 'Привет! Как ваши дела?';
  FResponseTemplates[0].Templates[2] := 'Добрый день! Чем могу помочь?';
  FResponseTemplates[0].Templates[3] := 'Приветствую! Что вас интересует?';

  // 2. Вопросные ответы
  FResponseTemplates[1].Category := 'question';
  SetLength(FResponseTemplates[1].Templates, 6); // ↑ больше вариантов
  FResponseTemplates[1].Templates[0] := 'Интересный вопрос! Давайте обсудим это.';
  FResponseTemplates[1].Templates[1] := 'Хороший вопрос. Что вы сами об этом думаете?';
  FResponseTemplates[1].Templates[2] := 'Понял ваш вопрос. Могу рассказать подробнее.';
  FResponseTemplates[1].Templates[3] := 'Это действительно важная тема для обсуждения.';
  FResponseTemplates[1].Templates[4] := 'Отличный вопрос! Мне тоже это интересно.';
  FResponseTemplates[1].Templates[5] := 'Спасибо за вопрос! Давайте разберемся вместе.';

  // 3. Информационные ответы (НОВЫЕ - для новостей)
  FResponseTemplates[2].Category := 'informative';
  SetLength(FResponseTemplates[2].Templates, 8); // ↑ больше вариантов
  FResponseTemplates[2].Templates[0] := 'Интересная новость! Что вы об этом думаете?';
  FResponseTemplates[2].Templates[1] := 'Важная информация. Спасибо, что поделились.';
  FResponseTemplates[2].Templates[2] := 'Это действительно значимое событие.';
  FResponseTemplates[2].Templates[3] := 'Понял информацию. Есть что добавить?';
  FResponseTemplates[2].Templates[4] := 'Спасибо за новость! Интересно ваше мнение.';
  FResponseTemplates[2].Templates[5] := 'Заметил эту информацию. Что дальше?';
  FResponseTemplates[2].Templates[6] := 'Важное сообщение. Хотите обсудить детали?';
  FResponseTemplates[2].Templates[7] := 'Понял суть. Что вас особенно заинтересовало?';

  // 4. Поддерживающие ответы
  FResponseTemplates[3].Category := 'supportive';
  SetLength(FResponseTemplates[3].Templates, 4);
  FResponseTemplates[3].Templates[0] := 'Понимаю вашу точку зрения.';
  FResponseTemplates[3].Templates[1] := 'Это действительно интересная мысль!';
  FResponseTemplates[3].Templates[2] := 'Спасибо, что поделились этим.';
  FResponseTemplates[3].Templates[3] := 'Я вас понимаю.';

  // 5. Уточняющие ответы (РЕДКИЕ - только при низкой уверенности)
  FResponseTemplates[4].Category := 'clarifying';
  SetLength(FResponseTemplates[4].Templates, 3); // ↓ меньше вариантов
  FResponseTemplates[4].Templates[0] := 'Не совсем понял. Можете объяснить по-другому?';
  FResponseTemplates[4].Templates[1] := 'Хорошо, но мне нужно больше контекста.';
  FResponseTemplates[4].Templates[2] := 'Интересно, но не совсем ясно. О чем именно речь?';

  // 6. Продолжающие ответы
  FResponseTemplates[5].Category := 'continuing';
  SetLength(FResponseTemplates[5].Templates, 4);
  FResponseTemplates[5].Templates[0] := 'Продолжайте, пожалуйста.';
  FResponseTemplates[5].Templates[1] := 'Интересно! Что еще?';
  FResponseTemplates[5].Templates[2] := 'Расскажите больше об этом.';
  FResponseTemplates[5].Templates[3] := 'И что было дальше?';

  // 7. Заключительные ответы
  FResponseTemplates[6].Category := 'closing';
  SetLength(FResponseTemplates[6].Templates, 4);
  FResponseTemplates[6].Templates[0] := 'Было приятно пообщаться!';
  FResponseTemplates[6].Templates[1] := 'Надеюсь, я был полезен.';
  FResponseTemplates[6].Templates[2] := 'Обращайтесь, если будут вопросы!';
  FResponseTemplates[6].Templates[3] := 'До новых встреч!';

  // 8. Нейтральные ответы (ОСНОВНЫЕ - запасной вариант)
  FResponseTemplates[7].Category := 'neutral';
  SetLength(FResponseTemplates[7].Templates, 8); // ↑ больше вариантов
  FResponseTemplates[7].Templates[0] := 'Интересно! Расскажите подробнее.';
  FResponseTemplates[7].Templates[1] := 'Понятно. Что вы об этом думаете?';
  FResponseTemplates[7].Templates[2] := 'Спасибо за информацию!';
  FResponseTemplates[7].Templates[3] := 'Я вас слушаю.';
  FResponseTemplates[7].Templates[4] := 'Это действительно важно.';
  FResponseTemplates[7].Templates[5] := 'Хорошо, продолжайте.';
  FResponseTemplates[7].Templates[6] := 'Интересная тема для обсуждения.';
  FResponseTemplates[7].Templates[7] := 'Понял. Есть что добавить?';
end;

{
function TResponseGenerator.MatrixToFeatures(const outputMatrix: TDoubleMatrix): TDoubleArray;
var
  i, j: Integer;
  featureSums: TDoubleArray;
begin
  if (Length(outputMatrix) = 0) or (Length(outputMatrix[0]) = 0) then
  begin
    SetLength(Result, 8); // Возвращаем нейтральные фичи
    FillChar(Result[0], Length(Result) * SizeOf(Double), 0);
    Exit;
  end;

  // Создаем 8-мерный вектор признаков из выходной матрицы
  SetLength(featureSums, 8);
  SetLength(Result, 8);
  
  // 1. Уверенность (среднее значение)
  for i := 0 to High(outputMatrix) do
    for j := 0 to High(outputMatrix[i]) do
      featureSums[0] := featureSums[0] + outputMatrix[i][j];
  Result[0] := featureSums[0] / (Length(outputMatrix) * Length(outputMatrix[0]));

  // 2. Вариативность (дисперсия)
  for i := 0 to High(outputMatrix) do
    for j := 0 to High(outputMatrix[i]) do
      featureSums[1] := featureSums[1] + Sqr(outputMatrix[i][j] - Result[0]);
  Result[1] := Sqrt(featureSums[1] / (Length(outputMatrix) * Length(outputMatrix[0])));

  // 3. Максимальная активация
  Result[2] := outputMatrix[0][0];
  for i := 0 to High(outputMatrix) do
    for j := 0 to High(outputMatrix[i]) do
      if outputMatrix[i][j] > Result[2] then
        Result[2] := outputMatrix[i][j];

  // 4. Минимальная активация
  Result[3] := outputMatrix[0][0];
  for i := 0 to High(outputMatrix) do
    for j := 0 to High(outputMatrix[i]) do
      if outputMatrix[i][j] < Result[3] then
        Result[3] := outputMatrix[i][j];

  // 5-8. Активации по квадрантам (упрощенная пространственная информация)
  Result[4] := 0; Result[5] := 0; Result[6] := 0; Result[7] := 0;
  
  for i := 0 to High(outputMatrix) do
  begin
    for j := 0 to High(outputMatrix[i]) do
    begin
      if (i < Length(outputMatrix) div 2) and (j < Length(outputMatrix[i]) div 2) then
        Result[4] := Result[4] + outputMatrix[i][j]
      else if (i < Length(outputMatrix) div 2) and (j >= Length(outputMatrix[i]) div 2) then
        Result[5] := Result[5] + outputMatrix[i][j]
      else if (i >= Length(outputMatrix) div 2) and (j < Length(outputMatrix[i]) div 2) then
        Result[6] := Result[6] + outputMatrix[i][j]
      else
        Result[7] := Result[7] + outputMatrix[i][j];
    end;
  end;
  
  // Нормализуем квадранты
  for i := 4 to 7 do
    Result[i] := Result[i] / ((Length(outputMatrix) * Length(outputMatrix[0])) / 4);
end;
}
function TResponseGenerator.MatrixToFeatures(const outputMatrix: TDoubleMatrix): TDoubleArray;
var
  i, j,quadrantSize: Integer;
  featureSums: TDoubleArray;
  nonZeroCount: Integer;
matrixMax,matrixMin:Double;
normalizedMatrix:TDoubleMatrix;
begin
  if (Length(outputMatrix) = 0) or (Length(outputMatrix[0]) = 0) then
  begin
    SetLength(Result, 8);
    // ✅ ИСПРАВЛЕНИЕ: Устанавливаем нейтральные значения вместо нулей
    Result[0] := 0.5; // уверенность
    Result[1] := 0.3; // вариативность
    Result[2] := 0.6; // макс активация
    Result[3] := 0.4; // мин активация
    for i := 4 to 7 do Result[i] := 0.5;
    Exit;
  end;

  SetLength(featureSums, 8);
  SetLength(Result, 8);
  
  // ✅ ИСПРАВЛЕНИЕ 1: Нормализуем значения матрицы перед анализом
  normalizedMatrix := CopyMatrix(outputMatrix);
  matrixMax := outputMatrix[0][0];
  matrixMin := outputMatrix[0][0];
  
  // Находим min/max для нормализации
  for i := 0 to High(outputMatrix) do
    for j := 0 to High(outputMatrix[i]) do
    begin
      if outputMatrix[i][j] > matrixMax then matrixMax := outputMatrix[i][j];
      if outputMatrix[i][j] < matrixMin then matrixMin := outputMatrix[i][j];
    end;
  
  // Нормализуем к [0,1]
  if matrixMax > matrixMin then
  begin
    for i := 0 to High(outputMatrix) do
      for j := 0 to High(outputMatrix[i]) do
        normalizedMatrix[i][j] := (outputMatrix[i][j] - matrixMin) / (matrixMax - matrixMin);
  end;

  // 1. Уверенность (среднее нормализованное значение)
  nonZeroCount := 0;
  for i := 0 to High(normalizedMatrix) do
    for j := 0 to High(normalizedMatrix[i]) do
    begin
      featureSums[0] := featureSums[0] + normalizedMatrix[i][j];
      if normalizedMatrix[i][j] > 0.1 then Inc(nonZeroCount);
    end;
  Result[0] := featureSums[0] / (Length(normalizedMatrix) * Length(normalizedMatrix[0]));
  
  // ✅ ИСПРАВЛЕНИЕ 2: Увеличиваем уверенность если много ненулевых активаций
  if nonZeroCount > (Length(normalizedMatrix) * Length(normalizedMatrix[0]) div 4) then
    Result[0] := Min(1.0, Result[0] * 1.3);

  // 2. Вариативность (нормализованная)
  for i := 0 to High(normalizedMatrix) do
    for j := 0 to High(normalizedMatrix[i]) do
      featureSums[1] := featureSums[1] + Sqr(normalizedMatrix[i][j] - Result[0]);
  Result[1] := Sqrt(featureSums[1] / (Length(normalizedMatrix) * Length(normalizedMatrix[0])));

  // 3-4. Макс/мин активации (нормализованные)
  Result[2] := normalizedMatrix[0][0];
  Result[3] := normalizedMatrix[0][0];
  for i := 0 to High(normalizedMatrix) do
    for j := 0 to High(normalizedMatrix[i]) do
    begin
      if normalizedMatrix[i][j] > Result[2] then Result[2] := normalizedMatrix[i][j];
      if normalizedMatrix[i][j] < Result[3] then Result[3] := normalizedMatrix[i][j];
    end;

  // 5-8. Квадранты (упрощенные)
  Result[4] := 0; Result[5] := 0; Result[6] := 0; Result[7] := 0;
  quadrantSize := (Length(normalizedMatrix) * Length(normalizedMatrix[0])) div 4;
  if quadrantSize = 0 then quadrantSize := 1;
  
  for i := 0 to High(normalizedMatrix) do
  begin
    for j := 0 to High(normalizedMatrix[i]) do
    begin
      if (i < Length(normalizedMatrix) div 2) and (j < Length(normalizedMatrix[i]) div 2) then
        Result[4] := Result[4] + normalizedMatrix[i][j]
      else if (i < Length(normalizedMatrix) div 2) and (j >= Length(normalizedMatrix[i]) div 2) then
        Result[5] := Result[5] + normalizedMatrix[i][j]
      else if (i >= Length(normalizedMatrix) div 2) and (j < Length(normalizedMatrix[i]) div 2) then
        Result[6] := Result[6] + normalizedMatrix[i][j]
      else
        Result[7] := Result[7] + normalizedMatrix[i][j];
    end;
  end;
  
  // Нормализуем квадранты
  for i := 4 to 7 do
    Result[i] := Result[i] / quadrantSize;
end;

{
function TResponseGenerator.FindBestTemplate(const outputFeatures: TDoubleArray): string;
var
  i, bestCategory: Integer;
  categoryScores: TDoubleArray;
  maxScore: Double;
begin
  if Length(outputFeatures) < 8 then
  begin
    // Возвращаем нейтральный ответ по умолчанию
    Result := 'Интересно! Расскажите подробнее.';
    Exit;
  end;

  SetLength(categoryScores, Length(FResponseTemplates));
  
  // Вычисляем скоры для каждой категории на основе фичей
  for i := 0 to High(FResponseTemplates) do
  begin
    case i of
      0: // greeting - высокая уверенность + низкая вариативность
        categoryScores[i] := outputFeatures[0] * (1 - outputFeatures[1]);
      1: // question - средняя уверенность + средняя вариативность
        categoryScores[i] := outputFeatures[0] * outputFeatures[1];
      2: // informative - высокая уверенность + активация в первом квадранте
        categoryScores[i] := outputFeatures[0] * outputFeatures[4];
      3: // supportive - средняя уверенность + равномерное распределение
        categoryScores[i] := outputFeatures[0] * (1 - Abs(outputFeatures[4] - outputFeatures[7]));
      4: // clarifying - низкая уверенность + высокая вариативность
        categoryScores[i] := (1 - outputFeatures[0]) * outputFeatures[1];
      5: // continuing - средняя уверенность + активация в последних квадрантах
        categoryScores[i] := outputFeatures[0] * (outputFeatures[6] + outputFeatures[7]);
      6: // closing - низкая уверенность + низкая вариативность
        categoryScores[i] := (1 - outputFeatures[0]) * (1 - outputFeatures[1]);
      7: // neutral - нейтральные значения
        categoryScores[i] := 0.5 * (1 - Abs(outputFeatures[0] - 0.5));
    end;
  end;

  // Находим лучшую категорию
  maxScore := categoryScores[0];
  bestCategory := 0;
  for i := 1 to High(categoryScores) do
  begin
    if categoryScores[i] > maxScore then
    begin
      maxScore := categoryScores[i];
      bestCategory := i;
    end;
  end;

  // Выбираем случайный шаблон из лучшей категории
  if (Length(FResponseTemplates[bestCategory].Templates) > 0) and (maxScore > 0.1) then
  begin
    Result := FResponseTemplates[bestCategory].Templates[
      Random(Length(FResponseTemplates[bestCategory].Templates))
    ];
  end
  else
  begin
    // Fallback на нейтральный ответ
    Result := 'Понятно. Чем еще могу помочь?';
  end;
end;
}
{
function TResponseGenerator.FindBestTemplate(const outputFeatures: TDoubleArray): string;
var
  i, bestCategory: Integer;
  categoryScores: TDoubleArray;
  maxScore: Double;
begin
  if Length(outputFeatures) < 8 then
  begin
    // ✅ ИСПРАВЛЕНИЕ 3: Более осмысленный fallback
    Result := 'Интересная информация! Что вы об этом думаете?';
    Exit;
  end;

  SetLength(categoryScores, Length(FResponseTemplates));
  
  // ✅ ИСПРАВЛЕНИЕ 4: Улучшенная логика выбора категорий
  for i := 0 to High(FResponseTemplates) do
  begin
    case i of
      0: // greeting - высокая уверенность + низкая вариативность
        categoryScores[i] := outputFeatures[0] * (1 - outputFeatures[1]) * 1.2;
      1: // question - средняя уверенность + равномерное распределение
        categoryScores[i] := outputFeatures[0] * (1 - Abs(outputFeatures[4] - outputFeatures[7])) * 1.1;
      2: // informative - высокая уверенность + активация в первом квадранте
        categoryScores[i] := outputFeatures[0] * outputFeatures[4] * 1.3;
      3: // supportive - средняя уверенность + низкая вариативность
        categoryScores[i] := outputFeatures[0] * (1 - outputFeatures[1]);
      4: // clarifying - НИЗКАЯ уверенность + высокая вариативность
        categoryScores[i] := (1 - outputFeatures[0]) * outputFeatures[1] * 0.7; // ↓ понижаем вес
      5: // continuing - средняя уверенность + активация в последних квадрантах
        categoryScores[i] := outputFeatures[0] * (outputFeatures[6] + outputFeatures[7]) * 1.1;
      6: // closing - низкая уверенность + низкая вариативность
        categoryScores[i] := (1 - outputFeatures[0]) * (1 - outputFeatures[1]);
      7: // neutral - запасной вариант
        categoryScores[i] := 0.6 * (1 - Abs(outputFeatures[0] - 0.5));
    end;
    
    // ✅ ИСПРАВЛЕНИЕ 5: Минимальный порог для избегания случайных выборов
    if categoryScores[i] < 0.1 then
      categoryScores[i] := 0;
  end;

  // Находим лучшую категорию
  maxScore := categoryScores[0];
  bestCategory := 0;
  for i := 1 to High(categoryScores) do
  begin
    if categoryScores[i] > maxScore then
    begin
      maxScore := categoryScores[i];
      bestCategory := i;
    end;
  end;

  // ✅ ИСПРАВЛЕНИЕ 6: Fallback на нейтральные ответы если все скоры низкие
  if maxScore < 0.15 then
    bestCategory := 7; // neutral category

  // Выбираем случайный шаблон из лучшей категории
  if Length(FResponseTemplates[bestCategory].Templates) > 0 then
  begin
    Result := FResponseTemplates[bestCategory].Templates[
      Random(Length(FResponseTemplates[bestCategory].Templates))
    ];
    
    // ✅ ИСПРАВЛЕНИЕ 7: Логируем выбор для отладки
    WriteLn('Debug: Категория=', FResponseTemplates[bestCategory].Category, 
            ' Уверенность=', outputFeatures[0]:0:3,
            ' Скор=', maxScore:0:3);
  end
  else
  begin
    Result := 'Интересно! Расскажите подробнее.';
  end;
end;
}
function TResponseGenerator.FindBestTemplate(const outputFeatures: TDoubleArray; const userMessage: string): string;
var
  messageLower: string;
begin
  if userMessage = '' then
  begin
    // Если нет сообщения, используем нейтральный ответ
    Result := FResponseTemplates[7].Templates[Random(Length(FResponseTemplates[7].Templates))];
    WriteLn('Debug: Категория=neutral (нет сообщения)');
    Exit;
  end;
  
  messageLower := UTF8LowerCase(userMessage);
  
  // Приветствия
  if (messageLower.Contains('привет')) or (messageLower.Contains('здравствуй')) or 
     (messageLower.Contains('hello')) or (messageLower.Contains('hi')) then
  begin
    Result := FResponseTemplates[0].Templates[Random(Length(FResponseTemplates[0].Templates))];
    WriteLn('Debug: Категория=greeting (по содержанию)');
    Exit;
  end;
  
  // Вопросы о делах
  if (messageLower.Contains('как дела')) or (messageLower.Contains('how are')) then
  begin
    Result := FResponseTemplates[1].Templates[Random(Length(FResponseTemplates[1].Templates))];
    WriteLn('Debug: Категория=question (по содержанию)');
    Exit;
  end;
  
  // Вопросы о возможностях
  if (messageLower.Contains('что ты умеешь')) or (messageLower.Contains('твои возможности')) or
     (messageLower.Contains('что можешь')) then
  begin
    Result := 'Я - чат-бот с искусственным интеллектом! Могу общаться на разные темы, отвечать на вопросы и поддерживать беседу.';
    WriteLn('Debug: Категория=informative (специальный ответ)');
    Exit;
  end;
  
  // Математические вопросы
  if (messageLower.Contains('сколько будет')) or (messageLower.Contains('2+2')) or
     (messageLower.Contains('посчитай')) then
  begin
    Result := 'Я пока не умею выполнять математические вычисления, но это интересная задача для обсуждения!';
    WriteLn('Debug: Категория=informative (специальный ответ)');
    Exit;
  end;
  
  // Вопросы о программировании
  if (messageLower.Contains('язык программирования')) or (messageLower.Contains('написан')) then
  begin
    Result := 'Я написан на FreePascal с использованием архитектуры трансформера!';
    WriteLn('Debug: Категория=informative (специальный ответ)');
    Exit;
  end;
  
  // Вопросы о форме/физике
  if (messageLower.Contains('форма')) and (messageLower.Contains('радуг')) then
  begin
    Result := 'Радуга имеет форму дуги из-за преломления света в каплях воды! Хотите обсудить физику этого явления?';
    WriteLn('Debug: Категория=informative (специальный ответ)');
    Exit;
  end;
  
  // Вопросы (содержат знак вопроса)
  if userMessage.Contains('?') then
  begin
    Result := FResponseTemplates[1].Templates[Random(Length(FResponseTemplates[1].Templates))];
    WriteLn('Debug: Категория=question (по знаку вопроса)');
    Exit;
  end;
  
  // Утверждения
  if (messageLower.Contains('да,')) or (messageLower.Contains('ты прав')) or
     (messageLower.Contains('соглас')) then
  begin
    Result := FResponseTemplates[3].Templates[Random(Length(FResponseTemplates[3].Templates))];
    WriteLn('Debug: Категория=supportive (по содержанию)');
    Exit;
  end;
  
  // Fallback - нейтральные ответы
  Result := FResponseTemplates[7].Templates[Random(Length(FResponseTemplates[7].Templates))];
  WriteLn('Debug: Категория=neutral (fallback)');
end;

{
function TResponseGenerator.GenerateFromVocabulary(const outputMatrix: TDoubleMatrix): string;
var
  i, j: Integer;
  avgEmbedding: TDoubleArray;
  keywords: TStringArray;
begin
  if (FWordEmbeddings = nil) or (Length(outputMatrix) = 0) then
  begin
    Result := '';
    Exit;
  end;

  try
    // Усредняем выходную матрицу в один вектор
    SetLength(avgEmbedding, Length(outputMatrix[0]));
    FillChar(avgEmbedding[0], Length(avgEmbedding) * SizeOf(Double), 0);
    
    for i := 0 to High(outputMatrix) do
      for j := 0 to High(outputMatrix[i]) do
        avgEmbedding[j] := avgEmbedding[j] + outputMatrix[i][j];
        
    for j := 0 to High(avgEmbedding) do
      avgEmbedding[j] := avgEmbedding[j] / Length(outputMatrix);

    // Находим ближайшие слова
    keywords := FindClosestWords(avgEmbedding, FWordEmbeddings, 3);
    
    // Создаем ответ на основе ключевых слов
    if Length(keywords) > 0 then
      Result := CreateResponseFromKeywords(keywords)
    else
      Result := '';

  except
    Result := '';
  end;
end;
}
{
function TResponseGenerator.GenerateFromVocabulary(const outputMatrix: TDoubleMatrix): string;
var
  i, j: Integer;
  avgEmbedding: TDoubleArray;
  keywords: TStringArray;
  validKeywords: TStringList;
begin
  if (FWordEmbeddings = nil) or (Length(outputMatrix) = 0) then
  begin
    Result := '';
    Exit;
  end;

  try
    // Усредняем выходную матрицу в один вектор
    SetLength(avgEmbedding, Length(outputMatrix[0]));
    FillChar(avgEmbedding[0], Length(avgEmbedding) * SizeOf(Double), 0);
    
    for i := 0 to High(outputMatrix) do
      for j := 0 to High(outputMatrix[i]) do
        avgEmbedding[j] := avgEmbedding[j] + outputMatrix[i][j];
        
    for j := 0 to High(avgEmbedding) do
      avgEmbedding[j] := avgEmbedding[j] / Length(outputMatrix);

    // ✅ ИСПРАВЛЕНИЕ 8: Фильтруем ключевые слова
    keywords := FindClosestWords(avgEmbedding, FWordEmbeddings, 10); // Берем больше слов
    
    validKeywords := TStringList.Create;
    try
      // Фильтруем только осмысленные слова (длина > 3, не служебные)
      for i := 0 to High(keywords) do
      begin
        if (UTF8Length(keywords[i]) >= 4) and 
           (not keywords[i].Contains('.')) and
           (not keywords[i].Contains(',')) and
           (not keywords[i].Contains('0')) and
           (not keywords[i].Contains('1')) and
           (not keywords[i].Contains('2')) and
           (not keywords[i].Contains('3')) and
           (not keywords[i].Contains('4')) and
           (not keywords[i].Contains('5')) and
           (not keywords[i].Contains('6')) and
           (not keywords[i].Contains('7')) and
           (not keywords[i].Contains('8')) and
           (not keywords[i].Contains('9')) then
        begin
          validKeywords.Add(keywords[i]);
        end;
      end;
      
      // Берем только 2-3 лучших ключевых слова
      SetLength(keywords, Min(3, validKeywords.Count));
      for i := 0 to High(keywords) do
        keywords[i] := validKeywords[i];
        
    finally
      validKeywords.Free;
    end;
    
    // Создаем ответ на основе ключевых слов
    if Length(keywords) > 0 then
    begin
      Result := CreateResponseFromKeywords(keywords);
      WriteLn('Debug: Ключевые слова: ', string.Join(', ', keywords));
    end
    else
      Result := '';

  except
    on E: Exception do
    begin
      WriteLn('Ошибка в GenerateFromVocabulary: ', E.Message);
      Result := '';
    end;
  end;
end;
}
function TResponseGenerator.GenerateFromVocabulary(const outputMatrix: TDoubleMatrix; const userMessage: string): string;
begin
  // ✅ ИСПРАВЛЕНИЕ: ОТКЛЮЧАЕМ КЛЮЧЕВЫЕ СЛОВА - пока не обучена модель
  // Возвращаем пустую строку чтобы не использовать словарные ответы
  Result := '';
end;

{
function TResponseGenerator.CreateContextualResponse(const features: TDoubleArray; const context: string): string;
var
  baseResponse, contextualPart: string;
begin
  baseResponse := FindBestTemplate(features);
  
  // Добавляем контекстуальную часть если есть контекст
  if context <> '' then
  begin
    if context.Contains('?') then
      contextualPart := 'Относительно вашего вопроса, '
    else if context.Contains('!') then
      contextualPart := 'Как интересно! '
    else if context.Contains('спасибо') then
      contextualPart := 'Пожалуйста! '
    else
      contextualPart := 'Кстати, ';
      
    Result := contextualPart + LowerCase(Copy(baseResponse, 1, 1)) + Copy(baseResponse, 2, Length(baseResponse));
  end
  else
    Result := baseResponse;
end;
}
function TResponseGenerator.CreateContextualResponse(const features: TDoubleArray; const context: string; const userMessage: string): string;
var
  baseResponse: string;
begin
  // ✅ ИСПРАВЛЕНИЕ: ПЕРЕДАЕМ userMessage в FindBestTemplate
  baseResponse := FindBestTemplate(features, userMessage);
  
  // Убираем "Относительно вашего вопроса" - это звучит неестественно
  Result := baseResponse;
end;

{
function TResponseGenerator.GenerateResponse(const outputMatrix: TDoubleMatrix; const context: string = ''): string;
var
  features: TDoubleArray;
  vocabResponse, templateResponse: string;
begin
  // 1. Извлекаем признаки из выходной матрицы
  features := MatrixToFeatures(outputMatrix);
  
  // 2. Генерируем ответ через шаблоны
  templateResponse := CreateContextualResponse(features, context);
  
  // 3. Пытаемся сгенерировать ответ через словарь (опционально)
  vocabResponse := GenerateFromVocabulary(outputMatrix);
  
  // 4. Комбинируем результаты
  if (vocabResponse <> '') and (Random(100) < 30) then // 30% chance использовать словарный ответ
  begin
    Result := vocabResponse;
  end
  else
  begin
    Result := templateResponse;
    
    // Добавляем вариативность через суффиксы
    case Random(4) of
      0: Result := Result + ' Что вы думаете?';
      1: Result := Result + ' Хотите обсудить это?';
      2: Result := Result + ' Есть ли вопросы?';
      3: ; // Без изменений
    end;
  end;
  
  // Ограничиваем длину ответа
  if UTF8Length(Result) > 200 then
    Result := UTF8Copy(Result, 1, 200) + '...';
end;
}
{
function TResponseGenerator.GenerateResponse(const outputMatrix: TDoubleMatrix; const context: string = ''): string;
var
  features: TDoubleArray;
  vocabResponse, templateResponse: string;
  useVocabResponse: Boolean;
begin
  // ✅ ИСПРАВЛЕНИЕ 9: Улучшенная логика комбинирования
  
  // 1. Извлекаем признаки из выходной матрицы
  features := MatrixToFeatures(outputMatrix);
  
  // 2. Генерируем ответ через шаблоны
  templateResponse := CreateContextualResponse(features, context);
  
  // 3. Пытаемся сгенерировать ответ через словарь
  vocabResponse := GenerateFromVocabulary(outputMatrix);
  
  // 4. Решаем какой ответ использовать
  useVocabResponse := (vocabResponse <> '') and 
                     (features[0] > 0.4) and // Достаточная уверенность
                     (Random(100) < 40); // 40% chance
  
  if useVocabResponse then
  begin
    Result := vocabResponse;
    WriteLn('Debug: Используем словарный ответ');
  end
  else
  begin
    Result := templateResponse;
    
    // ✅ ИСПРАВЛЕНИЕ 10: Более осмысленные суффиксы
    if features[0] > 0.6 then // Высокая уверенность
    begin
      case Random(3) of
        0: Result := Result + ' Что вы об этом думаете?';
        1: Result := Result + ' Интересно ваше мнение.';
        2: ; // Без изменений
      end;
    end
    else if features[0] < 0.3 then // Низкая уверенность
    begin
      case Random(3) of
        0: Result := Result + ' Можете уточнить?';
        1: Result := Result + ' Правильно ли я понял?';
        2: ; // Без изменений
      end;
    end;
  end;
  
  // Ограничиваем длину ответа
  if UTF8Length(Result) > 200 then
    Result := UTF8Copy(Result, 1, 200) + '...';
end;
}
{
function TResponseGenerator.GenerateResponse(const outputMatrix: TDoubleMatrix; const context: string = ''; const userMessage: string = ''): string;
var
  features: TDoubleArray;
  vocabResponse, templateResponse: string;
  useVocabResponse: Boolean;
begin
  // ✅ ИСПРАВЛЕНИЕ: УЧИТЫВАЕМ ВСЕ ПАРАМЕТРЫ
  
  // 1. Извлекаем фичи (игнорируя реальный выход модели пока не обучена)
  features := MatrixToFeatures(outputMatrix);
  
  // 2. Генерируем ответ на основе сообщения пользователя ИЛИ выхода модели
  if userMessage <> '' then
  begin
    // Используем эвристики на основе содержания сообщения
    templateResponse := CreateContextualResponse(features, context, userMessage);
    WriteLn('Debug: Ответ основан на содержании сообщения');
  end
  else
  begin
    // Используем выход модели (будет работать после обучения)
    templateResponse := CreateContextualResponse(features, context, '');
    WriteLn('Debug: Ответ основан на выходе модели');
  end;
  
  // 3. Пытаемся сгенерировать ответ через словарь (только если модель обучена)
  vocabResponse := GenerateFromVocabulary(outputMatrix, userMessage);
  
  // 4. Решаем какой ответ использовать
  useVocabResponse := (vocabResponse <> '') and 
                     (features[0] > 0.4) and // Достаточная уверенность
                     (Random(100) < 20); // 20% chance (низкая пока)
  
  if useVocabResponse then
  begin
    Result := vocabResponse;
    WriteLn('Debug: Используем словарный ответ');
  end
  else
  begin
    Result := templateResponse;
  end;
  
  // Ограничиваем длину ответа
  if UTF8Length(Result) > 200 then
    Result := UTF8Copy(Result, 1, 200) + '...';
end;
}
function TResponseGenerator.GenerateResponse(const outputMatrix: TDoubleMatrix; const context: string = ''; const userMessage: string = ''): string;
var
  features: TDoubleArray;
  vocabResponse, templateResponse: string;
  useVocabResponse: Boolean;
begin
  // 1. Извлекаем фичи из выходной матрицы
  features := MatrixToFeatures(outputMatrix);
  
  // 2. Генерируем ответ через шаблоны
  templateResponse := CreateContextualResponse(features, context, userMessage);
  
  // 3. Пытаемся сгенерировать ответ через словарь
  vocabResponse := GenerateFromVocabulary(outputMatrix, userMessage);
  
  // 4. Решаем какой ответ использовать
  useVocabResponse := (vocabResponse <> '') and 
                     (features[0] > 0.4) and // Достаточная уверенность
                     (Random(100) < 20); // 20% chance (низкая пока)
  
  if useVocabResponse then
  begin
    Result := vocabResponse;
    WriteLn('Debug: Используем словарный ответ');
  end
  else
  begin
    Result := templateResponse;
    
    // Добавляем вариативность через суффиксы
    if features[0] > 0.6 then // Высокая уверенность
    begin
      case Random(3) of
        0: Result := Result + ' Что вы об этом думаете?';
        1: Result := Result + ' Интересно ваше мнение.';
        2: ; // Без изменений
      end;
    end
    else if features[0] < 0.3 then // Низкая уверенность
    begin
      case Random(3) of
        0: Result := Result + ' Можете уточнить?';
        1: Result := Result + ' Правильно ли я понял?';
        2: ; // Без изменений
      end;
    end;
  end;
  
  // Ограничиваем длину ответа
  if UTF8Length(Result) > 200 then
    Result := UTF8Copy(Result, 1, 200) + '...';
end;

// Вспомогательные функции

function FindClosestWords(const embedding: TDoubleArray; WordEmbeddings: TWordEmbeddings; topK: Integer = 5): TStringArray;
var
  i, j, bestIndex: Integer;
  bestSimilarity, similarity: Double;
  similarities: TDoubleArray;
  usedIndices: array of Boolean;
begin
  SetLength(Result, 0);
  if (WordEmbeddings = nil) or (Length(embedding) = 0) then Exit;
  
  SetLength(similarities, WordEmbeddings.FVocab.Count);
  SetLength(usedIndices, WordEmbeddings.FVocab.Count);
  
  try
    // Вычисляем схожести со всеми словами
    for i := 0 to WordEmbeddings.FVocab.Count - 1 do
    begin
      similarities[i] := CosineSimilarity(embedding, WordEmbeddings.GetEmbeddingFastByIndex(i));
      usedIndices[i] := False;
    end;
    
    // Находим topK самых близких слов
    for i := 1 to Min(topK, WordEmbeddings.FVocab.Count) do
    begin
      bestSimilarity := -2;
      bestIndex := -1;
      
      for j := 0 to WordEmbeddings.FVocab.Count - 1 do
      begin
        if not usedIndices[j] and (similarities[j] > bestSimilarity) then
        begin
          bestSimilarity := similarities[j];
          bestIndex := j;
        end;
      end;
      
      if bestIndex >= 0 then
      begin
        SetLength(Result, Length(Result) + 1);
        Result[High(Result)] := WordEmbeddings.FVocab[bestIndex];
        usedIndices[bestIndex] := True;
      end;
    end;
    
  except
    // В случае ошибки возвращаем пустой массив
    SetLength(Result, 0);
  end;
end;

function SampleFromDistribution(const probabilities: TDoubleArray): Integer;
var
  i: Integer;
  sum, randVal, cumulative: Double;
begin
  sum := 0;
  for i := 0 to High(probabilities) do
    sum := sum + probabilities[i];
    
  if sum = 0 then
  begin
    Result := Random(Length(probabilities));
    Exit;
  end;
  
  randVal := Random * sum;
  cumulative := 0;
  
  for i := 0 to High(probabilities) do
  begin
    cumulative := cumulative + probabilities[i];
    if randVal <= cumulative then
    begin
      Result := i;
      Exit;
    end;
  end;
  
  Result := High(probabilities);
end;

function CreateResponseFromKeywords(const keywords: TStringArray): string;
const
  templates: array[0..7] of string = (
    'Вы упомянули %s. Это интересная тема!',
    'По поводу %s - что именно вас интересует?',
    '%s - хорошее направление для обсуждения.',
    'Мне нравится, что вы затронули тему %s.',
    'Относительно %s - есть что добавить?',
    '%s действительно важно. Хотите поговорить об этом?',
    'Заметил ваше внимание к %s. Продолжайте!',
    '%s - отличная тема для беседы.'
  );
var
  keywordStr: string;
  i: Integer;
begin
  if Length(keywords) = 0 then
  begin
    Result := '';
    Exit;
  end;
  
  // Объединяем ключевые слова
  keywordStr := keywords[0];
  for i := 1 to High(keywords) do
  begin
    if i = High(keywords) then
      keywordStr := keywordStr + ' и ' + keywords[i]
    else
      keywordStr := keywordStr + ', ' + keywords[i];
  end;
  
  // Выбираем случайный шаблон
  Result := Format(templates[Random(Length(templates))], [keywordStr]);
end;

function CosineSimilarity(const A, B: TDoubleArray): Double;
var
  i: Integer;
  DotProduct, NormA, NormB: Double;
begin
  if (Length(A) = 0) or (Length(B) = 0) or (Length(A) <> Length(B)) then
    Exit(0.0);

  DotProduct := 0;
  NormA := 0;
  NormB := 0;
  
  for i := 0 to High(A) do
  begin
    DotProduct := DotProduct + A[i] * B[i];
    NormA := NormA + A[i] * A[i];
    NormB := NormB + B[i] * B[i];
  end;

  if (NormA = 0) or (NormB = 0) then
    Exit(0.0);

  Result := DotProduct / (Sqrt(NormA) * Sqrt(NormB));
end;

end.