unit MatrixOps;
{$MODE OBJFPC}{$H+}
{$RANGECHECKS ON}
{$ASMMODE INTEL}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


interface

uses
  SysUtils, DataUtils, Math;

function RandomMatrix(rows, cols: Integer; minVal: Double = -0.5; maxVal: Double = 0.5): TDoubleMatrix;
function RandomArray(size: Integer; minVal: Double = -0.5; maxVal: Double = 0.5): TDoubleArray;
function AddVectors(const a, b: TDoubleArray): TDoubleArray;
function CreateZeroMatrix(rows, cols: Integer): TDoubleMatrix;
function MatrixNorm(const matrix: TDoubleMatrix): Double;
// Новые функции для Transformer
function LayerNorm(const m: TDoubleMatrix; const gamma, beta: TDoubleArray): TDoubleMatrix;
function AddMatrices(const a, b: TDoubleMatrix): TDoubleMatrix;
function ReLU(const m: TDoubleMatrix): TDoubleMatrix;
// Для Attention
function Softmax(const m: TDoubleMatrix): TDoubleMatrix;
function ConcatMatrices(const matrices: array of TDoubleMatrix): TDoubleMatrix;
function MatrixMultiply(const A, B: TDoubleMatrix): TDoubleMatrix;
function TransposeMatrix(const m: TDoubleMatrix): TDoubleMatrix;
function MatrixAdd(const A, B: TDoubleMatrix): TDoubleMatrix;
function ScaleMatrixCreate(const m: TDoubleMatrix; factor: Double): TDoubleMatrix;
procedure ScaleMatrix(var m: TDoubleMatrix; factor: Double);
function CopyMatrix(const m: TDoubleMatrix): TDoubleMatrix;
function CreateRandomMatrix(rows, cols: Integer; minVal: Double = -0.5; maxVal: Double = 0.5): TDoubleMatrix;
function Dropout(const m: TDoubleMatrix; rate: Double): TDoubleMatrix;
procedure AddNoise(var Matrix: TDoubleMatrix; NoiseLevel: Double);
procedure FillMatrix(var matrix: TDoubleMatrix; value: Double);
procedure FillArray(var arr: TDoubleArray; value: Double);
procedure UpdateMatrixAdam(var params, grads: TDoubleMatrix; var state: TAdamState; learningRate: Double);
// функции для работы с накопленными градиентами
function ScaleMatrixToSize(const m: TDoubleMatrix; newRows, newCols: Integer): TDoubleMatrix;
procedure PrintMatrix(const matrix: TDoubleMatrix; maxRows: Integer = 10; maxCols: Integer = 10; precision: Integer = 4);
procedure MatrixAddInPlace(var A: TDoubleMatrix; const B: TDoubleMatrix);
procedure ScaleMatrixInPlace(var A: TDoubleMatrix; factor: Double);
function MatrixMultiplyFast(const A, B: TDoubleMatrix): TDoubleMatrix;

implementation

{$I asmf.inc}

function RandomMatrix(rows, cols: Integer; minVal: Double; maxVal: Double): TDoubleMatrix;
var i,j: Integer;
begin Result:=nil;
  SetLength(Result, rows);
  for i := 0 to rows - 1 do begin
    SetLength(Result[i], cols);
    for j := 0 to cols - 1 do
      Result[i][j] := minVal + (maxVal - minVal) * Random;
  end;
end;

function RandomArray(size: Integer; minVal: Double; maxVal: Double): TDoubleArray;
var i: Integer;
begin Result:=nil;
SetLength(Result, size);
for i := 0 to size - 1 do Result[i] := minVal + (maxVal - minVal) * Random;
end;

function AddVectors(const a, b: TDoubleArray): TDoubleArray;
var i: Integer;
begin Result:=nil;
  if Length(a) <> Length(b) then
    raise Exception.Create('Vector lengths mismatch');

  SetLength(Result, Length(a));
  for i := 0 to High(a) do
    Result[i] := a[i] + b[i];
end;

function CreateZeroMatrix(rows, cols: Integer): TDoubleMatrix;
var i: Integer;
begin Result:=nil;
SetLength(Result, rows);
for i := 0 to rows - 1 do SetLength(Result[i], cols);
end;

function MatrixNorm(const matrix: TDoubleMatrix): Double;
var i,j: Integer;
begin Result := 0;
  for i := 0 to High(matrix) do
    for j := 0 to High(matrix[i]) do
      Result := Result + Sqr(matrix[i][j]);
  Result := Sqrt(Result);
end;

function LayerNorm(const m: TDoubleMatrix; const gamma, beta: TDoubleArray): TDoubleMatrix;
var
  i, j, size: Integer;
  mean, variance, sum_val, stddev: Double;
begin Result:=nil;
  if Length(m) = 0 then begin
    SetLength(Result, 0, 0);
    Exit;
  end;

  size := Length(m[0]);

  if (Length(gamma) <> size) or (Length(beta) <> size) then begin
    WriteLn('Ошибка LayerNorm: Несовпадение размеров. Matrix:', size, 
            ' gamma:', Length(gamma), ' beta:', Length(beta));
    Result := CopyMatrix(m);
    Exit;
  end;

  SetLength(Result, Length(m), size);

  for i := 0 to High(m) do begin
    // Вычисляем среднее для текущей строки
    sum_val := 0.0;
    for j := 0 to size - 1 do
      sum_val := sum_val + m[i][j];
    mean := sum_val / size;

    // Вычисляем дисперсию для текущей строки
    sum_val := 0.0;
    for j := 0 to size - 1 do
      sum_val := sum_val + Sqr(m[i][j] - mean);
    variance := sum_val / size;
    stddev := Sqrt(variance + 1e-8);

    // Нормализуем и применяем масштаб и смещение
    for j := 0 to size - 1 do
      Result[i][j] := gamma[j] * ((m[i][j] - mean) / stddev) + beta[j];
  end;
end;

function AddMatrices(const a, b: TDoubleMatrix): TDoubleMatrix;
var i,j: Integer;
begin Result:=nil;
  if (Length(a) <> Length(b)) or (Length(a[0]) <> Length(b[0])) then
    raise Exception.Create('Matrix dimensions mismatch in AddMatrices');
  SetLength(Result, Length(a), Length(a[0]));
  for i := 0 to High(a) do
    for j := 0 to High(a[i]) do
      Result[i][j] := a[i][j] + b[i][j];
end;

function ReLU(const m: TDoubleMatrix): TDoubleMatrix;
var i,j: Integer;
begin Result:=nil;
  SetLength(Result, Length(m), Length(m[0]));
  for i := 0 to High(m) do
    for j := 0 to High(m[i]) do
      Result[i][j] := Max(0, m[i][j]);
end;

function Softmax(const m: TDoubleMatrix): TDoubleMatrix;
var
  i, j: Integer;
  maxVal, sum: Double;
  expValues: TDoubleMatrix;
begin Result:=nil;
  SetLength(expValues, Length(m), Length(m[0]));
  SetLength(Result, Length(m), Length(m[0]));
  for i := 0 to High(m) do begin
    maxVal := MaxValue(m[i]);
    sum := 0;
    // Вычисляем экспоненты и сумму
    for j := 0 to High(m[i]) do begin
      expValues[i][j] := Exp(m[i][j] - maxVal);
      sum := sum + expValues[i][j];
    end;
    // Нормализуем
    for j := 0 to High(m[i]) do
      Result[i][j] := expValues[i][j] / sum;
  end;
end;

{
function ConcatMatrices(const matrices: array of TDoubleMatrix): TDoubleMatrix;
var i,j,k,totalCols,offset: Integer;
begin
  if Length(matrices) = 0 then
    Exit(nil);
  // Проверяем согласованность размеров
  for i := 1 to High(matrices) do
    if Length(matrices[i]) <> Length(matrices[0]) then
      raise Exception.Create('All matrices must have same number of rows');
  // Вычисляем общее количество столбцов
  totalCols := 0;
  for i := 0 to High(matrices) do
    Inc(totalCols, Length(matrices[i][0]));
  // Создаем результирующую матрицу
  SetLength(Result, Length(matrices[0]), totalCols);
  // Заполняем результат
  for i := 0 to High(matrices[0]) do begin
    offset := 0;
    for j := 0 to High(matrices) do
    begin
      for k := 0 to High(matrices[j][0]) do begin
        Result[i][offset + k] := matrices[j][i][k];
      end;
      Inc(offset, Length(matrices[j][0]));
    end;
  end;
end;
}
function ConcatMatrices(const matrices: array of TDoubleMatrix): TDoubleMatrix;
var
  i, j, k, totalCols, offset: Integer;
begin
  if Length(matrices) = 0 then
    Exit(nil);
    
  // Проверяем согласованность размеров
  for i := 1 to High(matrices) do
    if Length(matrices[i]) <> Length(matrices[0]) then
      raise Exception.Create('All matrices must have same number of rows');
      
  // Вычисляем общее количество столбцов
  totalCols := 0;
  for i := 0 to High(matrices) do
  begin
    if Length(matrices[i]) > 0 then
      Inc(totalCols, Length(matrices[i][0]));
  end;
  
  WriteLn('    ConcatMatrices: объединяем ', Length(matrices), ' матриц в ', totalCols, ' столбцов');
  
  // Создаем результирующую матрицу
  SetLength(Result, Length(matrices[0]), totalCols);
  
  // Заполняем результат
  for i := 0 to High(matrices[0]) do
  begin
    offset := 0;
    for j := 0 to High(matrices) do
    begin
      if Length(matrices[j]) > 0 then
      begin
        for k := 0 to High(matrices[j][0]) do
        begin
          Result[i][offset + k] := matrices[j][i][k];
        end;
        Inc(offset, Length(matrices[j][0]));
      end;
    end;
  end;
end;

{ полная версия работает
function MatrixMultiply(const A, B: TDoubleMatrix): TDoubleMatrix;
var i,j,k: Integer;
begin Result:=nil;
  WriteLn('MatrixMultiply: A=', Length(A), 'x', 
          IfThen(Length(A) > 0, IntToStr(Length(A[0])), '?'), 
          ', B=', Length(B), 'x', 
          IfThen(Length(B) > 0, IntToStr(Length(B[0])), '?'));

  // Проверка на пустые матрицы
  if (Length(A) = 0) or (Length(B) = 0) then begin
    WriteLn('ОШИБКА: Пустая матрица в умножении');
    SetLength(Result, 0, 0);
Halt;
    Exit;
  end;

  // Проверка совместимости размеров
  if Length(A[0]) <> Length(B) then begin
    WriteLn('ОШИБКА: Несовпадение размерностей для умножения');
    WriteLn('  A cols: ', Length(A[0]));
    WriteLn('  B rows: ', Length(B));
Halt;    
    // Возвращаем нулевую матрицу правильного размера
    SetLength(Result, Length(A), Length(B[0]));
    for i := 0 to High(Result) do
      for j := 0 to High(Result[0]) do
        Result[i][j] := 0.0;
    Exit;
  end;

  SetLength(Result, Length(A), Length(B[0]));
  WriteLn('Result size: ', Length(Result), 'x', Length(Result[0]));

  for i := 0 to High(A) do begin
    for j := 0 to High(B[0]) do begin
      Result[i][j] := 0;
      for k := 0 to High(B) do begin
        // Детальная проверка границ
        if (i < Length(A)) and (k < Length(A[i])) and 
           (k < Length(B)) and (j < Length(B[k])) then
        begin
          Result[i][j] := Result[i][j] + A[i][k] * B[k][j];
        end
        else
        begin
          WriteLn('ВНИМАНИЕ: Выход за границы при умножении [', i, ',', j, ',', k, ']');
          WriteLn('  A bounds: ', Length(A), 'x', Length(A[0]));
          WriteLn('  B bounds: ', Length(B), 'x', Length(B[0]));
          WriteLn('  A[i] length: ', Length(A[i]));
          WriteLn('  B[k] length: ', Length(B[k]));
Halt;
        end;
      end;
    end;
  end;

  WriteLn('MatrixMultiply: завершено успешно');
end;
убыстренная версия:}

function MatrixMultiply(const A, B: TDoubleMatrix): TDoubleMatrix;
var
  i, j, k: Integer;
  sum: Double;
begin
  // Блочное умножение для лучшей кэш-локальности
  SetLength(Result, Length(A), Length(B[0]));

  for i := 0 to High(A) do
  begin
    for k := 0 to High(B) do
    begin
      if A[i][k] <> 0 then // Пропускаем нулевые значения
      begin
        for j := 0 to High(B[0]) do
        begin
          Result[i][j] := Result[i][j] + A[i][k] * B[k][j];
        end;
      end;
    end;
  end;
end;


function TransposeMatrix(const m: TDoubleMatrix): TDoubleMatrix;
var i,j: Integer;
begin Result:=nil;
  SetLength(Result, Length(m[0]), Length(m));
  for i := 0 to High(m) do
    for j := 0 to High(m[0]) do
      Result[j][i] := m[i][j];
end;

function MatrixAdd(const A, B: TDoubleMatrix): TDoubleMatrix;
var i,j,rows,cols: Integer;
begin
// Проверка размеров (из предыдущих версий)
if (Length(A) <> Length(B)) or (Length(A[0]) <> Length(B[0])) then begin
WriteLn('ОШИБКА MatrixAdd: Несовпадение размеров ');
WriteLn('  A: ', Length(A), 'x', Length(A[0]));
WriteLn('  B: ', Length(B), 'x', Length(B[0]));
Halt;
end;

  // Если одна из матриц пустая, возвращаем другую
  if (Length(A) = 0) or (Length(A[0]) = 0) then begin
    WriteLn('MatrixAdd: Матрица A пустая, возвращаем B');
Halt;
    Exit(CopyMatrix(B));
  end;

  if (Length(B) = 0) or (Length(B[0]) = 0) then
  begin
    WriteLn('MatrixAdd: Матрица B пустая, возвращаем A');
Halt;
    Exit(CopyMatrix(A));
  end;

  // Выбираем минимальные размеры
  rows := Min(Length(A), Length(B));
  cols := Min(Length(A[0]), Length(B[0]));

  SetLength(Result, rows, cols);

  for i := 0 to rows - 1 do
  begin
    for j := 0 to cols - 1 do
    begin
      Result[i][j] := A[i][j] + B[i][j];
    end;
  end;
end;

// Масштабирование матрицы (создаёт новую матрицу)
function ScaleMatrixCreate(const m: TDoubleMatrix; factor: Double): TDoubleMatrix;
var i,j: Integer;
begin Result:=nil;
  SetLength(Result, Length(m), Length(m[0]));
  for i := 0 to High(m) do
    for j := 0 to High(m[0]) do
      Result[i][j] := m[i][j] * factor;
end;

// Масштабирование матрицы (in-place модификация)
procedure ScaleMatrix(var m: TDoubleMatrix; factor: Double);
var i,j: Integer;
begin
  for i := 0 to High(m) do
    for j := 0 to High(m[0]) do
      m[i][j] := m[i][j] * factor;
end;

function CopyMatrix(const m: TDoubleMatrix): TDoubleMatrix;
var
  i: Integer;
begin
  // ✅ ЗАЩИТА: Проверяем входные данные
  if (Length(m) = 0) then
  begin
    SetLength(Result, 0, 0);
    Exit;
  end;

  SetLength(Result, Length(m));

  for i := 0 to High(m) do
  begin
    // ✅ ЗАЩИТА: Проверяем каждую строку
    if Length(m[i]) > 0 then
      Result[i] := Copy(m[i])
    else
      SetLength(Result[i], 0);
  end;
end;

function Dropout(const m: TDoubleMatrix; rate: Double): TDoubleMatrix;
var i,j: Integer;
begin
  Result := CopyMatrix(m);
  if rate > 0 then
  begin
    for i := 0 to High(m) do
      for j := 0 to High(m[0]) do
        if Random < rate then
          Result[i][j] := 0
        else
          Result[i][j] := m[i][j] / (1 - rate);
  end;
end;

function CreateRandomMatrix(rows, cols: Integer; minVal: Double; maxVal: Double): TDoubleMatrix;
var i,j: Integer;
begin Result:=nil;
  SetLength(Result, rows, cols);
  for i := 0 to rows - 1 do begin
    for j := 0 to cols - 1 do begin
      Result[i][j] := minVal + (maxVal - minVal) * Random;
    end;
  end;
end;

procedure AddNoise(var Matrix: TDoubleMatrix; NoiseLevel: Double);
var i,j: Integer;
begin
  for i := 0 to High(Matrix) do
    for j := 0 to High(Matrix[i]) do
      Matrix[i][j] := Matrix[i][j] + (Random * 2 - 1) * NoiseLevel;
end;

procedure FillMatrix(var matrix: TDoubleMatrix; value: Double);
var i,j: Integer;
begin
  for i := 0 to High(matrix) do
    for j := 0 to High(matrix[i]) do
      matrix[i][j] := value;
end;

procedure FillArray(var arr: TDoubleArray; value: Double);
var i: Integer;
begin
for i := 0 to High(arr) do arr[i] := value;
end;

procedure UpdateMatrixAdam(var params, grads: TDoubleMatrix; 
                         var state: TAdamState; learningRate: Double);
var
  i,j: Integer;
  mHat, vHat: Double;
begin
  Inc(state.Timestep);
  for i := 0 to High(params) do
    for j := 0 to High(params[0]) do begin
      state.M[i][j] := state.Beta1 * state.M[i][j] + (1 - state.Beta1) * grads[i][j];
      state.V[i][j] := state.Beta2 * state.V[i][j] + (1 - state.Beta2) * Sqr(grads[i][j]);

      mHat := state.M[i][j] / (1 - Power(state.Beta1, state.Timestep));
      vHat := state.V[i][j] / (1 - Power(state.Beta2, state.Timestep));

      params[i][j] := params[i][j] - learningRate * mHat / (Sqrt(vHat) + 1e-8);
    end;
end;

function ScaleMatrixToSize(const m: TDoubleMatrix; newRows, newCols: Integer): TDoubleMatrix;
var
  i, j: Integer;
begin
  if (newRows <= 0) or (newCols <= 0) then
  begin
    SetLength(Result, 0, 0);
    Exit;
  end;

  SetLength(Result, newRows, newCols);
  FillMatrix(Result, 0.0);

  // Копируем данные с обрезкой или дополнением нулями
  for i := 0 to Min(newRows - 1, High(m)) do
  begin
    for j := 0 to Min(newCols - 1, High(m[i])) do
    begin
      Result[i][j] := m[i][j];
    end;
  end;
end;

procedure PrintMatrix(const matrix: TDoubleMatrix; maxRows: Integer = 10; maxCols: Integer = 10; precision: Integer = 4);
var
  i, j, displayedRows, displayedCols: Integer;
  rowStr: string;
begin
  if Length(matrix) = 0 then begin
    WriteLn('Matrix is empty');
    Exit;
  end;

  // Определяем сколько строк и столбцов будем выводить
  displayedRows := Min(maxRows, Length(matrix));
  displayedCols := Min(maxCols, Length(matrix[0]));

  WriteLn('Matrix [', Length(matrix), 'x', Length(matrix[0]), ']:');

  for i := 0 to displayedRows - 1 do begin
    rowStr := '';
    for j := 0 to displayedCols - 1 do begin
      // Форматируем число с заданной точностью
      rowStr := rowStr + Format('%.' + IntToStr(precision) + 'f', [matrix[i][j]]) + ' ';
    end;

    // Добавляем многоточие, если не все столбцы показаны
    if displayedCols < Length(matrix[0]) then rowStr := rowStr + '...';

    WriteLn(rowStr);
  end;

  // Добавляем многоточие, если не все строки показаны
  if displayedRows < Length(matrix) then
    WriteLn('... (', Length(matrix) - displayedRows, ' more rows)');
end;

procedure MatrixAddInPlace(var A: TDoubleMatrix; const B: TDoubleMatrix);
var
  i, j: Integer;
begin
  for i := 0 to High(A) do
    for j := 0 to High(A[i]) do
      A[i][j] := A[i][j] + B[i][j];
end;

procedure ScaleMatrixInPlace(var A: TDoubleMatrix; factor: Double);
var
  i, j: Integer;
begin
  for i := 0 to High(A) do
    for j := 0 to High(A[i]) do
      A[i][j] := A[i][j] * factor;
end;

function MatrixMultiplyFast(const A, B: TDoubleMatrix): TDoubleMatrix;
var
  i, j, k: Integer;
  sum: Double;
begin
  // Оптимизированная версия умножения с лучшей кэш-локальностью
  SetLength(Result, Length(A), Length(B[0]));

  // Заранее заполняем нулями
  for i := 0 to High(Result) do
    for j := 0 to High(Result[0]) do
      Result[i][j] := 0.0;

  for i := 0 to High(A) do
  begin
    for k := 0 to High(B) do
    begin
      if A[i][k] <> 0 then // Пропускаем нулевые значения
      begin
        for j := 0 to High(B[0]) do
        begin
          Result[i][j] := Result[i][j] + A[i][k] * B[k][j];
        end;
      end;
    end;
  end;
end;

initialization
Randomize;
end.