unit SymbolicEngineUnit;

{$MODE OBJFPC}{$H+}
{$CODEPAGE UTF8}
{$RANGECHECKS ON}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


interface

uses
  SysUtils, Classes, DataUtils, ucs4unit, ucs4opunit, ucs4functionsunit, TextEmbeddings, lazutf8, Math;

type
  TFact = record
    Subject: string;
    Relation: string;
    _Object: string;
    Confidence: Double;
  end;
  TFactArray = array of TFact;

  TRule = record
    Condition: string; // Логическое выражение
    Action: string;    // Действие/вывод
    Priority: Integer;
    UsageCount: Integer;
    Cition:string;
  end;
  TRuleArray = array of TRule;

  TSymbolicEngine = class
  private
    FFacts: TFactArray;
    FRules: TRuleArray;
    FWorkingMemory: TStringList;

    // Вспомогательные методы для парсинга условий
    function ParseCondition(const Condition: string): TStringArray;
    function EvaluateCondition(const Condition: string; const Input: string): Boolean;

  public
    constructor Create;
    destructor Destroy; override;

    // Управление фактами
    procedure AddFact(const Subject, Relation, _Object: string; Confidence: Double = 1.0);
    function FindFacts(const Subject, Relation, _Object: string): TFactArray;
    procedure RemoveFact(Index: Integer);

    // Управление правилами
    procedure AddRule(const Condition, Action: string; Priority: Integer = 1);
    function FindMatchingRules(const Input: string): TRuleArray;

    // Основные функции
    function ExecuteRule(const Input: string): string;
    function Infer(const Query: string): TFactArray;

    // Обучение и извлечение правил
    function ShouldExtractRule(const UserMessage, AIResponse: string): Boolean;
    function ExtractPatterns(const Text: string): TStringArray;
    function BuildRuleFromPatterns(const Patterns: TStringArray; const Response: string): string;
    procedure LearnFromDialogue(const UserMessage, AIResponse: string);

    // Сохранение/загрузка
    procedure SaveToFile(const Filename: string);
    procedure LoadFromFile(const Filename: string);

    // Утилиты
    function GetStats: string;
    procedure ClearRules;

    procedure DebugRuleMatching(const Input: string);
    procedure AddEssentialRules;
  end;

// Глобальные функции для работы с текстом
function ContainsAny(const Text: string; const Patterns: array of string): Boolean;
function ExtractKeywords(const Text: string): TStringArray;
function CalculatePatternScore(const Pattern: string): Double;

function ParseEntitiesFromOutput(const OutputMatrix: TDoubleMatrix): TStringArray;
function BuildSymbolicContext(const History: TStringList): string;

var
  SymbolicEngine: TSymbolicEngine;

implementation

function ContainsAny(const Text: string; const Patterns: array of string): Boolean;
var
  I: Integer;
  LowerText: string;
l,p:ucs4;
begin
LowerText := UTF8LowerCase(Text);
l.Init; p.Init;
l:=LowerText;
  for I := 0 to High(Patterns) do
  begin
    p:=UTF8LowerCase(Patterns[I]);
    if Contains(l, p) then begin p.Clear; Exit(True); end;
    p.Clear;
  end;
l.Clear;
Exit(false);
end;

function ExtractKeywords(const Text: string): TStringArray;
var
  tokens: TUC4Array;
  i, count: Integer;
  tokenStr: string;
t: ucs4;
begin
  SetLength(Result, 0);

  // Токенизируем текст
  t := text;
  tokens := TokenizeForNLP(NormalizeForAI(RemovePunctuation(t)));
  if Length(tokens) = 0 then Exit;

  // Фильтруем стоп-слова и короткие токены
  count := 0;
  SetLength(Result, Length(tokens));

  for i := 0 to High(tokens) do
  begin
    tokenStr := tokens[i].ToUTF8.Trim;

    // Пропускаем короткие слова и стоп-слова
    if (Length(tokenStr) > 2) and 
       (not ContainsAny(tokenStr, ['это', 'что', 'как', 'где', 'когда', 'почему', 'зачем'])) then
    begin
      Result[count] := tokenStr;
      Inc(count);
    end;
  end;

  SetLength(Result, count);
end;

function CalculatePatternScore(const Pattern: string): Double;
var
  keywords: TStringArray;
  i: Integer;
begin
  Result := 0.0;
  keywords := ExtractKeywords(Pattern);

  // Оцениваем паттерн на основе количества ключевых слов и их значимости
  for i := 0 to High(keywords) do
  begin
    // Простая эвристика: более длинные слова обычно более значимы
    Result := Result + Length(keywords[i]) * 0.1;

    // Особо значимые слова
    if ContainsAny(keywords[i], ['погод', 'врем', 'температур', 'градус']) then
      Result := Result + 1.0;
    if ContainsAny(keywords[i], ['имя', 'зовут', 'называ']) then
      Result := Result + 1.0;
    if ContainsAny(keywords[i], ['счет', 'математ', 'числ', 'плюс', 'минус']) then
      Result := Result + 1.5;
  end;
end;

constructor TSymbolicEngine.Create;
begin
  inherited;
  FWorkingMemory := TStringList.Create;
  SetLength(FFacts, 0);
  SetLength(FRules, 0);

  // Базовые правила
  AddRule('Contains(Input, "привет") OR Contains(Input, "здравств")', 'Greeting', 10);
  AddRule('Contains(Input, "пока") OR Contains(Input, "до свидан")', 'Farewell', 10);
  AddRule('Contains(Input, "спасибо") OR Contains(Input, "благодар")', 'Thanks', 8);
  AddRule('EndsWith(Input, "?") AND Length(Input) > 3', 'Question', 5);
  AddRule('Contains(Input, "погод")', 'WeatherQuery', 7);
  AddRule('Contains(Input, "имя") OR Contains(Input, "зовут")', 'NameQuery', 6);
  AddRule('Contains(Input, "счет") OR Contains(Input, "посчитай")', 'MathQuery', 9);

  // Добавляем больше конкретных правил
  AddRule('Contains(Input, "где") AND Contains(Input, "мы")', 'LocationQuery', 8);
  AddRule('Contains(Input, "москв") OR Contains(Input, "питер") OR Contains(Input, "город")', 'CityQuery', 7);
  AddRule('Contains(Input, "нормальн") OR Contains(Input, "хорош") OR Contains(Input, "отличн")', 'PositiveState', 6);
  AddRule('StartsWith(Input, "мы ") AND Length(Input) > 5', 'WeStatement', 5);

  // Добавляем правила для вопросов о личности
  AddRule('Contains(Input, "кто ты") OR Contains(Input, "твое имя") OR Contains(Input, "тебя зовут")', 'IdentityQuery', 9);
  AddRule('Contains(Input, "что ты") AND EndsWith(Input, "?")', 'CapabilityQuery', 8);
  AddRule('StartsWith(Input, "кто ") AND EndsWith(Input, "?")', 'WhoQuestion', 7);
  AddRule('StartsWith(Input, "что ") AND EndsWith(Input, "?")', 'WhatQuestion', 7);

  AddEssentialRules;
end;

destructor TSymbolicEngine.Destroy;
begin
  FWorkingMemory.Free;
  inherited;
end;

procedure TSymbolicEngine.AddFact(const Subject, Relation, _Object: string; Confidence: Double);
begin
  SetLength(FFacts, Length(FFacts) + 1);
  with FFacts[High(FFacts)] do
  begin
    Subject := Subject;
    Relation := Relation;
    _Object := _Object;
    Confidence := Confidence;
  end;
end;

function TSymbolicEngine.FindFacts(const Subject, Relation, _Object: string): TFactArray;
var
  i, count: Integer;
begin
  SetLength(Result, 0);
  count := 0;

  for i := 0 to High(FFacts) do
  begin
    if ((Subject = '') or (FFacts[i].Subject = Subject)) and
       ((Relation = '') or (FFacts[i].Relation = Relation)) and
       ((_Object = '') or (FFacts[i]._Object = _Object)) then
    begin
      SetLength(Result, count + 1);
      Result[count] := FFacts[i];
      Inc(count);
    end;
  end;
end;

procedure TSymbolicEngine.RemoveFact(Index: Integer);
var
  i: Integer;
begin
  if (Index < 0) or (Index > High(FFacts)) then Exit;

  for i := Index to High(FFacts) - 1 do
    FFacts[i] := FFacts[i + 1];

  SetLength(FFacts, Length(FFacts) - 1);
end;

procedure TSymbolicEngine.AddRule(const Condition, Action: string; Priority: Integer);
begin
SetLength(FRules, Length(FRules) + 1);
FRules[High(FRules)].Condition := Condition;
FRules[High(FRules)].Action := Action;
FRules[High(FRules)].Priority := Priority;
FRules[High(FRules)].UsageCount := 0;
end;

function TSymbolicEngine.ParseCondition(const Condition: string): TStringArray;
begin
  // Простой парсер условий - разбиваем по OR и AND
  SetLength(Result, 0);

  if Condition.Contains(' OR ') then
  begin
    SetLength(Result, 2);
    Result[0] := Copy(Condition, 1, Pos(' OR ', Condition) - 1);
    Result[1] := Copy(Condition, Pos(' OR ', Condition) + 4, MaxInt);
  end
  else if Condition.Contains(' AND ') then
  begin
    SetLength(Result, 2);
    Result[0] := Copy(Condition, 1, Pos(' AND ', Condition) - 1);
    Result[1] := Copy(Condition, Pos(' AND ', Condition) + 5, MaxInt);
  end
  else
  begin
    SetLength(Result, 1);
    Result[0] := Condition;
  end;
end;

function TSymbolicEngine.EvaluateCondition(const Condition: string; const Input: string): Boolean;
var
  parts: TStringArray;
  lowerInput: string;
l,p:ucs4;
begin
  Result := False;
  lowerInput := UTF8LowerCase(Input.Trim);
  l.Init; p.Init;
  l:=lowerInput;

//WriteLn('TSymbolicEngine.EvaluateCondition Condition = ', Condition, '   Input = ', Input);
//Sleep(500);

  // Упрощаем условия для лучшего распознавания
  if Condition.StartsWith('Contains(Input, "') then
  begin
    parts := Condition.Split(['"']);
    if Length(parts) >= 2 then
    begin
      // Проверяем вхождение подстроки
      p:=UTF8LowerCase(parts[1]);
      Result := Contains(l,p);
    end;
  end
  else if StartsWith(Condition,'StartsWith(Input, "') then
  begin
    parts := Condition.Split(['"']);
    if Length(parts) >= 2 then begin
      p:=UTF8LowerCase(parts[1]);
      Result := StartsWith(l,p);
    end;
  end
  else if Condition.StartsWith('EndsWith(Input, "') then
  begin
    parts := Condition.Split(['"']);
    if Length(parts) >= 2 then begin
      p:=UTF8LowerCase(parts[1]);
      Result := EndsWith(l,p);
    end;
  end
  else if Condition.Contains(' OR ') then
  begin
    parts := ParseCondition(Condition);
    Result := EvaluateCondition(parts[0], Input) or EvaluateCondition(parts[1], Input);
  end
  else if Condition.Contains(' AND ') then
  begin
    parts := ParseCondition(Condition);
    Result := EvaluateCondition(parts[0], Input) and EvaluateCondition(parts[1], Input);
  end;

  // Дополнительная отладка
  if Result then
    WriteLn('  ✓ Правило сработало: ', Condition)
  else
    WriteLn('  ✗ Правило не сработало: ', Condition);

l.Clear; p.Clear;
end;

function TSymbolicEngine.FindMatchingRules(const Input: string): TRuleArray;
var
  i, count: Integer;
begin
  SetLength(Result, 0);
  count := 0;

  for i := 0 to High(FRules) do
  begin
    if EvaluateCondition(FRules[i].Condition, Input) then
    begin
      SetLength(Result, count + 1);
      Result[count] := FRules[i];
      Inc(count);
    end;
  end;
end;

function TSymbolicEngine.ExecuteRule(const Input: string): string;
var
  matchingRules: TRuleArray;
  bestRule: TRule;
  i: Integer;
begin
  Result := '';
  matchingRules := FindMatchingRules(Input);

  if Length(matchingRules) = 0 then Exit;

  // Выбираем правило с наивысшим приоритетом
  bestRule := matchingRules[0];
  for i := 1 to High(matchingRules) do
  begin
    if matchingRules[i].Priority > bestRule.Priority then
      bestRule := matchingRules[i];
  end;

  // Обновляем счетчик использования
  for i := 0 to High(FRules) do
  begin
    if FRules[i].Condition = bestRule.Cition then
    begin
      Inc(FRules[i].UsageCount);
      Break;
    end;
  end;

  Result := bestRule.Action;
end;

function TSymbolicEngine.Infer(const Query: string): TFactArray;
begin
  // Упрощенный логический вывод
  // В реальной реализации здесь будет полноценный логический движок
  Result := FindFacts('', '', '');
end;

// Реализация недостающих функций

function TSymbolicEngine.ShouldExtractRule(const UserMessage, AIResponse: string): Boolean;
var
  msgScore, responseScore: Double;
begin
  // Решаем, стоит ли извлекать правило из этого диалога

  // Критерии для извлечения правил:
  // 1. Сообщение не слишком короткое
  // 2. Ответ не шаблонный
  // 3. Высокая уверенность в качестве диалога

  if Length(UserMessage) < 5 then Exit(False);
  if Length(AIResponse) < 3 then Exit(False);

  // Оцениваем значимость сообщения
  msgScore := CalculatePatternScore(UserMessage);
  responseScore := CalculatePatternScore(AIResponse);

  // Пропускаем простые приветствия и прощания
  if ContainsAny(UserMessage, ['привет', 'пока', 'спасибо']) and 
     ContainsAny(AIResponse, ['Здравствуйте', 'До свидания', 'Пожалуйста']) then
    Exit(False);

  // Извлекаем правила для значимых диалогов
  Result := (msgScore > 1.0) and (responseScore > 0.5);
end;

function TSymbolicEngine.ExtractPatterns(const Text: string): TStringArray;
var
  keywords: TStringArray;
  i: Integer;
begin
  SetLength(Result, 0);
  keywords := ExtractKeywords(Text);

  // Формируем паттерны на основе ключевых слов
  for i := 0 to High(keywords) do
  begin
    SetLength(Result, Length(Result) + 1);

    // Создаем условие для каждого значимого ключевого слова
    if Length(keywords[i]) > 3 then
      Result[High(Result)] := 'Contains(Input, "' + keywords[i] + '")'
    else
      SetLength(Result, Length(Result) - 1); // Удаляем если слово слишком короткое
  end;
end;

function TSymbolicEngine.BuildRuleFromPatterns(const Patterns: TStringArray; const Response: string): string;
var
  condition: string;
  i: Integer;
begin
  if Length(Patterns) = 0 then
  begin
    Result := '';
    Exit;
  end;

  // Объединяем паттерны через OR
  condition := Patterns[0];
  for i := 1 to High(Patterns) do
    condition := condition + ' OR ' + Patterns[i];

  Result := condition;
end;

procedure TSymbolicEngine.LearnFromDialogue(const UserMessage, AIResponse: string);
var
  patterns: TStringArray;
  newRule: string;
begin
  if not ShouldExtractRule(UserMessage, AIResponse) then Exit;

  WriteLn('Символьное обучение: извлекаем правило из диалога');
  WriteLn('  Вход: ', UserMessage);
  WriteLn('  Выход: ', AIResponse);

  patterns := ExtractPatterns(UserMessage);
  if Length(patterns) = 0 then Exit;

  newRule := BuildRuleFromPatterns(patterns, AIResponse);
  if newRule <> '' then
  begin
    AddRule(newRule, AIResponse, 3); // Низкий приоритет для изученных правил
    WriteLn('  Добавлено правило: ', newRule, ' -> ', AIResponse);
  end;
end;

procedure TSymbolicEngine.SaveToFile(const Filename: string);
var
  F: TextFile;
  i: Integer;
begin
  AssignFile(F, Filename);
  try
    Rewrite(F);

    // Сохраняем факты
    WriteLn(F, '[Facts]');
    for i := 0 to High(FFacts) do
    begin
      WriteLn(F, FFacts[i].Subject, '|', FFacts[i].Relation, '|', 
              FFacts[i]._Object, '|', FFacts[i].Confidence:0:4);
    end;

    // Сохраняем правила
    WriteLn(F, '[Rules]');
    for i := 0 to High(FRules) do
    begin
      WriteLn(F, FRules[i].Condition, '|', FRules[i].Action, '|', 
              FRules[i].Priority, '|', FRules[i].UsageCount);
    end;

  finally
    CloseFile(F);
  end;
end;

procedure TSymbolicEngine.LoadFromFile(const Filename: string);
var
  F: TextFile;
  Line: string;
  Parts: TStringArray;
  section: string;
begin
  if not FileExists(Filename) then Exit;

  AssignFile(F, Filename);
  try
    Reset(F);
    section := '';

    while not Eof(F) do
    begin
      ReadLn(F, Line);
      Line := Line.Trim;

      if Line = '' then Continue;

      if Line = '[Facts]' then
        section := 'facts'
      else if Line = '[Rules]' then
        section := 'rules'
      else if section = 'facts' then
      begin
        Parts := Line.Split(['|']);
        if Length(Parts) >= 4 then
        begin
          AddFact(Parts[0], Parts[1], Parts[2], StrToFloatDef(Parts[3], 1.0));
        end;
      end
      else if section = 'rules' then
      begin
        Parts := Line.Split(['|']);
        if Length(Parts) >= 3 then
        begin
          AddRule(Parts[0], Parts[1], StrToIntDef(Parts[2], 1));
        end;
      end;
    end;

  finally
    CloseFile(F);
  end;
end;

function TSymbolicEngine.GetStats: string;
begin
  Result := Format('Фактов: %d, Правил: %d', [Length(FFacts), Length(FRules)]);
end;

procedure TSymbolicEngine.ClearRules;
begin
  SetLength(FRules, 0);
end;

function ParseEntitiesFromOutput(const OutputMatrix: TDoubleMatrix): TStringArray;
var
  i, j, entityCount: Integer;
  maxVal: Double;
  maxIndex: Integer;
  entities: array of string;
begin
  SetLength(Result, 0);
  if (Length(OutputMatrix) = 0) or (Length(OutputMatrix[0]) = 0) then Exit;

  // Упрощенная реализация - в реальности здесь будет полноценное извлечение сущностей
  // Сейчас просто возвращаем пустой массив для демонстрации

  // В реальной реализации здесь будет:
  // 1. Классификация токенов на сущности (PER, ORG, LOC и т.д.)
  // 2. Извлечение отношений между сущностями
  // 3. Формирование фактов

  WriteLn('ParseEntitiesFromOutput: извлечение сущностей из матрицы ', 
          Length(OutputMatrix), 'x', Length(OutputMatrix[0]));

  // Заглушка - возвращаем пустой массив
  // В рабочей версии здесь будет реальное извлечение сущностей
end;

function BuildSymbolicContext(const History: TStringList): string;
var
  i: Integer;
  contextFacts: TStringList;
  recentMessages: string;
  fact: TFact;
  facts: TFactArray;
begin
  contextFacts := TStringList.Create;
  try
    // Собираем последние сообщения для контекста
    recentMessages := '';
    for i := Max(0, History.Count - 4) to History.Count - 1 do
    begin
      if i >= 0 then
        recentMessages := recentMessages + History[i] + ' ';
    end;

    // Извлекаем факты из истории
    facts := SymbolicEngine.FindFacts('', '', '');

    // Формируем контекст в виде строки
    Result := 'Recent: ' + recentMessages.Trim;

    if Length(facts) > 0 then
    begin
      Result := Result + ' | Facts: ';
      for i := 0 to Min(High(facts), 2) do // Берем только 3 последних факта
      begin
        Result := Result + facts[i].Subject + '-' + facts[i].Relation + '-' + 
                 facts[i]._Object + '; ';
      end;
    end;

    // Добавляем информацию о текущих правилах
    Result := Result + ' | Rules: ' + IntToStr(Length(SymbolicEngine.FRules));

  finally
    contextFacts.Free;
  end;
end;

procedure TSymbolicEngine.DebugRuleMatching(const Input: string);
var
  i: Integer;
  matches: Boolean;
begin
  WriteLn('=== ОТЛАДКА ПРАВИЛ ДЛЯ: "', Input, '" ===');

  for i := 0 to High(FRules) do
  begin
    matches := EvaluateCondition(FRules[i].Condition, Input);
    WriteLn('Правило ', i, ': "', FRules[i].Condition, '" -> ', 
            IfThen(matches, 'СОВПАЛО', 'не совпало'));
  end;

  WriteLn('=== КОНЕЦ ОТЛАДКИ ПРАВИЛ ===');
end;

procedure TSymbolicEngine.AddEssentialRules;
begin
  // Принудительно добавляем критически важные правила
  AddRule('Input = "кто ты" OR Input = "кто ты?"', 'IdentityQuery', 20);
  AddRule('Input = "что ты" OR Input = "что ты?"', 'CapabilityQuery', 20);
  AddRule('Input = "как дела" OR Input = "как дела?"', 'HowAreYou', 20);
  AddRule('Input = "привет" OR Input = "здравствуй"', 'Greeting', 20);
end;

//initialization
//  SymbolicEngine := TSymbolicEngine.Create;

//finalization
//  SymbolicEngine.Free;

end.