unit TextEmbeddings;
{$MODE OBJFPC}{$H+}{$CODEPAGE UTF8}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


interface

uses
  SysUtils, Classes, DataUtils, MatrixOps, ucs4unit, ucs4opunit, ucs4functionsunit, Word2Vec, Math;

// Статистика использования
procedure PrintEmbeddingStats;
function TextToEmbedding(const text: ucs4; embeddingSize: Integer): TDoubleArray;
function CreateTokenEmbeddings(const tokens: TUC4Array; embeddingSize: Integer): TDoubleMatrix;
function TextsToMatrix(const texts: TUC4Array; embeddingSize: Integer): TDoubleMatrix;

// ✅ НОВЫЕ ОПТИМИЗИРОВАННЫЕ ФУНКЦИИ С ИНДЕКСАМИ
function TextToEmbeddingIndices(const text: ucs4; Embeddings: TWordEmbeddings; out WordIndices: TIntegerArray): Boolean;
function CreateEmbeddingFromIndices(const WordIndices: TIntegerArray; Embeddings: TWordEmbeddings; embeddingSize: Integer): TDoubleArray;
function TextsToMatrixIndices(const texts: TUC4Array; Embeddings: TWordEmbeddings; embeddingSize: Integer): TDoubleMatrix;
function TextsToMatrixFallback(const texts: TUC4Array; embeddingSize: Integer): TDoubleMatrix;
function FastTextToMatrixIndices(const Text: string; Embeddings: TWordEmbeddings; EmbeddingSize: Integer): TDoubleMatrix;
function FastTextsToMatrixIndices(const Texts: array of string; Embeddings: TWordEmbeddings; EmbeddingSize: Integer): TDoubleMatrix;
function GetCachedEmbedding(const Text: string; EmbeddingSize: Integer): TDoubleMatrix;

implementation

var
  FallbackUsageCount: Integer = 0;
  Word2VecUsageCount: Integer = 0;
  IndexedUsageCount: Integer = 0;
  EmbeddingCache: TStringList;

procedure PrintEmbeddingStats;
var
total:Integer;
begin
  WriteLn('=== EMBEDDING STATISTICS ===');
  WriteLn('Word2Vec usage: ', Word2VecUsageCount);
  WriteLn('Indexed usage: ', IndexedUsageCount);
  WriteLn('Fallback usage: ', FallbackUsageCount);
  total := Word2VecUsageCount + IndexedUsageCount + FallbackUsageCount;
  if total > 0 then begin
    WriteLn('Indexed rate: ', (IndexedUsageCount * 100) / total:0:1, '%');
    WriteLn('Fallback rate: ', (FallbackUsageCount * 100) / total:0:1, '%');
  end;
end;

// ✅ ОПТИМИЗИРОВАННАЯ ВЕРСИЯ: Использование индексов вместо строк
function NotContains(sl:TStringList;word:string):bytebool;
var f:Integer;
begin
for f := 0 to sl.Count-1 do if sl[f] = word then Exit(false);
Exit(true);
end;

function TextToEmbeddingIndices(const text: ucs4; Embeddings: TWordEmbeddings; out WordIndices: TIntegerArray): Boolean;
var
  tokens: TUC4Array;
  i, wordIndex: Integer;
  word: string;
  validWords: Integer;
  KnownUnknownWords: TStringList; // ← ДОБАВИТЬ ЭТО
begin
  Result := False;
  SetLength(WordIndices, 0);

  if not Assigned(Embeddings) then
    Exit;

  // ✅ ИНИЦИАЛИЗИРУЕМ KnownUnknownWords
  KnownUnknownWords := TStringList.Create;
  KnownUnknownWords.Sorted := True;
  KnownUnknownWords.Duplicates := dupIgnore;
  KnownUnknownWords.CaseSensitive := False;

  try
    // Токенизируем текст
    tokens := TokenizeForNLP(NormalizeForAI(RemovePunctuation(text)));

    if Length(tokens) = 0 then Exit;

    SetLength(WordIndices, Length(tokens));
    validWords := 0;

    if VerboseEmbeddingLogs then
      WriteLn('  TextToEmbeddingIndices: токенизировано ', Length(tokens), ' слов');

    // Для каждого токена получаем индекс в словаре
    for i := 0 to High(tokens) do begin
      word := tokens[i].ToUTF8;

      // ✅ ЗАЩИТА: Проверяем что слово не пустое
      if word.Trim = '' then
        Continue;

      // ✅ ЗАЩИТА: Используем безопасный поиск
      try
        wordIndex := Embeddings.GetWordIndex(word);

        if wordIndex >= 0 then begin
          // ✅ ЗАЩИТА: Проверяем границы массива
          if validWords < Length(WordIndices) then begin
            WordIndices[validWords] := wordIndex;
            Inc(validWords);
          end else begin
            WriteLn('Предупреждение: превышение размера WordIndices');
            Break;
          end;
        end else begin
          // Логируем только ПЕРВОЕ вхождение неизвестного слова
          if NotContains(KnownUnknownWords,word) then begin
            // ✅ ИСПОЛЬЗУЕМ ПРАВИЛЬНОЕ ИМЯ МЕТОДА
            if Embeddings.IsInUnknownCache(word) then
              WriteLn('    Слово "', word, '" из кэша неизвестных')
            else if VerboseEmbeddingLogs then
              WriteLn('    Слово "', word, '" не найдено в словаре (новое)');

            KnownUnknownWords.Add(word);
          end;
        end;
      except
        on E: Exception do begin
          WriteLn('Ошибка поиска слова "', word, '": ', E.Message);
          Continue; // Продолжаем со следующим словом
        end;
      end;
    end;

    // Обрезаем до реального количества найденных слов
    SetLength(WordIndices, validWords);
    Result := validWords > 0;

    if VerboseEmbeddingLogs then
      WriteLn('  Найдено слов в словаре: ', validWords, '/', Length(tokens));

  except
    on E: Exception do begin
      WriteLn('Критическая ошибка в TextToEmbeddingIndices: ', E.Message);
      Halt;
    end;
  end;
KnownUnknownWords.Free;
end;

// ✅ ОПТИМИЗИРОВАННАЯ ВЕРСИЯ: Создание эмбеддинга из индексов
function CreateEmbeddingFromIndices(const WordIndices: TIntegerArray; Embeddings: TWordEmbeddings; embeddingSize: Integer): TDoubleArray;
var
  i, j: Integer;
  wordEmb: TDoubleArray;
begin
  SetLength(Result, embeddingSize);
  FillArray(Result, 0.0);

  if (Length(WordIndices) = 0) or not Assigned(Embeddings) then Exit;

  if VerboseEmbeddingLogs then
    WriteLn('  CreateEmbeddingFromIndices: усреднение ', Length(WordIndices), ' эмбеддингов');

  try
    // ✅ ОПТИМИЗАЦИЯ: Прямой доступ к эмбеддингам по индексу
    for i := 0 to High(WordIndices) do begin
      // ✅ ЗАЩИТА: Проверяем валидность индекса
      if (WordIndices[i] < 0) or (WordIndices[i] >= Embeddings.VocabularySize) then begin
        if VerboseEmbeddingLogs then
          WriteLn('    Предупреждение: неверный индекс слова: ', WordIndices[i]);
        Continue;
      end;

      // Используем быструю версию без проверок
      wordEmb := Embeddings.GetEmbeddingFastByIndex(WordIndices[i]);

      if Length(wordEmb) > 0 then begin
        // ✅ ЗАЩИТА: Проверяем размер эмбеддинга
        for j := 0 to Min(embeddingSize, Length(wordEmb)) - 1 do
          Result[j] := Result[j] + wordEmb[j];
      end;
    end;

    // Усредняем эмбеддинги
    if Length(WordIndices) > 0 then begin
      for j := 0 to embeddingSize - 1 do
        Result[j] := Result[j] / Length(WordIndices);
    end;

  except
    on E: Exception do begin
      WriteLn('Ошибка в CreateEmbeddingFromIndices: ', E.Message);
Halt;
      FillArray(Result, 0.0); // Возвращаем нулевой эмбеддинг при ошибке
    end;
  end;
end;

// ✅ ОПТИМИЗИРОВАННАЯ ВЕРСИЯ: Создание матрицы из текстов с использованием индексов
function TextsToMatrixIndices(const texts: TUC4Array; Embeddings: TWordEmbeddings; embeddingSize: Integer): TDoubleMatrix;
var
  i: Integer;
  wordIndices: TIntegerArray;
begin
  if VerboseEmbeddingLogs then
    WriteLn('TextsToMatrixIndices: обработка ', Length(texts), ' текстов с индексами');

  SetLength(Result, Length(texts));

  for i := 0 to High(texts) do begin
    try
      // Пытаемся получить индексы слов
      if TextToEmbeddingIndices(texts[i], Embeddings, wordIndices) then begin
        // Создаем эмбеддинг из индексов
        Result[i] := CreateEmbeddingFromIndices(wordIndices, Embeddings, embeddingSize);

        if VerboseEmbeddingLogs then
          WriteLn('  Текст ', i, ': создан эмбеддинг из ', Length(wordIndices), ' слов');
      end else begin
        // Fallback: создаем нулевой эмбеддинг
        SetLength(Result[i], embeddingSize);
        FillArray(Result[i], 0.0);

        if VerboseEmbeddingLogs then
          WriteLn('  Текст ', i, ': использован fallback (нет известных слов)');
      end;
    except
      on E: Exception do begin
        WriteLn('ОШИБКА в примере ', i, ': ', E.Message);
        // Создаем безопасный эмбеддинг
        SetLength(Result[i], embeddingSize);
        FillArray(Result[i], 0.0);
      end;
    end;
  end;

  Inc(IndexedUsageCount);
end;

// 🔄 СТАРАЯ ВЕРСИЯ: Для обратной совместимости
function TextsToMatrixFallback(const texts: TUC4Array; embeddingSize: Integer): TDoubleMatrix;
var
  i,j: Integer;
  textStr : string;
begin
  if VerboseEmbeddingLogs then
    WriteLn('TextsToMatrixFallback: input texts count: ', Length(texts));

  if Length(texts) = 0 then begin
    WriteLn('Warning: No texts provided');
    SetLength(Result, 1, embeddingSize);
    for j := 0 to embeddingSize - 1 do
      Result[0][j] := Random * 0.02 - 0.01;
    Exit;
  end;

  SetLength(Result, Length(texts));

  for i := 0 to High(texts) do begin
    textStr := texts[i].ToUTF8;
    if textStr.Trim = '' then begin
      WriteLn('Warning: Empty text at index ', i);
      SetLength(Result[i], embeddingSize);
      for j := 0 to embeddingSize - 1 do
        Result[i][j] := Random * 0.02 - 0.01;
      Continue;
    end;

    Result[i] := TextToEmbedding(texts[i], embeddingSize);
  end;

  Inc(Word2VecUsageCount);
end;

// 🎯 ОСНОВНАЯ ФУНКЦИЯ: Автоматически выбирает оптимальный метод
function TextsToMatrix(const texts: TUC4Array; embeddingSize: Integer): TDoubleMatrix;
begin
  // ✅ АВТОМАТИЧЕСКОЕ ПЕРЕКЛЮЧЕНИЕ: Используем оптимизированную версию если доступна
  if Assigned(WordEmbeddings) then
  begin
    if VerboseEmbeddingLogs then
      WriteLn('TextsToMatrix: используем оптимизированную версию с индексами');

    Result := TextsToMatrixIndices(texts, WordEmbeddings, embeddingSize);
  end
  else
  begin
    if VerboseEmbeddingLogs then
      WriteLn('TextsToMatrix: WordEmbeddings не доступен, используем стандартную версию');

    Result := TextsToMatrixFallback(texts, embeddingSize);
  end;
end;

// 🔄 СТАРАЯ ВЕРСИЯ: Для обратной совместимости
function TextToEmbedding(const text: ucs4; embeddingSize: Integer): TDoubleArray;
var
  tokens: TUC4Array;
  wordEmb, sumEmb: TDoubleArray;
  i, j, k, validWords: Integer;
  hash: DWord;
  cleanText: ucs4;
  word: string;

procedure UseFallback;
var f,ff:Integer;
begin
  // НАДЕЖНЫЙ ФАЛБЭК: Создаем эмбеддинг на основе хеширования
  if VerboseEmbeddingLogs then
    WriteLn('  Using reliable hash fallback');

  for ff := 0 to embeddingSize - 1 do begin
    hash := 2166136261;
    for f := 0 to cleanText.Length - 1 do begin
      hash := (hash xor cleanText[f]) * 16777619;
      hash := hash and $7FFFFFFF; // Ограничиваем диапазон
    end;
    // Нормализуем значения в диапазон [-0.5, 0.5]
    Result[ff] := ((hash mod 10000) / 10000) - 0.5;
  end;

  if VerboseEmbeddingLogs then
    WriteLn('  Fallback embedding created successfully');

  Inc(FallbackUsageCount);
end;

begin 
  // Всегда возвращаем массив правильного размера
  SetLength(Result, embeddingSize);

  // Очищаем текст
  cleanText := Trim(text);

  if cleanText.Length = 0 then begin
    if VerboseEmbeddingLogs then
      WriteLn('TextToEmbedding: Empty text after trimming');
    for j := 0 to embeddingSize - 1 do
      Result[j] := 0.0;
    Exit;
  end;

  if VerboseEmbeddingLogs then
    WriteLn('TextToEmbedding: Processing text: "', cleanText.ToUTF8, '"');

  if Assigned(WordEmbeddings) then
  begin
    try
      // Используем Word2Vec если модель загружена
      tokens := TokenizeForNLP(NormalizeForAI(RemovePunctuation(cleanText)));

      if VerboseEmbeddingLogs then
        WriteLn('  Tokens: ', Length(tokens));

      SetLength(sumEmb, WordEmbeddings.EmbeddingSize);
      FillArray(sumEmb, 0.0);
      validWords := 0;

      for i := 0 to High(tokens) do begin
        word := tokens[i].ToUTF8;

        if VerboseEmbeddingLogs then
          WriteLn('  Processing word: "', word, '"');

        wordEmb := WordEmbeddings.GetEmbeddingWithCache(word);
        if Length(wordEmb) > 0 then begin
          if VerboseEmbeddingLogs then
            WriteLn('    Word embedding found, length: ', Length(wordEmb));

          for j := 0 to High(wordEmb) do
            sumEmb[j] := sumEmb[j] + wordEmb[j];
          Inc(validWords);
        end else begin
          if VerboseEmbeddingLogs then
            WriteLn('    No embedding for word: "', word, '"');
        end;
      end;

      if VerboseEmbeddingLogs then
        WriteLn('  Valid words: ', validWords);

      if validWords > 0 then begin
        for j := 0 to High(sumEmb) do
          sumEmb[j] := sumEmb[j] / validWords;
      end else begin
        if VerboseEmbeddingLogs then
          WriteLn('  No valid words, using fallback');
        // Переходим на фалбэк
        UseFallback;
        Exit;
      end;

      // Копируем в результат
      for j := 0 to embeddingSize - 1 do begin
        if j < Length(sumEmb) then
          Result[j] := sumEmb[j]
        else
          Result[j] := 0.0;
      end;

      Inc(Word2VecUsageCount);

    except
      on E: Exception do begin
        WriteLn('  Error in Word2Vec processing: ', E.Message);
        Halt;
      end;
    end;
  end else begin
    // Word2Vec не доступен, используем фалбэк
    WriteLn('  WordEmbeddings not available, using hash fallback');
    Halt;
  end;

  if VerboseEmbeddingLogs then
    WriteLn('  Embedding created successfully');
end;

function CreateTokenEmbeddings(const tokens: TUC4Array; embeddingSize: Integer): TDoubleMatrix;
var
  i: Integer;
begin
  // ✅ ОПТИМИЗАЦИЯ: Используем индексную версию если доступна
  if Assigned(WordEmbeddings) then
  begin
    if VerboseEmbeddingLogs then
      WriteLn('CreateTokenEmbeddings: используем оптимизированную версию');

    Result := TextsToMatrixIndices(tokens, WordEmbeddings, embeddingSize);
  end
  else
  begin
    if VerboseEmbeddingLogs then
      WriteLn('CreateTokenEmbeddings: используем стандартную версию');

    SetLength(Result, Length(tokens));
    for i := 0 to High(tokens) do
      Result[i] := TextToEmbedding(tokens[i], embeddingSize);
  end;
end;

// ✅ ИНИЦИАЛИЗАЦИЯ: Можно вызвать в начале программы
procedure InitializeTextEmbeddings(Embeddings: TWordEmbeddings; Verbose: Boolean = True);
begin
  WordEmbeddings := Embeddings;
  VerboseEmbeddingLogs := Verbose;

  WriteLn('TextEmbeddings initialized');
  WriteLn('  WordEmbeddings: ', Assigned(WordEmbeddings));
  WriteLn('  Verbose logs: ', VerboseEmbeddingLogs);
end;

// В TextEmbeddings.pas - добавляем быстрые функции для обучения
function FastTextToMatrixIndices(const Text: string; Embeddings: TWordEmbeddings; EmbeddingSize: Integer): TDoubleMatrix;
var
  tokens: TUC4Array;
  i, j, tokenIndex: Integer;
begin
  tokens := TokenizeForNLP(Text);
  SetLength(Result, Length(tokens), EmbeddingSize);

  for i := 0 to High(tokens) do begin
    tokenIndex := Embeddings.GetWordIndex(tokens[i].ToUTF8);
    if tokenIndex >= 0 then begin
      // Прямое копирование эмбеддинга по индексу
      for j := 0 to EmbeddingSize - 1 do begin
        if j < Length(Embeddings.FEmbeddings[tokenIndex]) then
          Result[i][j] := Embeddings.FEmbeddings[tokenIndex][j]
        else
          Result[i][j] := 0.0;
      end;
    end else begin
      // Fallback: нулевой вектор для неизвестных слов
      for j := 0 to EmbeddingSize - 1 do
        Result[i][j] := 0.0;
    end;
  end;
end;

function CountTokens(const text: string): Integer;
var
  tokens: TUC4Array;
begin
  tokens := TokenizeForNLP(text);
  Result := Length(tokens);
end;

function FastTextsToMatrixIndices(const Texts: array of string; Embeddings: TWordEmbeddings; EmbeddingSize: Integer): TDoubleMatrix;
var
  totalRows, currentRow, i, j, tokenCount: Integer;
  singleMatrix: TDoubleMatrix;
begin
  // Быстрое объединение матриц
  totalRows := 0;
  for i := 0 to High(Texts) do begin
    tokenCount := CountTokens(Texts[i]);
    Inc(totalRows, Max(tokenCount, 1)); // Минимум 1 строка
  end;

  SetLength(Result, totalRows, EmbeddingSize);
  FillMatrix(Result, 0.0); // Инициализируем нулями
  currentRow := 0;

  for i := 0 to High(Texts) do begin
    singleMatrix := FastTextToMatrixIndices(Texts[i], Embeddings, EmbeddingSize);

    for j := 0 to High(singleMatrix) do begin
      if currentRow < totalRows then begin
        // Копируем строку
        Result[currentRow] := Copy(singleMatrix[j], 0, EmbeddingSize);
        Inc(currentRow);
      end;
    end;

    // Если токенов меньше чем ожидали, заполняем нулями
    while (currentRow < totalRows) and (j = Length(singleMatrix)) do begin
      FillArray(Result[currentRow], 0.0);
      Inc(currentRow);
    end;
  end;
end;

function GetCachedEmbedding(const Text: string; EmbeddingSize: Integer): TDoubleMatrix;
var
  cacheIndex: Integer;
  cacheKey: string;
begin
  cacheKey := Text + '|' + IntToStr(EmbeddingSize);
  cacheIndex := EmbeddingCache.IndexOfName(cacheKey);

  if cacheIndex >= 0 then begin
    // 🔥 ВОЗВРАЩАЕМ ИЗ КЭША
    Result := TDoubleMatrix(EmbeddingCache.Objects[cacheIndex]);
    Exit;
  end;

  // Вычисляем и кэшируем
  if Assigned(WordEmbeddings) then
    Result := FastTextToMatrixIndices(Text, WordEmbeddings, EmbeddingSize)
  else
    Result := TextsToMatrix([Text], EmbeddingSize);

  // Сохраняем в кэш (ограничиваем размер)
  if EmbeddingCache.Count > 1000 then
    EmbeddingCache.Delete(0);

  EmbeddingCache.AddObject(cacheKey, TObject(Result));
end;

initialization
  EmbeddingCache:= TStringList.Create;
  EmbeddingCache.Sorted := True;
  EmbeddingCache.Duplicates := dupError;
finalization
  EmbeddingCache.Free;
end.