unit UniversalFileReader;
{$MODE OBJFPC}{$H+}
{$MODESWITCH ADVANCEDRECORDS}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


interface

uses
  SysUtils, Classes, BaseUnix, Unix, termio, Math, syscall;

const
  MAP_POPULATE = $08000;
//  MLOCK_LIMIT = 16 * 1024 * 1024; // Макс. 16MB для блокировки в RAM
MLOCK_LIMIT = 256 * 1024 * 1024; // 256MB 
  SYS_mlock = 149; // Номер syscall для mlock в x86_64
  SYS_munlock = 150;

type
  TFileContent = record
    Data: Pointer;
    Size: Int64;
    FileName: String;
    IsText: Boolean;
    IsMapped: Boolean;
    InRAM: Boolean;
  end;

  { TUniversalFileReader }
  TUniversalFileReader = class
  private
    FFileHandle: cint;
    FContent: TFileContent;
    class function DetectTextFile(Data: Pointer; Size: Int64): Boolean; static;
    function LoadToRAM: Boolean;
    function SysMlock(addr: Pointer; len: SizeInt): Boolean;
    function SysMunlock(addr: Pointer; len: SizeInt): Boolean;
  public
    constructor Create(const FileName: String; LoadInRAM: Boolean = False);
    destructor Destroy; override;
    function ReadContent: TFileContent;
    function ReadAsText: String;
    function ReadAsBytes: TBytes;
    function IsTextFile: Boolean;
    function IsInRAM: Boolean;
    class function QuickRead(const FileName: String; LoadInRAM: Boolean = False): TFileContent; static;
    function GetMemoryUsage: String;
  end;

implementation

{ TUniversalFileReader }

function TUniversalFileReader.GetMemoryUsage: String;
begin
  Result := Format('Allocated: %d MB', [FContent.Size div (1024*1024)]);
end;

function TUniversalFileReader.SysMlock(addr: Pointer; len: SizeInt): Boolean;
begin
  Result := do_SysCall(SYS_mlock, QWord(addr), QWord(len)) = 0;
end;

function TUniversalFileReader.SysMunlock(addr: Pointer; len: SizeInt): Boolean;
begin
  Result := do_SysCall(SYS_munlock, QWord(addr), QWord(len)) = 0;
end;

constructor TUniversalFileReader.Create(const FileName: String; LoadInRAM: Boolean);
begin
  if fpAccess(FileName, R_OK) <> 0 then
    raise EFOpenError.CreateFmt('No read access to file %s', [FileName]);

  FContent.FileName := FileName;
  FContent.IsMapped := False;
  FContent.InRAM := False;
  
  FFileHandle := FpOpen(FileName, O_RDONLY or O_LARGEFILE);
  if FFileHandle = -1 then
    raise EFOpenError.CreateFmt('Cannot open file %s', [FileName]);

  FContent.Size := FpLSeek(FFileHandle, 0, SEEK_END);
  FpLSeek(FFileHandle, 0, SEEK_SET);

  FContent.Data := FpMMap(nil, FContent.Size, PROT_READ, MAP_SHARED or MAP_POPULATE, FFileHandle, 0);
  if FContent.Data = Pointer(-1) then
  begin
    FpClose(FFileHandle);
    raise EStreamError.Create('MMap failed');
  end;
  
  FContent.IsMapped := True;
  FContent.IsText := DetectTextFile(FContent.Data, FContent.Size);

  if LoadInRAM and not LoadToRAM then
    WriteLn('Warning: Could not lock file in RAM (size: ', FContent.Size, ' bytes)');
end;

destructor TUniversalFileReader.Destroy;
begin
  if FContent.IsMapped then
  begin
    if FContent.InRAM then
    begin
      SysMunlock(FContent.Data, FContent.Size);
      FreeMem(FContent.Data);
      FpMUnMap(Pointer(NativeUInt(FContent.Data) - FContent.Size), FContent.Size);
    end
    else
    begin
      FpMUnMap(FContent.Data, FContent.Size);
    end;
    FpClose(FFileHandle);
  end;
  inherited Destroy;
end;

function TUniversalFileReader.LoadToRAM: Boolean;
var
  NewData: Pointer;
begin
  Result := False;
  
  if FContent.Size > MLOCK_LIMIT then
    Exit;
    
  try
    GetMem(NewData, FContent.Size);
    Move(FContent.Data^, NewData^, FContent.Size);
    
    if SysMlock(NewData, FContent.Size) then
    begin
      FpMUnMap(FContent.Data, FContent.Size);
      FContent.Data := NewData;
      FContent.IsMapped := False;
      FContent.InRAM := True;
      Result := True;
    end
    else
    begin
      FreeMem(NewData);
      WriteLn('mlock failed: ', fpGetErrno);
    end;
  except
    on E: Exception do
      WriteLn('RAM load error: ', E.Message);
  end;
end;

function TUniversalFileReader.ReadContent: TFileContent;
begin
  Result := FContent;
end;

function TUniversalFileReader.ReadAsText: String;
begin
  if not FContent.IsText then
    raise EStreamError.Create('File is not text');
    
  SetString(Result, PChar(FContent.Data), FContent.Size div SizeOf(Char));
end;

function TUniversalFileReader.ReadAsBytes: TBytes;
begin
  SetLength(Result, FContent.Size);
  if FContent.Size > 0 then
    Move(FContent.Data^, Result[0], FContent.Size);
end;

function TUniversalFileReader.IsTextFile: Boolean;
begin
  Result := FContent.IsText;
end;

function TUniversalFileReader.IsInRAM: Boolean;
begin
  Result := FContent.InRAM;
end;

class function TUniversalFileReader.QuickRead(const FileName: String; LoadInRAM: Boolean): TFileContent;
var
  Reader: TUniversalFileReader;
begin
  Reader := TUniversalFileReader.Create(FileName, LoadInRAM);
  try
    Result := Reader.ReadContent;
  finally
    Reader.Free;
  end;
end;

class function TUniversalFileReader.DetectTextFile(Data: Pointer; Size: Int64): Boolean;
const
  TEXT_THRESHOLD = 0.95;
var
  i: SizeInt;
  printable, control: Integer;
  p: PByte;
begin
  if (Data = nil) or (Size = 0) then
    Exit(False);
    
  printable := 0;
  control := 0;
  p := PByte(Data);
  
  for i := 0 to Min(Size - 1, 4095) do
  begin
    case p[i] of
      $09, $0A, $0D: Inc(printable);
      $20..$7E: Inc(printable);
      $00..$08, $0B..$0C, $0E..$1F: Inc(control);
    end;
  end;
  
  Result := (printable / (printable + control)) > TEXT_THRESHOLD;
end;

end.