unit Word2Vec;
{$MODE OBJFPC}{$H+}
{$RANGECHECKS ON}
{$ASMMODE INTEL}
{$CODEPAGE UTF8}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


interface

uses
  SysUtils, Classes, MatrixOps, Math, DataUtils, LazUTF8, Contnrs,ucs4unit, ucs4opunit{, UniversalFileReader}, MMTextFileReader;

const
  EMBEDDING_SIZE = 300;
  MERGE_THRESHOLD = 64;
  CACHE_SIZE = 10000;
  UNKNOWN_CACHE_SIZE = 5000;
  HASH_TABLE_SIZE = 1048576; // 1M buckets для больших словарей

type
  TScore = record
    Word: string;
    Score: Double;
  end;
  TScoreArray = array of TScore;

  // Запись для хеш-таблицы
  THashEntry = record
    WordHash: Cardinal;
    WordIndex: Integer;
    Next: Integer;
  end;

  // Быстрый кэш с временными метками
  TFastCacheEntry = record
    Word: string;
    WordHash: Cardinal;
    Embedding: TDoubleArray;
    Timestamp: Int64;
  end;

  TWordEmbeddings = class
  private
    // Хеш-функция
    function ComputeHash(const Word: string): Cardinal;
    function FindInHashTable(const Word: string; WordHash: Cardinal): Integer;
    procedure SimpleAddToHashTable(const Word: string; Index: Integer);
    procedure InitializeHashTable;

    // Кэши
    procedure CacheInsert(const Word: string; const Emb: TDoubleArray);
    function CacheFind(const Word: string; out Emb: TDoubleArray): Boolean;
    procedure AddToUnknownCache(const Word: string);

    // Сортировка и поиск
    procedure QuickSort(var A: TScoreArray; L, R: Integer);
    procedure PartialSort(var A: TScoreArray; TopN: Integer);
    function SelectPivot(var A: TScoreArray; L, R: Integer): Double;

  public
    FVocab: TStringList;
    FEmbeddings: TDoubleMatrix;
    FEmbeddingNorms: TDoubleArray;
    FEmbeddingSize: Integer;

    // Компактная хеш-таблица
    FHashTable: array of Integer;
    FEntries: array of THashEntry;
    FEntryCount: Integer;
    FUseHashTable: Boolean;

    // Быстрый LRU кэш
    FCache: array of TFastCacheEntry;
    FCacheSize: Integer;
    FCacheHits: Integer;
    FCacheMisses: Integer;

    // Кэш ненайденных слов (отсортированный массив для бинарного поиска)
    FUnknownWords: array of string;
    FUnknownWordsCount: Integer;
    FUnknownWordsHits: Integer;
    FUnknownWordsMisses: Integer;

    // Статистика
    FHashCollisions: Integer;

    function IsInUnknownCache(const Word: string): Boolean;
    constructor Create(const ModelFile: string; CacheSize: Integer = CACHE_SIZE);
    destructor Destroy; override;

    // Основные методы
    function GetWordIndex(const Word: string): Integer;
    function GetEmbedding(const Word: string): TDoubleArray;
    function GetEmbeddingFastByIndex(Index: Integer): TDoubleArray;
    function GetEmbeddingWithCache(const Word: string): TDoubleArray;

    // Похожесть и поиск
    function Similarity(const Word1, Word2: string): Double;
    function FastSimilarity(const Word1, Word2: string): Double;
    function MostSimilar(const Word: string; TopN: Integer = 10): TStringArray;
    function FastSimilarityScore(const Emb1, Emb2: TDoubleArray): Double;

    // Управление кэшем
    procedure ClearCache;
    function GetCacheStats: string;
    procedure PrintHashTableStats;

    // Предзагрузка
    procedure PreloadCommonWords(const Words: array of string);

    // Статистика
    property CacheHits: Integer read FCacheHits;
    property CacheMisses: Integer read FCacheMisses;
    property EmbeddingSize: Integer read FEmbeddingSize;
    function GetVocabCount:Integer;
    property VocabularySize: Integer read GetVocabCount;
    procedure TestHashTable;
    procedure DebugWord(const Word: string);
    procedure CheckDuplicates;
  end;

var
  WordEmbeddings: TWordEmbeddings = nil;
  VerboseEmbeddingLogs: Boolean = True;

implementation

{$I asmf.inc}

function TWordEmbeddings.GetVocabCount:Integer;
begin
Exit(FVocab.Count);
end;

function TWordEmbeddings.ComputeHash(const Word: string): Cardinal;
begin
  // ✅ САМЫЙ НАДЕЖНЫЙ: используем встроенный хеш от строки
  try
    Result := HashName(PChar(Word)); // Функция из FPC
    Result := Result and $7FFFFFFF; // Гарантируем положительный
  except
    Result := Length(Word); // Fallback
  end;
end;

function TWordEmbeddings.FindInHashTable(const Word: string; WordHash: Cardinal): Integer;
var
  BucketIndex, CurrentIndex: Integer;
  Entry: THashEntry;
begin
  Result := -1;

  if not FUseHashTable or (Length(FHashTable) = 0) then
    Exit;

  BucketIndex := WordHash mod Cardinal(Length(FHashTable));

  if (BucketIndex < 0) or (BucketIndex >= Length(FHashTable)) then Exit;

  CurrentIndex := FHashTable[BucketIndex];

  while CurrentIndex >= 0 do begin
    if (CurrentIndex < 0) or (CurrentIndex >= Length(FEntries)) then Break;

    Entry := FEntries[CurrentIndex];

    if (Entry.WordIndex >= 0) and (Entry.WordIndex < FVocab.Count) then begin
      if (Entry.WordHash = WordHash) and (UTF8CompareStr(FVocab[Entry.WordIndex], Word) = 0) then begin
        // ✅ ВОЗВРАЩАЕМ ПЕРВОЕ СОВПАДЕНИЕ (самый маленький индекс)
        if (Result = -1) or (Entry.WordIndex < Result) then
          Result := Entry.WordIndex;
        // Не выходим сразу, ищем самое первое вхождение
      end;
    end;

    CurrentIndex := Entry.Next;
    Inc(FHashCollisions);
  end;
end;

procedure TWordEmbeddings.SimpleAddToHashTable(const Word: string; Index: Integer);
var
  WordHash: Cardinal;
  BucketIndex: Integer;
begin
  if (FEntryCount < 0) or (FEntryCount >= Length(FEntries)) then begin
    WriteLn('ОШИБКА: FEntryCount за границами: ', FEntryCount);
    Exit;
  end;

  try
    WordHash := ComputeHash(Word);
    BucketIndex := WordHash mod Cardinal(Length(FHashTable));

    if (BucketIndex < 0) or (BucketIndex >= Length(FHashTable)) then begin
      WriteLn('ОШИБКА: BucketIndex за границами: ', BucketIndex);
      Exit;
    end;

    // ✅ ЗАПОЛНЯЕМ ЗАПИСЬ ПРАВИЛЬНЫМ ИНДЕКСОМ
    // Index - это позиция в FVocab, которую мы и должны сохранить
    FEntries[FEntryCount].WordHash := WordHash;
    FEntries[FEntryCount].WordIndex := Index; // ← ЭТО ДОЛЖЕН БЫТЬ Index, а не FEntryCount!
    FEntries[FEntryCount].Next := FHashTable[BucketIndex];

    // Добавляем в цепочку
    FHashTable[BucketIndex] := FEntryCount;
    Inc(FEntryCount);

  except
    on E: Exception do begin
      WriteLn('Ошибка добавления в хеш-таблицу: ', E.Message);
    end;
  end;
end;

procedure TWordEmbeddings.InitializeHashTable;
var
  I: Integer;
begin
  WriteLn('Инициализация хеш-таблицы для ', FVocab.Count, ' слов...');

  FUseHashTable := False;

  if (FVocab = nil) or (FVocab.Count = 0) then
  begin
    WriteLn('Словарь пустой, хеш-таблица не создана');
    Exit;
  end;

  try
    // Для больших словарей используем упрощенную версию
    SetLength(FHashTable, HASH_TABLE_SIZE);
    for I := 0 to High(FHashTable) do
      FHashTable[I] := -1;

    SetLength(FEntries, FVocab.Count);
    FEntryCount := 0;
    FHashCollisions := 0;

    WriteLn('Заполнение хеш-таблицы...');
    for I := 0 to FVocab.Count - 1 do begin
      if (I > 0) and (I mod 50000 = 0) then
        WriteLn('  Обработано слов: ', I, '/', FVocab.Count);

      SimpleAddToHashTable(FVocab[I], I);
    end;

    FUseHashTable := True;
    WriteLn('Хеш-таблица создана. Записей: ', FEntryCount, ', коллизий: ', FHashCollisions);

  except
    on E: Exception do begin
      WriteLn('Ошибка хеш-таблицы: ', E.Message);
      SetLength(FHashTable, 0);
      SetLength(FEntries, 0);
      FUseHashTable := False;
    end;
  end;
end;

// Кэш эмбеддингов
procedure TWordEmbeddings.CacheInsert(const Word: string; const Emb: TDoubleArray);
var
  I, OldestIndex: Integer;
  OldestTime: Int64;
  CurrentTime: Int64;
begin
  CurrentTime := GetTickCount64;

  // Ищем пустой слот или заменяем самый старый
  OldestIndex := 0;
  OldestTime := High(Int64);

  for I := 0 to High(FCache) do begin
    if FCache[I].Word = '' then begin
      // Нашли пустой слот
      FCache[I].Word := Word;
      FCache[I].WordHash := ComputeHash(Word);
      FCache[I].Embedding := Copy(Emb);
      FCache[I].Timestamp := CurrentTime;
      Exit;
    end;

    if FCache[I].Timestamp < OldestTime then begin
      OldestTime := FCache[I].Timestamp;
      OldestIndex := I;
    end;
  end;

  // Все слоты заняты - заменяем самый старый
  FCache[OldestIndex].Word := Word;
  FCache[OldestIndex].WordHash := ComputeHash(Word);
  FCache[OldestIndex].Embedding := Copy(Emb);
  FCache[OldestIndex].Timestamp := CurrentTime;
end;

function TWordEmbeddings.CacheFind(const Word: string; out Emb: TDoubleArray): Boolean;
var
  I: Integer;
  WordHash: Cardinal;
  CurrentTime: Int64;
begin
  Result := False;
  WordHash := ComputeHash(Word);
  CurrentTime := GetTickCount64;

  for I := 0 to High(FCache) do begin
    if (FCache[I].WordHash = WordHash) and (FCache[I].Word = Word) then begin
      Emb := Copy(FCache[I].Embedding);
      FCache[I].Timestamp := CurrentTime; // Обновляем время доступа
      Result := True;
      Exit;
    end;
  end;
end;

// Кэш ненайденных слов
procedure TWordEmbeddings.AddToUnknownCache(const Word: string);
var
  L, R, M, I: Integer;
  CleanWord: string;
begin
  // ✅ ЗАЩИТА: Проверяем что массив инициализирован
  if FUnknownWords = nil then
    Exit;

  CleanWord := UTF8LowerCase(Word.Trim);
  if CleanWord = '' then Exit;

  try
    // Бинарный поиск для вставки в отсортированный массив
    L := 0;
    R := FUnknownWordsCount - 1;

    while L <= R do begin
      M := (L + R) shr 1;
      // ✅ ЗАЩИТА: Проверяем границы массива
      if (M < 0) or (M >= Length(FUnknownWords)) then Break;

      if FUnknownWords[M] = CleanWord then
        Exit // Уже в кэше
      else if FUnknownWords[M] < CleanWord then
        L := M + 1
      else
        R := M - 1;
    end;

    // ✅ ЗАЩИТА: Проверяем что не превышаем размер массива
    if FUnknownWordsCount >= Length(FUnknownWords) then begin
      // Удаляем самые старые (первые) если кэш полный
      for I := 1 to FUnknownWordsCount - 1 do begin
        if I - 1 < Length(FUnknownWords) then
          FUnknownWords[I - 1] := FUnknownWords[I];
      end;
      Dec(FUnknownWordsCount);
    end;

    // ✅ ЗАЩИТА: Проверяем что L в пределах массива
    if (L < 0) or (L > Length(FUnknownWords)) then
      L := FUnknownWordsCount;

    // Сдвигаем элементы
    for I := FUnknownWordsCount downto L + 1 do begin
      if I < Length(FUnknownWords) then
        FUnknownWords[I] := FUnknownWords[I - 1];
    end;

    // Вставляем новое слово
    if L < Length(FUnknownWords) then begin
      FUnknownWords[L] := CleanWord;
      Inc(FUnknownWordsCount);
    end;

  except
    on E: Exception do begin
      WriteLn('Ошибка добавления в unknown cache: ', E.Message);
      // Игнорируем ошибки добавления в кэш
    end;
  end;
end;

function TWordEmbeddings.IsInUnknownCache(const Word: string): Boolean;
var
  L, R, M: Integer;
  CleanWord: string;
begin
  // ✅ ЗАЩИТА: Проверяем что массив инициализирован и не пустой
  if (FUnknownWords = nil) or (FUnknownWordsCount = 0) then
    Exit(False);

  CleanWord := UTF8LowerCase(Word.Trim);
  if CleanWord = '' then Exit(False);

  try
    // Бинарный поиск в отсортированном массиве
    L := 0;
    R := FUnknownWordsCount - 1;

    while L <= R do begin
      M := (L + R) shr 1;
      // ✅ ЗАЩИТА: Проверяем границы массива
      if (M < 0) or (M >= Length(FUnknownWords)) then Break;

      if FUnknownWords[M] = CleanWord then
        Exit(True)
      else if FUnknownWords[M] < CleanWord then
        L := M + 1
      else
        R := M - 1;
    end;

  except
    on E: Exception do begin
      WriteLn('Ошибка поиска в unknown cache: ', E.Message);
      // В случае ошибки считаем что слова нет в кэше
    end;
  end;

  Result := False;
end;

{ // orig
constructor TWordEmbeddings.Create(const ModelFile: string; CacheSize: Integer = CACHE_SIZE);
var
  F: TextFile;
  Line: string;
  Parts: TStringArray;
  vocabSize, embedSize, I, J: Integer;
StartTime: QWord;
begin
  inherited Create;

  FVocab := TStringList.Create;
  FVocab.Sorted := False;
  FVocab.CaseSensitive := False;

  // ✅ ГАРАНТИРУЕМ ИНИЦИАЛИЗАЦИЮ МАССИВОВ
  FCacheSize := CacheSize;
  SetLength(FCache, FCacheSize);

  // Явно инициализируем кэш
  for I := 0 to High(FCache) do begin
    FCache[I].Word := '';
    FCache[I].WordHash := 0;
    FCache[I].Timestamp := 0;
  end;

  FCacheHits := 0;
  FCacheMisses := 0;

  // Явно инициализируем кэш неизвестных слов
  SetLength(FUnknownWords, UNKNOWN_CACHE_SIZE);
  for I := 0 to High(FUnknownWords) do
    FUnknownWords[I] := '';
  FUnknownWordsCount := 0;
  FUnknownWordsHits := 0;
  FUnknownWordsMisses := 0;

  // Загрузка модели
  if not FileExists(ModelFile) then begin
    WriteLn('Файл модели не найден: ', ModelFile);
    Halt;
  end;

StartTime := GetTickCount64;
  AssignFile(F, ModelFile);
  try
    Reset(F);

    // Читаем заголовок
    if not Eof(F) then
    begin
      ReadLn(F, Line);
      Parts := Line.Split([' '], TStringSplitOptions.ExcludeEmpty);

      if Length(Parts) < 2 then begin
        WriteLn('Неверный формат файла');
        CloseFile(F);
        Halt;
      end;

//      vocabSize := Min(StrToIntDef(Parts[0], 0), 2000000); // До 2M слов
vocabSize := 300000;
      embedSize := Min(Max(StrToIntDef(Parts[1], 300), 50), 600);

      WriteLn('Word2Vec: vocab=', vocabSize, ', embedding size=', embedSize);

      FVocab.Capacity := vocabSize;
      FEmbeddingSize := embedSize;
      SetLength(FEmbeddings, vocabSize, embedSize);

      // Читаем данные
      for I := 0 to vocabSize - 1 do begin
        if Eof(F) then Break;

        ReadLn(F, Line);
        Parts := Line.Split([' '], TStringSplitOptions.ExcludeEmpty);

        if Length(Parts) < embedSize + 1 then
          Continue;

        // Нормализуем сразу при загрузке
        FVocab.Add(UTF8LowerCase(Parts[0].Trim));

        for J := 0 to embedSize - 1 do begin
          if J + 1 < Length(Parts) then
            FEmbeddings[I][J] := StrToFloatDef(Parts[J + 1], 0.0)
          else
            FEmbeddings[I][J] := 0.0;
        end;

        if (I > 0) and (I mod 50000 = 0) then
          WriteLn('  Загружено ', I, '/', vocabSize, ' слов');
      end;

      WriteLn('Успешно загружено ', FVocab.Count, ' слов');
    end;

  except
    on E: Exception do begin
      WriteLn('Ошибка загрузки: ', E.Message);
      Halt;
    end;
  end;

  CloseFile(F);
WriteLn('Время загрузки: ', GetTickCount64 - StartTime, ' мс');
Halt;

  // ✅ ПРОВЕРЯЕМ ДУБЛИКАТЫ
//  CheckDuplicates;

  // Инициализируем системы
  InitializeHashTable;

  // ✅ ТЕСТИРУЕМ ХЕШ-ТАБЛИЦУ
  TestHashTable;

  // Предвычисляем нормы
  SetLength(FEmbeddingNorms, FVocab.Count);
  for I := 0 to FVocab.Count - 1 do begin
    FEmbeddingNorms[I] := Sqrt(SumOfSquares(FEmbeddings[I]));
  end;

  WriteLn('Word2Vec модель загружена');
  if FUseHashTable then
    WriteLn('  Хеш-таблица: активна')
  else
    WriteLn('  Хеш-таблица: отключена, используется простой поиск');
end;
}

 //mmap
constructor TWordEmbeddings.Create(const ModelFile: string; CacheSize: Integer = CACHE_SIZE);
var
  F: Int64;
  fd:TMMTextFile;
  Line: string;
  Parts: TStringArray;
  vocabSize, embedSize, I, J: Integer;
//StartTime: QWord;
begin
  inherited Create;

  FVocab := TStringList.Create;
  FVocab.Sorted := False;
  FVocab.CaseSensitive := False;

  // ✅ ГАРАНТИРУЕМ ИНИЦИАЛИЗАЦИЮ МАССИВОВ
  FCacheSize := CacheSize;
  SetLength(FCache, FCacheSize);

  // Явно инициализируем кэш
  for I := 0 to High(FCache) do begin
    FCache[I].Word := '';
    FCache[I].WordHash := 0;
    FCache[I].Timestamp := 0;
  end;

  FCacheHits := 0;
  FCacheMisses := 0;

  // Явно инициализируем кэш неизвестных слов
  SetLength(FUnknownWords, UNKNOWN_CACHE_SIZE);
  for I := 0 to High(FUnknownWords) do
    FUnknownWords[I] := '';
  FUnknownWordsCount := 0;
  FUnknownWordsHits := 0;
  FUnknownWordsMisses := 0;

  // Загрузка модели
  if not FileExists(ModelFile) then begin
    WriteLn('Файл модели не найден: ', ModelFile);
    Halt;
  end;

  try
//StartTime := GetTickCount64;
if MMTOpen(fd, ModelFile) then Halt;

    // Читаем заголовок
      MMTNextLine(fd, Line);

      Parts := Line.Split([' '], TStringSplitOptions.ExcludeEmpty);

      if Length(Parts) < 2 then begin
        WriteLn('Неверный формат файла');
        MMTClose(fd);
        Halt;
      end;

//      vocabSize := Min(StrToIntDef(Parts[0], 0), 2000001); // До 2M слов
vocabSize := 300000;
      embedSize := Min(Max(StrToIntDef(Parts[1], 300), 50), 600);

      WriteLn('Word2Vec: vocab=', vocabSize, ', embedding size=', embedSize);

      FVocab.Capacity := vocabSize;
      FEmbeddingSize := embedSize;
      SetLength(FEmbeddings, vocabSize, embedSize);

      // Читаем данные
      for I := 0 to vocabSize - 1 do begin

        MMTNextLine(fd, Line);
        Parts := Line.Split([' '], TStringSplitOptions.ExcludeEmpty);

        if Length(Parts) < embedSize + 1 then
          Continue;

        // Нормализуем сразу при загрузке
        FVocab.Add(UTF8LowerCase(Parts[0].Trim));

        for J := 0 to embedSize - 1 do begin
          if J + 1 < Length(Parts) then
            FEmbeddings[I][J] := StrToFloatDef(Parts[J + 1], 0.0)
          else
            FEmbeddings[I][J] := 0.0;
        end;

        if (I > 0) and (I mod 50000 = 0) then
          WriteLn('  Загружено ', I, '/', vocabSize, ' слов');
      end;

      WriteLn('Успешно загружено ', FVocab.Count, ' слов');

  except
    on E: Exception do begin
      WriteLn('Ошибка загрузки: ', E.Message);
      Halt;
    end;
  end;

MMTClose(fd);

//WriteLn('Время загрузки: ', GetTickCount64 - StartTime, ' мс');
//Halt;

  // ✅ ПРОВЕРЯЕМ ДУБЛИКАТЫ
//  CheckDuplicates;

  // Инициализируем системы
  InitializeHashTable;

  // ✅ ТЕСТИРУЕМ ХЕШ-ТАБЛИЦУ
  TestHashTable;

  // Предвычисляем нормы
  SetLength(FEmbeddingNorms, FVocab.Count);
  for I := 0 to FVocab.Count - 1 do begin
    FEmbeddingNorms[I] := Sqrt(SumOfSquares(FEmbeddings[I]));
  end;

  WriteLn('Word2Vec модель загружена');
  if FUseHashTable then
    WriteLn('  Хеш-таблица: активна')
  else
    WriteLn('  Хеш-таблица: отключена, используется простой поиск');
end;


{ // ufr
constructor TWordEmbeddings.Create(const ModelFile: string; CacheSize: Integer = CACHE_SIZE);
var
  Reader: TUniversalFileReader;
  FileContent: TFileContent;
  ContentStr: string;
  Lines: TStringArray;
  Parts: TStringArray;
  vocabSize, embedSize, I, J, LineIndex: Integer;
  StartPos, EndPos: Integer;
begin
  inherited Create;

  FVocab := TStringList.Create;
  FVocab.Sorted := False;
  FVocab.CaseSensitive := False;

  // ✅ ГАРАНТИРУЕМ ИНИЦИАЛИЗАЦИЮ МАССИВОВ
  FCacheSize := CacheSize;
  SetLength(FCache, FCacheSize);

  // Явно инициализируем кэш
  for I := 0 to High(FCache) do begin
    FCache[I].Word := '';
    FCache[I].WordHash := 0;
    FCache[I].Timestamp := 0;
  end;

  FCacheHits := 0;
  FCacheMisses := 0;

  // Явно инициализируем кэш неизвестных слов
  SetLength(FUnknownWords, UNKNOWN_CACHE_SIZE);
  for I := 0 to High(FUnknownWords) do
    FUnknownWords[I] := '';
  FUnknownWordsCount := 0;
  FUnknownWordsHits := 0;
  FUnknownWordsMisses := 0;

  // Загрузка модели с использованием mmap
  if not FileExists(ModelFile) then begin
    WriteLn('Файл модели не найден: ', ModelFile);
    Halt;
  end;

  WriteLn('Загрузка модели Word2Vec через mmap: ', ModelFile);
  
  try
    Reader := TUniversalFileReader.Create(ModelFile, True); // LoadInRAM = True для лучшей производительности
    try
      FileContent := Reader.ReadContent;
      
      if not FileContent.IsText then
      begin
        WriteLn('Файл модели не является текстовым');
        Halt;
      end;

      ContentStr := Reader.ReadAsText;
      
      WriteLn('Файл загружен в память, размер: ', Length(ContentStr), ' байт');
      WriteLn('В RAM: ', Reader.IsInRAM);

    finally
      Reader.Free;
    end;
  except
    on E: Exception do begin
      WriteLn('Ошибка загрузки файла через mmap: ', E.Message);
      Halt;
    end;
  end;

  // Разбиваем содержимое на строки
  WriteLn('Разбор содержимого...');
  Lines := ContentStr.Split([#10], TStringSplitOptions.ExcludeEmpty);
  WriteLn('Получено строк: ', Length(Lines));

  if Length(Lines) = 0 then begin
    WriteLn('Файл модели пуст');
    Halt;
  end;

  // Обрабатываем заголовок (первая строка)
  Parts := Lines[0].Split([' '], TStringSplitOptions.ExcludeEmpty);
  if Length(Parts) < 2 then begin
    WriteLn('Неверный формат файла');
    Halt;
  end;

  vocabSize := 300000; // Ограничиваем для тестирования
  embedSize := Min(Max(StrToIntDef(Parts[1], 300), 50), 600);

  WriteLn('Word2Vec: vocab=', vocabSize, ', embedding size=', embedSize);

  FVocab.Capacity := vocabSize;
  FEmbeddingSize := embedSize;
  SetLength(FEmbeddings, vocabSize, embedSize);

  // Обрабатываем остальные строки
  LineIndex := 1; // Начинаем со второй строки (после заголовка)
  I := 0;
  
  while (I < vocabSize) and (LineIndex < Length(Lines)) do
  begin
    if Lines[LineIndex] = '' then
    begin
      Inc(LineIndex);
      Continue;
    end;

    // Более эффективный парсинг строки
    Parts := Lines[LineIndex].Split([' '], TStringSplitOptions.ExcludeEmpty);
    
    if Length(Parts) >= embedSize + 1 then
    begin
      // Нормализуем сразу при загрузке
      FVocab.Add(UTF8LowerCase(Parts[0].Trim));

      // Быстрая загрузка чисел
      for J := 0 to embedSize - 1 do
      begin
        if J + 1 < Length(Parts) then
          FEmbeddings[I][J] := StrToFloatDef(Parts[J + 1], 0.0)
        else
          FEmbeddings[I][J] := 0.0;
      end;

      Inc(I);
    end;

    Inc(LineIndex);

    if (I > 0) and (I mod 50000 = 0) then
      WriteLn('  Загружено ', I, '/', vocabSize, ' слов');
  end;

  // Корректируем фактический размер
  if I < vocabSize then
  begin
    SetLength(FEmbeddings, I);
    WriteLn('Фактически загружено слов: ', I);
  end;

  WriteLn('Успешно загружено ', FVocab.Count, ' слов');

  // ✅ ПРОВЕРЯЕМ ДУБЛИКАТЫ
  // CheckDuplicates;

  // Инициализируем системы
  InitializeHashTable;

  // ✅ ТЕСТИРУЕМ ХЕШ-ТАБЛИЦУ
  TestHashTable;

  // Предвычисляем нормы
  SetLength(FEmbeddingNorms, FVocab.Count);
  for I := 0 to FVocab.Count - 1 do begin
    FEmbeddingNorms[I] := Sqrt(SumOfSquares(FEmbeddings[I]));
  end;

  WriteLn('Word2Vec модель загружена');
  if FUseHashTable then
    WriteLn('  Хеш-таблица: активна')
  else
    WriteLn('  Хеш-таблица: отключена, используется простой поиск');
end;
не эффективно}

destructor TWordEmbeddings.Destroy;
begin
SetLength(FEmbeddings, 0);
SetLength(FEmbeddingNorms, 0);
SetLength(FHashTable, 0);
SetLength(FEntries, 0);
SetLength(FCache, 0);
SetLength(FUnknownWords, 0);
FreeAndNil(FVocab);
inherited Destroy;
end;

function TWordEmbeddings.GetWordIndex(const Word: string): Integer;
var
  CleanWord: string;
  WordHash: Cardinal;
I:Integer;
begin
  Result := -1;

  try
    CleanWord := UTF8LowerCase(Word.Trim);

    if CleanWord = '' then
      Exit;

    // ✅ ПРОВЕРЯЕМ КЭШ НЕНАЙДЕННЫХ СЛОВ
    if IsInUnknownCache(CleanWord) then begin
      Inc(FUnknownWordsHits);
      Exit(-1);
    end;

    // ✅ ИСПОЛЬЗУЕМ ХЕШ-ТАБЛИЦУ ЕСЛИ ОНА АКТИВНА
    if FUseHashTable then begin
      WordHash := ComputeHash(CleanWord);
      Result := FindInHashTable(CleanWord, WordHash);
    end else begin
      // Fallback: простой поиск
      for I := 0 to FVocab.Count - 1 do begin
        if UTF8CompareStr(FVocab[I], CleanWord) = 0 then begin
          Result := I;
          Break;
        end;
      end;
    end;

    // Если не найдено - добавляем в кэш
    if Result = -1 then begin
      Inc(FUnknownWordsMisses);
      AddToUnknownCache(CleanWord);
    end;

  except
    on E: Exception do begin
      WriteLn('Ошибка в GetWordIndex для "', Word, '": ', E.Message);
      Result := -1;
    end;
  end;
end;

function TWordEmbeddings.GetEmbedding(const Word: string): TDoubleArray;
var
  Idx: Integer;
begin
  Idx := GetWordIndex(Word);
  if Idx >= 0 then
    Result := Copy(FEmbeddings[Idx])
  else
    SetLength(Result, 0);
end;

function TWordEmbeddings.GetEmbeddingFastByIndex(Index: Integer): TDoubleArray;
begin
  if (Index >= 0) and (Index < FVocab.Count) then
    Result := FEmbeddings[Index]
  else
    SetLength(Result, 0);
end;

function TWordEmbeddings.GetEmbeddingWithCache(const Word: string): TDoubleArray;
var
  CleanWord: string;
Idx:Integer;
begin
  CleanWord := UTF8LowerCase(Word.Trim);

  if CleanWord = '' then begin
    SetLength(Result, 0);
    Exit;
  end;

  // Пытаемся найти в кэше
  if CacheFind(CleanWord, Result) then begin
    Inc(FCacheHits);
    Exit;
  end;

  Inc(FCacheMisses);

  // Не в кэше - ищем и добавляем
  Idx := GetWordIndex(CleanWord);
  if Idx >= 0 then begin
    Result := Copy(FEmbeddings[Idx]);
    CacheInsert(CleanWord, Result);
  end else begin
    SetLength(Result, 0);
  end;
end;

procedure TWordEmbeddings.PartialSort(var A: TScoreArray; TopN: Integer);
var
  I, J, MaxIndex: Integer;
  Temp: TScore;
begin
  // Частичная сортировка - находим только TopN максимальных элементов
  if TopN >= Length(A) then
  begin
    QuickSort(A, 0, High(A));
    Exit;
  end;

  for I := 0 to TopN - 1 do
  begin
    MaxIndex := I;
    for J := I + 1 to High(A) do
    begin
      if A[J].Score > A[MaxIndex].Score then
        MaxIndex := J;
    end;

    if MaxIndex <> I then
    begin
      Temp := A[I];
      A[I] := A[MaxIndex];
      A[MaxIndex] := Temp;
    end;
  end;

  // Обрезаем массив до TopN
  SetLength(A, TopN);
end;

function TWordEmbeddings.MostSimilar(const Word: string; TopN: Integer): TStringArray;
var
  TargetEmb: TDoubleArray;
  I: Integer;
  Scores: TScoreArray;
begin
  TargetEmb := GetEmbeddingWithCache(Word);
  if Length(TargetEmb) = 0 then
    Exit(nil);

  SetLength(Scores, FVocab.Count);

  // Вычисляем scores для всех слов
  for I := 0 to FVocab.Count - 1 do begin
    Scores[I].Word := FVocab[I];
    Scores[I].Score := FastSimilarityScore(TargetEmb, FEmbeddings[I]);
  end;

  // ✅ ОПТИМИЗАЦИЯ: частичная сортировка для больших словарей
  if (FVocab.Count > 10000) and (TopN < 100) then
    PartialSort(Scores, TopN) // Быстрая частичная сортировка
  else
    QuickSort(Scores, 0, High(Scores)); // Полная сортировка для маленьких TopN

  // Возвращаем результат
  SetLength(Result, Length(Scores));
  for I := 0 to High(Scores) do
    Result[I] := Scores[I].Word;
end;

procedure TWordEmbeddings.PreloadCommonWords(const Words: array of string);
var
  I: Integer;
  Emb: TDoubleArray;
begin
  WriteLn('Предзагрузка ', Length(Words), ' частых слов...');

  for I := 0 to High(Words) do
  begin
    Emb := GetEmbedding(Words[I]);
    if Length(Emb) > 0 then
      CacheInsert(Words[I], Emb); // Принудительно добавляем в кэш
  end;

  WriteLn('Предзагрузка завершена');
end;

procedure TWordEmbeddings.QuickSort(var A: TScoreArray; L, R: Integer);
var
  I, J: Integer;
  Pivot: Double;
  Temp: TScore;
begin
  if R - L <= 0 then Exit;

  // Для маленьких массивов используем простую сортировку
  if R - L < 10 then
  begin
    for I := L to R - 1 do
    begin
      for J := I + 1 to R do
      begin
        if A[J].Score > A[I].Score then
        begin
          Temp := A[I];
          A[I] := A[J];
          A[J] := Temp;
        end;
      end;
    end;
    Exit;
  end;

  // Выбираем опорный элемент (медиана трех)
  Pivot := SelectPivot(A, L, R);

  I := L;
  J := R - 1;

  while True do
  begin
    repeat Inc(I) until A[I].Score <= Pivot;
    repeat Dec(J) until A[J].Score >= Pivot;

    if I >= J then Break;

    Temp := A[I];
    A[I] := A[J];
    A[J] := Temp;
  end;

  // Возвращаем медиану на место
  Temp := A[I];
  A[I] := A[R - 1];
  A[R - 1] := Temp;

  // Рекурсивно сортируем части
  QuickSort(A, L, I - 1);
  QuickSort(A, I + 1, R);
end;

procedure TWordEmbeddings.PrintHashTableStats;
var
  I, TotalChains, MaxChain, EmptyBuckets: Integer;
  TotalBuckets,ChainLength,CurrentIndex: Integer;
begin
  if not FUseHashTable then
  begin
    WriteLn('Хеш-таблица не активна');
    Exit;
  end;

  TotalBuckets := Length(FHashTable);
  TotalChains := 0;
  MaxChain := 0;
  EmptyBuckets := 0;

  for I := 0 to High(FHashTable) do
  begin
    if FHashTable[I] = -1 then
      Inc(EmptyBuckets)
    else
    begin
      ChainLength := 0;
      CurrentIndex := FHashTable[I];
      while CurrentIndex >= 0 do
      begin
        Inc(ChainLength);
        CurrentIndex := FEntries[CurrentIndex].Next;
      end;

      Inc(TotalChains, ChainLength);
      if ChainLength > MaxChain then
        MaxChain := ChainLength;
    end;
  end;

  WriteLn('=== ХЕШ-ТАБЛИЦА ===');
  WriteLn('Всего buckets: ', TotalBuckets);
  WriteLn('Пустых buckets: ', EmptyBuckets, ' (', (EmptyBuckets * 100) div TotalBuckets, '%)');
  WriteLn('Макс. длина цепочки: ', MaxChain);

  if (TotalBuckets - EmptyBuckets) > 0 then
    WriteLn('Ср. длина цепочки: ', TotalChains / (TotalBuckets - EmptyBuckets):0:2)
  else
    WriteLn('Ср. длина цепочки: 0.00');

  WriteLn('Всего коллизий: ', FHashCollisions);
  WriteLn('Записей: ', FEntryCount, '/', FVocab.Count);
end;

function TWordEmbeddings.Similarity(const Word1, Word2: string): Double;
var
  Emb1, Emb2: TDoubleArray;
  I: Integer;
  DotProduct, Norm1, Norm2: Double;
begin
  Emb1 := GetEmbeddingWithCache(Word1);
  Emb2 := GetEmbeddingWithCache(Word2);

  if (Length(Emb1) = 0) or (Length(Emb2) = 0) then
    Exit(0.0);

  DotProduct := 0.0;
  Norm1 := 0.0;
  Norm2 := 0.0;

  for I := 0 to High(Emb1) do begin
    DotProduct := DotProduct + Emb1[I] * Emb2[I];
    Norm1 := Norm1 + Sqr(Emb1[I]);
    Norm2 := Norm2 + Sqr(Emb2[I]);
  end;

  if (Norm1 = 0) or (Norm2 = 0) then
    Result := 0.0
  else
    Result := DotProduct / (Sqrt(Norm1) * Sqrt(Norm2));
end;

function TWordEmbeddings.FastSimilarity(const Word1, Word2: string): Double;
var
  Idx1, Idx2: Integer;
begin
  Idx1 := GetWordIndex(Word1);
  Idx2 := GetWordIndex(Word2);

  if (Idx1 < 0) or (Idx2 < 0) then
    Exit(0.0);

  Result := FastSimilarityScore(FEmbeddings[Idx1], FEmbeddings[Idx2]) / 
            (FEmbeddingNorms[Idx1] * FEmbeddingNorms[Idx2]);
end;

function TWordEmbeddings.FastSimilarityScore(const Emb1, Emb2: TDoubleArray): Double;
var
  I: Integer;
begin
  Result := 0.0;
  for I := 0 to FEmbeddingSize - 1 do
    Result := Result + Emb1[I] * Emb2[I];
end;

procedure TWordEmbeddings.ClearCache;
var
  I: Integer;
begin
  // Очищаем основной кэш
  for I := 0 to High(FCache) do begin
    FCache[I].Word := '';
    FCache[I].WordHash := 0;
    SetLength(FCache[I].Embedding, 0);
    FCache[I].Timestamp := 0;
  end;

  FCacheHits := 0;
  FCacheMisses := 0;

  // Очищаем кэш неизвестных слов
  for I := 0 to High(FUnknownWords) do
    FUnknownWords[I] := '';
  FUnknownWordsCount := 0;
  FUnknownWordsHits := 0;
  FUnknownWordsMisses := 0;

  WriteLn('Кэши очищены');
end;

function TWordEmbeddings.GetCacheStats: string;
var
  Total, UnknownTotal: Integer;
  CacheRatio, UnknownRatio: Double;
begin
  try
    Total := FCacheHits + FCacheMisses;
    UnknownTotal := FUnknownWordsHits + FUnknownWordsMisses;

    if Total > 0 then
      CacheRatio := (FCacheHits * 100.0) / Total
    else
      CacheRatio := 0;

    if UnknownTotal > 0 then
      UnknownRatio := (FUnknownWordsHits * 100.0) / UnknownTotal
    else
      UnknownRatio := 0;

    Result := Format('EmbeddingCache: Hits=%d, Misses=%d, Ratio=%.1f%% | ', 
      [FCacheHits, FCacheMisses, CacheRatio]);

    Result := Result + Format('UnknownCache: Hits=%d, Size=%d/%d', 
      [FUnknownWordsHits, FUnknownWordsCount, UNKNOWN_CACHE_SIZE]);

  except
    on E: Exception do
    begin
      Result := 'Cache stats error: ' + E.Message;
    end;
  end;
end;

function TWordEmbeddings.SelectPivot(var A: TScoreArray; L, R: Integer): Double;
var
  M: Integer;
Temp:TScore;
begin
  // Медиана трех для улучшения производительности QuickSort
  M := (L + R) shr 1;

  // Упорядочиваем A[L], A[M], A[R]
  if A[L].Score < A[M].Score then
  begin
    Temp := A[L];
    A[L] := A[M];
    A[M] := Temp;
  end;
  if A[L].Score < A[R].Score then
  begin
    Temp := A[L];
    A[L] := A[R];
    A[R] := Temp;
  end;
  if A[M].Score < A[R].Score then
  begin
    Temp := A[M];
    A[M] := A[R];
    A[R] := Temp;
  end;

  // Возвращаем медиану
  Result := A[M].Score;

  // Помещаем медиану в предпоследнюю позицию для удобства
  Temp := A[M];
  A[M] := A[R - 1];
  A[R - 1] := Temp;
end;

procedure TWordEmbeddings.TestHashTable;
var
  TestWords: array of string = ('привет', 'тест', 'слово', 'машина');
  I,J,K, HashIndex, SimpleIndex: Integer;
Word:string;
WordHash:Cardinal;
HashEmb,SimpleEmb:TDoubleArray;
Match:bytebool;
begin
  if not FUseHashTable then
  begin
    WriteLn('Хеш-таблица не активна, тест пропущен');
    Exit;
  end;

  WriteLn('=== ТЕСТ ХЕШ-ТАБЛИЦЫ ===');

  for I := 0 to High(TestWords) do
  begin
    Word := TestWords[I];
    WordHash := ComputeHash(Word);

    HashIndex := FindInHashTable(Word, WordHash);
    SimpleIndex := -1;

    // Находим правильный индекс через простой поиск
    for J := 0 to FVocab.Count - 1 do
    begin
      if UTF8CompareStr(FVocab[J], Word) = 0 then
      begin
        SimpleIndex := J;
        Break;
      end;
    end;

    if HashIndex = SimpleIndex then
    begin
      WriteLn('  "', Word, '": OK (индекс=', HashIndex, ')');

      // ✅ ДОПОЛНИТЕЛЬНАЯ ПРОВЕРКА: что эмбеддинги совпадают
      if HashIndex >= 0 then
      begin
        HashEmb := FEmbeddings[HashIndex];
        SimpleEmb := FEmbeddings[SimpleIndex];
        Match := True;

        for K := 0 to Min(High(HashEmb), High(SimpleEmb)) do
        begin
          if Abs(HashEmb[K] - SimpleEmb[K]) > 1e-10 then
          begin
            Match := False;
            Break;
          end;
        end;

        if Match then
          WriteLn('    Эмбеддинги: СОВПАДАЮТ')
        else
          WriteLn('    Эмбеддинги: НЕ СОВПАДАЮТ!');
      end
    end
    else
    begin
      WriteLn('  "', Word, '": ERROR');
      WriteLn('    Хеш-таблица вернула: ', HashIndex);
      WriteLn('    Простой поиск вернул: ', SimpleIndex);

      // Проверяем что возвращает хеш-таблица
      if HashIndex >= 0 then
      begin
        WriteLn('    Хеш-таблица нашла слово: "', FVocab[HashIndex], '"');
      end;
    end;
  end;

WriteLn('=== ДЕТАЛЬНАЯ ОТЛАДКА ===');
DebugWord('привет');
end;

procedure TWordEmbeddings.DebugWord(const Word: string);
var
  WordHash: Cardinal;
  BucketIndex, CurrentIndex,ChainPos: Integer;
  Entry: THashEntry;
begin
  WriteLn('=== ОТЛАДКА СЛОВА "', Word, '" ===');

  WordHash := ComputeHash(Word);
  BucketIndex := WordHash mod Cardinal(Length(FHashTable));

  WriteLn('Хеш: ', WordHash, ', Bucket: ', BucketIndex);

  CurrentIndex := FHashTable[BucketIndex];
  ChainPos := 0;

  while CurrentIndex >= 0 do
  begin
    if (CurrentIndex < 0) or (CurrentIndex >= Length(FEntries)) then
      Break;

    Entry := FEntries[CurrentIndex];

    WriteLn('  Цепочка ', ChainPos, ':');
    WriteLn('    Индекс записи: ', CurrentIndex);
    WriteLn('    Индекс слова: ', Entry.WordIndex);
    WriteLn('    Слово в словаре: "', FVocab[Entry.WordIndex], '"');
    WriteLn('    Хеш записи: ', Entry.WordHash);
    WriteLn('    Следующий: ', Entry.Next);

    if (Entry.WordHash = WordHash) and (UTF8CompareStr(FVocab[Entry.WordIndex], Word) = 0) then
    begin
      WriteLn('    >>> НАЙДЕНО СОВПАДЕНИЕ!');
    end;

    CurrentIndex := Entry.Next;
    Inc(ChainPos);
  end;
end;

procedure TWordEmbeddings.CheckDuplicates;
var
  I, J: Integer;
  DuplicateCount: Integer;
begin
  WriteLn('=== ПРОВЕРКА ДУБЛИКАТОВ В СЛОВАРЕ ===');

  DuplicateCount := 0;

  for I := 0 to FVocab.Count - 1 do begin
    for J := I + 1 to FVocab.Count - 1 do begin
      if UTF8CompareStr(FVocab[I], FVocab[J]) = 0 then begin
        WriteLn('Дубликат: "', FVocab[I], '" - индексы ', I, ' и ', J);
        Inc(DuplicateCount);
        // Не break, чтобы найти все дубликаты
      end;
    end;
  end;

  if DuplicateCount > 0 then
    WriteLn('Найдено дубликатов: ', DuplicateCount)
  else
    WriteLn('Дубликатов не найдено');
end;

initialization
  Randomize;
end.