unit ChatAI;
{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


interface

uses
  NLPCore, NLPUtils, NLPCache, Lemmatizer, UniversalFileReader, 
  SysUtils, Classes, RegExpr, fpexprpars, MathEvaluator, DateUtils, 
  PluginManagerUnit, PoetryPlugin, WeatherPlugin, PluginInterface,StrUtils;

type
  TChatAI = class
  private
    FTextProcessor: TTextProcessor;
    FNLPCache: TNLPCache;
    FLemmatizer: TLemmatizer;
    FPluginManager: TPluginManager; // Добавлено

    FLastWeatherRequest: TDateTime;
    FLastWeatherData: string;
    FWeatherCacheDuration: Integer;
    function IsWeatherCacheValid: Boolean;
  public
    constructor Create;
    destructor Destroy; override;
    function ProcessMessage(const AMessage: string): string;
    function GetWeatherWithCache(const City: string = ''): string;
  end;

function TryCalculateMath(const Question: string; out Answer: string): Boolean;

implementation

constructor TChatAI.Create;
begin
  FTextProcessor := TTextProcessor.Create('');
  FNLPCache := TNLPCache.Create('chat_cache.dat', 1000);
  FLemmatizer := TLemmatizer.Create('rules/');
  FPluginManager := TPluginManager.Create; // Добавлено
  if not FLemmatizer.LoadSuccess then
    WriteLn('Warning: Lemmatizer rules not loaded properly!');
  FWeatherCacheDuration := 30;
  FLastWeatherRequest := 0;
end;

destructor TChatAI.Destroy;
begin
  FreeAndNil(FPluginManager); // Добавлено
  FreeAndNil(FTextProcessor);
  FreeAndNil(FNLPCache);
  FreeAndNil(FLemmatizer);
  inherited;
end;

function TChatAI.IsWeatherCacheValid: Boolean;
begin
  Result := MinutesBetween(Now, FLastWeatherRequest) < FWeatherCacheDuration;
end;

function TChatAI.GetWeatherWithCache(const City: string): string;
begin
  if IsWeatherCacheValid and (FLastWeatherData <> '') then
  begin
    Result := FLastWeatherData + ' (from cache)';
    Exit;
  end;

  try
    FLastWeatherData := GetWeatherInfo(City);
    FLastWeatherRequest := Now;
    Result := FLastWeatherData;
  except
    on E: Exception do
      Result := 'Weather error: ' + E.Message;
  end;
end;

function TChatAI.ProcessMessage(const AMessage: string): string;
var
  LowerMsg: string;
  MathRegex: TRegExpr;
  MathExpr: string;
  MathResult: string;
  LTokens: TTokenArray;
  LLemmas: TStringList;
  LContext: TStringList;
  i: Integer;
begin
  LowerMsg := LowerCase(AMessage);
  
  // Проверка кэша перед обработкой
  if FNLPCache.Get(LowerMsg, Result) then
    Exit;

  // Пытаемся обработать как математическое выражение
  if TMathEvaluator.Evaluate(AMessage, MathResult) then
  begin
    Result := MathResult;
    FNLPCache.Add(LowerMsg, Result);
    Exit;
  end;

  // Основная обработка сообщений
  LTokens := FTextProcessor.Tokenize(AMessage);
  LLemmas := TStringList.Create;
  try
    // Лемматизация
    for i := 0 to High(LTokens) do
      LLemmas.Add(FLemmatizer.Lemmatize(LTokens[i].Text));

    // Анализ интентов
    if ContainsAny(LowerMsg, ['привет', 'здравствуй']) then
      Result := 'Привет! Как я могу помочь?'
    else if ContainsAny(LowerMsg, ['как дела', 'как жизнь']) then
      Result := 'У меня всё хорошо, спасибо!'
    else if ContainsAny(LowerMsg, ['кто ты', 'ты кто']) then
      Result := 'Я - ваш цифровой помощник ChatAI!'
    else
      Result := 'Я не совсем понял ваш вопрос. Можете уточнить?';

    // Кэширование
    LContext := TStringList.Create;
    try
      LContext.Add('User: ' + AMessage);
      LContext.Add('AI: ' + Result);
      FNLPCache.Add(LowerMsg, Result);
    finally
      LContext.Free;
    end;
  finally
    LLemmas.Free;
  end;
end;

function TryCalculateMath(const Question: string; out Answer: string): Boolean;
var
  Expr: string;
  Parser: TFPExpressionParser;
  ResultValue: TFPExpressionResult;
begin
  Result := False;
  Expr := StringReplace(Question, 'сколько будет', '', [rfIgnoreCase]);
  Expr := StringReplace(Expr, '?', '', [rfReplaceAll]);
  Expr := StringReplace(Expr, ' ', '', [rfReplaceAll]);
  Expr := Trim(Expr);

  if Expr = '' then Exit;

  try
    Parser := TFPExpressionParser.Create(nil);
    try
      Parser.BuiltIns := [bcMath];
      Parser.Expression := Expr;
      ResultValue := Parser.Evaluate;
      
      if ResultValue.ResultType = rtFloat then
        Answer := Format('%s = %.2f', [Expr, ResultValue.ResFloat])
      else if ResultValue.ResultType = rtInteger then
        Answer := Format('%s = %d', [Expr, ResultValue.ResInteger]);
        
      Result := True;
    finally
      Parser.Free;
    end;
  except
    on E: Exception do
      Result := False;
  end;
end;

end.