unit MFCCUnit;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2024-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}

interface

uses
  SysUtils, Math, fourier2;

type
  TDoubleArray = array of Double;
  TDoubleArrayArray = array of TDoubleArray;

function ComputeMFCC(Signal: TDoubleArray; SampleRate: Integer): TDoubleArray;

implementation

const
  NUM_MEL_FILTERS = 26;
  NUM_CEPSTRAL_COEFFS = 13;
  PREEMPHASIS_FACTOR = 0.97;

// Преэмфазный фильтр
procedure PreEmphasize(var Signal: TDoubleArray);
var
  i: Integer;
begin
  for i := High(Signal) downto 1 do
    Signal[i] := Signal[i] - PREEMPHASIS_FACTOR * Signal[i - 1];
end;

// Оконная функция Хэмминга
procedure ApplyHammingWindow(var Signal: TDoubleArray);
var
  i: Integer;
begin
  for i := 0 to High(Signal) do
    Signal[i] := Signal[i] * (0.54 - 0.46 * Cos(2 * Pi * i / (Length(Signal) - 1)));
end;

// Преобразование частоты в Mel-шкалу
function HzToMel(Freq: Double): Double;
begin
  Result := 2595 * Ln(1 + Freq / 700);
end;

// Преобразование Mel-шкалы в частоту
function MelToHz(Mel: Double): Double;
begin
  Result := 700 * (Exp(Mel / 2595) - 1);
end;

// Создание Mel-фильтров
function CreateMelFilters(FFTSize, SampleRate: Integer): TDoubleArrayArray;
var
  i, j, FreqIndex: Integer;
  MelPoints, HzPoints: TDoubleArray;
  FilterBank: array of TDoubleArray;
begin
  SetLength(MelPoints, NUM_MEL_FILTERS + 2);
  SetLength(HzPoints, NUM_MEL_FILTERS + 2);

  for i := 0 to NUM_MEL_FILTERS + 1 do
    MelPoints[i] := HzToMel(SampleRate / 2) * i / (NUM_MEL_FILTERS + 1);

  for i := 0 to NUM_MEL_FILTERS + 1 do
    HzPoints[i] := MelToHz(MelPoints[i]);

  SetLength(FilterBank, NUM_MEL_FILTERS, FFTSize div 2 + 1);

  for i := 1 to NUM_MEL_FILTERS do
  begin
    for j := 0 to FFTSize div 2 do
    begin
      FreqIndex := Round(j * SampleRate / FFTSize);
      if (FreqIndex >= HzPoints[i - 1]) and (FreqIndex <= HzPoints[i]) then
        FilterBank[i - 1][j] := (FreqIndex - HzPoints[i - 1]) / (HzPoints[i] - HzPoints[i - 1])
      else if (FreqIndex >= HzPoints[i]) and (FreqIndex <= HzPoints[i + 1]) then
        FilterBank[i - 1][j] := (HzPoints[i + 1] - FreqIndex) / (HzPoints[i + 1] - HzPoints[i])
      else
        FilterBank[i - 1][j] := 0;
    end;
  end;
  Result := FilterBank;
end;

// Функция DCT с предвычисленными коэффициентами
function ComputeDCT(Data: TDoubleArray): TDoubleArray;
var
  i, j: Integer;
  Sum: Double;
  CosTable: array of Double;
begin
  SetLength(Result, NUM_CEPSTRAL_COEFFS);
  SetLength(CosTable, Length(Data) * NUM_CEPSTRAL_COEFFS);

  // Предвычисление коэффициентов косинуса
  for i := 0 to NUM_CEPSTRAL_COEFFS - 1 do
    for j := 0 to High(Data) do
      CosTable[i * Length(Data) + j] := Cos(Pi * i * (2 * j + 1) / (2 * Length(Data)));

  for i := 0 to NUM_CEPSTRAL_COEFFS - 1 do
  begin
    Sum := 0;
    for j := 0 to High(Data) do
      Sum := Sum + Data[j] * CosTable[i * Length(Data) + j];
    Result[i] := Sum * Sqrt(2 / Length(Data));
  end;
end;

// Функция вычисления MFCC
function ComputeMFCC(Signal: TDoubleArray; SampleRate: Integer): TDoubleArray;
var
  FFTReal, FFTImag, MelSpectrum: TDoubleArray;
  MelFilters: TDoubleArrayArray;
  i, j: Integer;
begin
  if Length(Signal) = 0 then
    raise Exception.Create('Входной сигнал пуст');

  PreEmphasize(Signal);
  ApplyHammingWindow(Signal);

  // Инициализация реальной и мнимой частей
  SetLength(FFTReal, Length(Signal));
  SetLength(FFTImag, Length(Signal));
  for i := 0 to High(Signal) do
  begin
    FFTReal[i] := Signal[i];
    FFTImag[i] := 0;  // Мнимая часть инициализируется нулями
  end;

  ComputeFFT(FFTReal, FFTImag);

  MelFilters := CreateMelFilters(Length(FFTReal), SampleRate);

  // Проверка размеров MelFilters
  if Length(MelFilters) < NUM_MEL_FILTERS then
    raise Exception.Create('Неправильный размер MelFilters');

  for i := 0 to NUM_MEL_FILTERS - 1 do
  begin
    if Length(MelFilters[i]) < (Length(FFTReal) div 2 + 1) then
      raise Exception.CreateFmt('Неправильный размер MelFilters[%d]', [i]);
  end;

  SetLength(MelSpectrum, NUM_MEL_FILTERS);
  for i := 0 to NUM_MEL_FILTERS - 1 do
  begin
    MelSpectrum[i] := 0;
    for j := 0 to Min(High(FFTReal), High(MelFilters[i])) do
    begin
      MelSpectrum[i] := MelSpectrum[i] + FFTReal[j] * MelFilters[i][j];
    end;

    // Проверка на корректность значения перед вычислением логарифма
    if MelSpectrum[i] + 1e-10 <= 0 then
      MelSpectrum[i] := 1e-10  // Устанавливаем минимальное положительное значение
    else
      MelSpectrum[i] := Ln(MelSpectrum[i] + 1e-10);
  end;

  Result := ComputeDCT(MelSpectrum);
end;

end.