unit fourier2;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2024-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}

interface

uses
  SysUtils, Math;

type
  TDoubleArray = array of Double;

// Функция для вычисления FFT
procedure ComputeFFT(var RealData, ImagData: TDoubleArray);

implementation

// Рекурсивная реализация FFT (Cooley-Tukey)
procedure FFT(var RealData, ImagData: TDoubleArray; N: Integer);
var
  i, k: Integer;
  EvenReal, EvenImag, OddReal, OddImag: TDoubleArray;
  Angle, CosAngle, SinAngle: Double;
begin
  if N <= 1 then Exit;

  SetLength(EvenReal, N div 2);
  SetLength(EvenImag, N div 2);
  SetLength(OddReal, N div 2);
  SetLength(OddImag, N div 2);

  for i := 0 to (N div 2) - 1 do
  begin
    EvenReal[i] := RealData[2 * i];
    EvenImag[i] := ImagData[2 * i];
    OddReal[i] := RealData[2 * i + 1];
    OddImag[i] := ImagData[2 * i + 1];
  end;

  FFT(EvenReal, EvenImag, N div 2);
  FFT(OddReal, OddImag, N div 2);

  for k := 0 to (N div 2) - 1 do
  begin
    Angle := -2 * Pi * k / N;
    CosAngle := Cos(Angle);
    SinAngle := Sin(Angle);

    RealData[k] := EvenReal[k] + CosAngle * OddReal[k] - SinAngle * OddImag[k];
    ImagData[k] := EvenImag[k] + CosAngle * OddImag[k] + SinAngle * OddReal[k];
    RealData[k + N div 2] := EvenReal[k] - (CosAngle * OddReal[k] - SinAngle * OddImag[k]);
    ImagData[k + N div 2] := EvenImag[k] - (CosAngle * OddImag[k] + SinAngle * OddReal[k]);
  end;
end;

// Функция для вычисления FFT
procedure ComputeFFT(var RealData, ImagData: TDoubleArray);
var
  N: Integer;
begin
  N := Length(RealData);
  if N <> Length(ImagData) then
    raise Exception.Create('RealData and ImagData must have the same length');

  // Проверяем, что N является степенью двойки
  if (N and (N - 1)) <> 0 then
    raise Exception.Create('Length of input arrays must be a power of 2');

  FFT(RealData, ImagData, N);
end;

end.