unit rual;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2024-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, Math, AdaptiveDTW, MFCCUnit;

type
  // Определяем тип TFeatureArray (признаки сигнала)
  TFeatureArray = array of Double;

  // Шаблон для буквы
  TLetterTemplate = record
    Name: string; // Название буквы
    Features: array of TFeatureArray; // Признаки сигнала (MFCC)
  end;

  TRualProcessor = class
  private
    Templates: array of TLetterTemplate; // Шаблоны для букв
    function ExtractMFCCFeatures(Signal: array of Double): TFeatureArray; // Извлечение MFCC-признаков
  public
    constructor Create;
    function Recognize(Signal: array of Double): string; // Распознавание буквы
  end;

implementation

constructor TRualProcessor.Create;
begin
  // Инициализация шаблонов для всех букв русского алфавита
  SetLength(Templates, 33);

  // Шаблон для буквы "А"
  Templates[0].Name := 'А';
  SetLength(Templates[0].Features, 10);
//  Templates[0].Features[0] := [0.13, 107.67, 129.20, 161.50, 200.0, 250.0, 300.0, 350.0, 400.0, 450.0];
//  Templates[0].Features[1] := [0.13, 107.67, 129.20, 161.50, 200.0, 250.0, 300.0, 350.0, 400.0, 450.0];
Templates[0].Features[0] := [-1.3963, -0.7409, -0.6888, -1.4783, -0.3160, -0.9067, 0.6366, -0.4743, -0.5377, 0.3422, -0.0453, -0.1155, -0.2039];
Templates[0].Features[1] := [-1.3963, -0.7409, -0.6888, -1.4783, -0.3160, -0.9067, 0.6366, -0.4743, -0.5377, 0.3422, -0.0453, -0.1155, -0.2039];
  // ... (остальные признаки)

  // Шаблон для буквы "Б"
  Templates[1].Name := 'Б';
  SetLength(Templates[1].Features, 10);
  Templates[1].Features[0] := [-0.4727, -1.7585, 0.6427, -0.0049, 0.0668, 0.0687, 0.2294, 0.3223, 0.4055, -0.1143, 0.1688, 0.3869, -0.2037];
  Templates[1].Features[1] := [-0.4727, -1.7585, 0.6427, -0.0049, 0.0668, 0.0687, 0.2294, 0.3223, 0.4055, -0.1143, 0.1688, 0.3869, -0.2037];
  // ... (остальные признаки)

  // Шаблон для буквы "У"
  Templates[20].Name := 'У';
  SetLength(Templates[20].Features, 10);
Templates[20].Features[0] := [-2.9419, -0.8906, 2.0759, 0.9447, -0.1621, 0.1968, 0.4711, -0.4535, -0.4596, -0.4188, -0.0925, 0.5307, 0.2959];
Templates[20].Features[1] := [-2.9419, -0.8906, 2.0759, 0.9447, -0.1621, 0.1968, 0.4711, -0.4535, -0.4596, -0.4188, -0.0925, 0.5307, 0.2959];
  // ... (остальные признаки)

  // Шаблоны для остальных букв (аналогично)
  // ...
end;

// Извлечение MFCC-признаков из сигнала
function TRualProcessor.ExtractMFCCFeatures(Signal: array of Double): TFeatureArray;
var
  SignalArray: TDoubleArray;
  i: Integer;
begin
  if Length(Signal) = 0 then
    raise Exception.Create('Входной сигнал пуст');

  // Преобразуем открытый массив в динамический массив
  SetLength(SignalArray, Length(Signal));
  for i := 0 to High(Signal) do
    SignalArray[i] := Signal[i];

  // Используем MFCCUnit для извлечения признаков
  Result := ComputeMFCC(SignalArray, 44100); // SampleRate = 44100
end;

function TRualProcessor.Recognize(Signal: array of Double): string;
var
  i: Integer;
  InputFeatures: TFeatureArray;
  BestMatch: string;
  BestScore, Score: Double;
begin
  BestMatch := '?';
  BestScore := MaxDouble;

  // Извлекаем MFCC-признаки из входного сигнала
  InputFeatures := ExtractMFCCFeatures(Signal);

  // Вывод MFCC-признаков для отладки
  WriteLn('MFCC-признаки:');
  for i := 0 to High(InputFeatures) do
    WriteLn('MFCC[', i, '] = ', InputFeatures[i]:0:4);

  // Сравниваем с шаблонами
  for i := 0 to High(Templates) do
  begin
    // Проверяем, что шаблон и сигнал не пустые
    if (Length(InputFeatures) > 0) and (Length(Templates[i].Features) > 0) then
    begin
      // Используем DTW для сравнения MFCC-признаков
      Score := ComputeDTW([InputFeatures], Templates[i].Features, 20); // Увеличенный радиус
      WriteLn('Расстояние DTW для буквы ', Templates[i].Name, ': ', Score:0:4);

      if Score < BestScore then
      begin
        BestScore := Score;
        BestMatch := Templates[i].Name;
      end;
    end;
  end;

  // Если ни одна буква не распознана, возвращаем "?"
  if BestScore > 1.5 then // Увеличьте порог
    Result := '?'
  else
    Result := BestMatch;

  WriteLn('Распознана буква: ', Result);
end;

end.