program voicechat;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2024-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}
uses
  BaseUnix, Unix, SysUtils, ALSA, rual, AdaptiveDTW;

const
  SAMPLE_RATE = 44100;          // Частота дискретизации
  SILENCE_THRESHOLD = 0.4;     // Порог уровня звука
  BUFFER_SIZE = 4096;           // Размер буфера

var
  ShouldStop: Boolean = False;
  RualProcessor: TRualProcessor;
  LastWasSilence: Boolean = False; // Флаг для отслеживания тишины

procedure HandleSignal(Signal: cint); cdecl;
begin
  case Signal of
    SIGINT, SIGTERM:
      begin
        WriteLn('Получен сигнал завершения. Останавливаем программу...');
        ShouldStop := True;
      end;
  end;
end;

procedure ListenMicrophone;
var
  alsaHandle: Psnd_pcm_t;
  alsaParams: Psnd_pcm_hw_params_t;
  buffer: array of Int16;
  i: Integer;
  maxAmplitude: Double;
  rate: Cardinal;
  frames: snd_pcm_uframes_t;
  err: Integer;
  Signal: array of Double; // Изменён тип на array of Double
  RecognizedLetter: string;
begin
  if snd_pcm_open(@alsaHandle, 'default', SND_PCM_STREAM_CAPTURE, 0) < 0 then
  begin
    WriteLn('Ошибка открытия ALSA устройства для захвата');
    Exit;
  end;

  snd_pcm_hw_params_malloc(@alsaParams);
  snd_pcm_hw_params_any(alsaHandle, alsaParams);
  snd_pcm_hw_params_set_access(alsaHandle, alsaParams, SND_PCM_ACCESS_RW_INTERLEAVED);
  snd_pcm_hw_params_set_format(alsaHandle, alsaParams, SND_PCM_FORMAT_S16_LE);
  snd_pcm_hw_params_set_channels(alsaHandle, alsaParams, 1); // Моно

  rate := SAMPLE_RATE;
  err := snd_pcm_hw_params_set_rate_near(alsaHandle, alsaParams, @rate, nil);
  if err < 0 then
    WriteLn('Ошибка установки частоты дискретизации: ', snd_strerror(err));

  frames := BUFFER_SIZE;
  err := snd_pcm_hw_params_set_buffer_size_near(alsaHandle, alsaParams, @frames);
  if err < 0 then
    WriteLn('Ошибка установки размера буфера: ', snd_strerror(err));

  if snd_pcm_hw_params(alsaHandle, alsaParams) < 0 then
  begin
    WriteLn('Ошибка применения параметров ALSA');
    snd_pcm_hw_params_free(alsaParams);
    snd_pcm_close(alsaHandle);
    Exit;
  end;

  snd_pcm_hw_params_free(alsaParams);

  SetLength(buffer, BUFFER_SIZE);
  SetLength(Signal, BUFFER_SIZE);

  WriteLn('Слушаю микрофон...');
  while not ShouldStop do
  begin
    if snd_pcm_readi(alsaHandle, @buffer[0], BUFFER_SIZE) < 0 then
    begin
      WriteLn('Ошибка чтения данных из ALSA');
      Break;
    end;

    // Анализируем уровень звука
    maxAmplitude := 0;
    for i := 0 to BUFFER_SIZE - 1 do
    begin
      Signal[i] := buffer[i] / 32768.0; // Нормализация к диапазону [-1, 1]
      if Abs(Signal[i]) > maxAmplitude then
        maxAmplitude := Abs(Signal[i]);
    end;

    // Если уровень звука превышает порог
    if maxAmplitude > SILENCE_THRESHOLD then
    begin
      WriteLn('Слышу звук!');

      // Распознаём букву
      RecognizedLetter := RualProcessor.Recognize(Signal);
      WriteLn('Распознана буква: ', RecognizedLetter);

      // Сбрасываем флаг тишины
      LastWasSilence := False;
    end
    else
    begin
      // Если был звук, а теперь тишина, добавляем пробел
      if not LastWasSilence then
      begin
        WriteLn(' '); // Добавляем пробел между словами
        LastWasSilence := True;
      end;
    end;
  end;

  snd_pcm_close(alsaHandle);
end;

begin
  RualProcessor := TRualProcessor.Create;
  try
    fpSignal(SIGINT, @HandleSignal);
    fpSignal(SIGTERM, @HandleSignal);

    ListenMicrophone;
  finally
    RualProcessor.Free;
  end;
end.