program GBTest;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}
{$OPTIMIZATION LEVEL4}
uses
  SysUtils, Math, DateUtils, GradientBoosting, DataUtils, DecisionTreeGB;

const
  NUM_SAMPLES = 10000;
  NUM_FEATURES = 20;
  TEST_SIZE = 0.2;

procedure GenerateTestData(var x: TDoubleMatrix; var y: TDoubleArray);
var
  i, j: Integer;
begin
  SetLength(x, NUM_SAMPLES, NUM_FEATURES);
  SetLength(y, NUM_SAMPLES);
  
  RandSeed := 42;
  for i := 0 to High(x) do
  begin
    for j := 0 to High(x[i]) do
      x[i][j] := Random * 10;
    y[i] := 2 * x[i][0] + 3 * x[i][1] - 1.5 * x[i][2] + (Random - 0.5);
  end;
end;

procedure SafeSplitData(const x: TDoubleMatrix; const y: TDoubleArray;
                       out xTrain, xTest: TDoubleMatrix;
                       out yTrain, yTest: TDoubleArray);
var
  i, trainSize: Integer;
begin
  trainSize := Round(Length(x) * (1 - TEST_SIZE));
  
  SetLength(xTrain, trainSize, NUM_FEATURES);
  SetLength(yTrain, trainSize);
  SetLength(xTest, Length(x) - trainSize, NUM_FEATURES);
  SetLength(yTest, Length(y) - trainSize);

  // Безопасное копирование данных
  for i := 0 to trainSize - 1 do
  begin
    xTrain[i] := Copy(x[i], 0, NUM_FEATURES);
    yTrain[i] := y[i];
  end;
  
  for i := 0 to High(xTest) do
  begin
    xTest[i] := Copy(x[trainSize + i], 0, NUM_FEATURES);
    yTest[i] := y[trainSize + i];
  end;
end;

var
  x: TDoubleMatrix;
  y: TDoubleArray;
  xTrain, xTest: TDoubleMatrix;
  yTrain, yTest: TDoubleArray;
  model: TGradientBoosting;
  startTime: TDateTime;
  trainTime, predictTime: Double;
  i: Integer;
begin
  try
    // 1. Подготовка данных
    GenerateTestData(x, y);
    
    // 2. Безопасное разделение данных
    SafeSplitData(x, y, xTrain, xTest, yTrain, yTest);

    // 3. Обучение модели
    WriteLn('=== Gradient Boosting Benchmark ===');
    WriteLn('Samples: ', NUM_SAMPLES, ' Features: ', NUM_FEATURES);
    WriteLn('Train size: ', Length(xTrain), ' Test size: ', Length(xTest));
    
    startTime := Now;
    TrainGB(
      model,
      xTrain,
      yTrain,
      0.1,    // learningRate
      100,    // nEstimators
      3,      // maxDepth
      10,     // minSamplesSplit
      Round(Sqrt(NUM_FEATURES)) // featureSubsetSize
    );
    trainTime := MilliSecondsBetween(Now, startTime) / 1000;
    
    // 4. Предсказание
    startTime := Now;
    for i := 0 to High(xTest) do
      PredictGB(model, xTest[i]);
    predictTime := MilliSecondsBetween(Now, startTime) / 1000;
    
    // 5. Результаты
    WriteLn('Training time: ', trainTime:0:3, ' seconds');
    WriteLn('Prediction time: ', predictTime:0:4, ' seconds');
    WriteLn('Avg prediction: ', (predictTime*1000/Length(xTest)):0:4, ' ms/sample');
    
  except
    on E: Exception do
      WriteLn('Error: ', E.ClassName, ': ', E.Message);
  end;
end.