unit GnuPlotUtils;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}

{
    Gnu Plot visualizator.
    For GNU/Linux 64 bit version. Root priveleges or kernel patch needed.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2024-2025  Artyomov Alexander
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

interface

uses
  SysUtils, Classes, Process, DataTypes;

procedure SaveDataForPlot(const FileName: String; 
                         const Data: TDoubleMatrix;
                         const Labels: TIntegerArray);
procedure PlotKMeansClusters(const DataFile, CentroidFile: String; 
                           K: Integer);
procedure PlotGBPredictions(const DataFile, PredictionsFile: String);

implementation

procedure SaveDataForPlot(const FileName: String; 
                         const Data: TDoubleMatrix;
                         const Labels: TIntegerArray);
var
  f: TextFile;
  i: Integer;
begin
  AssignFile(f, FileName);
  Rewrite(f);
  try
    for i := 0 to High(Data) do
    begin
      if (Length(Labels) > 0) and (i <= High(Labels)) then
        WriteLn(f, Format('%g %g %d', [Data[i][0], Data[i][1], Labels[i]]))
      else
        WriteLn(f, Format('%g %g', [Data[i][0], Data[i][1]]));
    end;
  finally
    CloseFile(f);
  end;
end;

procedure PlotKMeansClusters(const DataFile, CentroidFile: String; 
                           K: Integer);
var
  gnuplot: TProcess;
  script: TStringList;
  scriptFile: String;
begin
  script := TStringList.Create;
  try
script.Add('set size square');
script.Add('set grid');
script.Add('set key outside');

    script.Add('set title "KMeans Clustering (K=' + IntToStr(K) + ')"');
    script.Add('set xlabel "Feature 1"');
    script.Add('set ylabel "Feature 2"');
    script.Add('set palette defined (0 "red", 1 "blue", 2 "green", 3 "purple")');
    script.Add('plot "' + DataFile + '" using 1:2:3 with points pt 7 ps 1 palette title "Data", \');
    script.Add('     "' + CentroidFile + '" with points pt 2 ps 3 lc black title "Centroids"');
    script.Add('pause -1 "Press Enter to continue"');
    
    scriptFile := ChangeFileExt(DataFile, '.plt');
    script.SaveToFile(scriptFile);
    
    gnuplot := TProcess.Create(nil);
    try
      gnuplot.Executable := 'gnuplot';
      gnuplot.Parameters.Add(scriptFile);
      gnuplot.Execute;
    finally
      gnuplot.Free;
    end;
  finally
    script.Free;
  end;
end;

procedure PlotGBPredictions(const DataFile, PredictionsFile: String);
var
  gnuplot: TProcess;
  script: TStringList;
  scriptFile: String;
begin
  script := TStringList.Create;
  try
    script.Add('set title "Gradient Boosting Predictions"');
    script.Add('set xlabel "Feature"');
    script.Add('set ylabel "Target"');
    script.Add('plot "' + DataFile + '" using 1:2 with points pt 7 title "Actual", \');
    script.Add('     "' + PredictionsFile + '" using 1:2 with lines lw 2 title "Predicted"');
    script.Add('pause -1 "Press Enter to continue"');
    
    scriptFile := ChangeFileExt(DataFile, '_gb.plt');
    script.SaveToFile(scriptFile);
    
    gnuplot := TProcess.Create(nil);
    try
      gnuplot.Executable := 'gnuplot';
      gnuplot.Parameters.Add(scriptFile);
      gnuplot.Execute;
    finally
      gnuplot.Free;
    end;
  finally
    script.Free;
  end;
end;

end.