unit GorgParser;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}

interface

uses
  SysUtils, Classes, StrUtils, Math, DateUtils, DataUtils, KMeans;

type
  TGorgAlarm = record
    hour, minute: Integer;
    weekMask: Integer;
    message: String;
    soundFile: String;
    isRealtime: Boolean;
    day, month, year: Integer;
  end;
  TGorgAlarmArray = array of TGorgAlarm;

function ParseGorgFile(const filename: String): TGorgAlarmArray;
function PrepareFeatures(const alarms: TGorgAlarmArray): TDoubleMatrix;
function AnalyzeCluster(const centroid: TDoubleArray): String;
procedure PrintClusterStats(const model: TKMeans);
function GetClusterName(cluster: Integer): String;

implementation

function GetClusterName(cluster: Integer): String;
begin
  case cluster of
    0: Result := 'Вечерние/общие';
    1: Result := 'Спортивные';
    2: Result := 'Особые события';
    else Result := 'Другие';
  end;
end;

procedure PrintClusterStats(const model: TKMeans);
var
  i, j: Integer;
begin
  for i := 0 to High(model.centroids) do
  begin
    WriteLn('Cluster ', i, ' (', GetClusterName(i), ')');
    WriteLn('Центроид: [');
    for j := 0 to High(model.centroids[i]) do
      Write(model.centroids[i][j]:0:2, ' ');
    WriteLn(']');
  end;
end;

function ContainsText(const text, substring: String): Boolean;
begin
  Result := Pos(LowerCase(substring), LowerCase(text)) > 0;
end;

function ParseGorgFile(const filename: String): TGorgAlarmArray;
var
  sl: TStringList;
  i, j: Integer;
  lineParts: TStringArray;
  alarm: TGorgAlarm;
begin
  sl := TStringList.Create;
  try
    sl.LoadFromFile(filename);
    SetLength(Result, sl.Count);

    for i := 0 to sl.Count - 1 do
    begin
      // Инициализация записи
      alarm.hour := 0;
      alarm.minute := 0;
      alarm.weekMask := 0;
      alarm.message := '';
      alarm.soundFile := '';
      alarm.isRealtime := False;
      alarm.day := 0;
      alarm.month := 0;
      alarm.year := 0;

      lineParts := sl[i].Split(#9);
      for j := 0 to High(lineParts) do
      begin
        if Pos('gorg1_hour:', lineParts[j]) = 1 then
          alarm.hour := StrToIntDef(Copy(lineParts[j], 12, MaxInt), 0)
        else if Pos('gorg1_minute:', lineParts[j]) = 1 then
          alarm.minute := StrToIntDef(Copy(lineParts[j], 14, MaxInt), 0)
        else if Pos('gorg1_week:', lineParts[j]) = 1 then
          alarm.weekMask := StrToIntDef(Copy(lineParts[j], 12, MaxInt), 0)
        else if Pos('gorg1_message:', lineParts[j]) = 1 then
          alarm.message := Copy(lineParts[j], 15, MaxInt)
        else if Pos('gorg2_music:', lineParts[j]) = 1 then
          alarm.soundFile := Copy(lineParts[j], 13, MaxInt)
        else if Pos('gorg2_sound:', lineParts[j]) = 1 then
          alarm.soundFile := Copy(lineParts[j], 13, MaxInt)
        else if Pos('gorg1_realtime:1', lineParts[j]) = 1 then
          alarm.isRealtime := True
        else if Pos('gorg1_day:', lineParts[j]) = 1 then
          alarm.day := StrToIntDef(Copy(lineParts[j], 11, MaxInt), 0)
        else if Pos('gorg1_month:', lineParts[j]) = 1 then
          alarm.month := StrToIntDef(Copy(lineParts[j], 13, MaxInt), 0)
        else if Pos('gorg1_year:', lineParts[j]) = 1 then
          alarm.year := StrToIntDef(Copy(lineParts[j], 12, MaxInt), 0);
      end;
      Result[i] := alarm;
    end;
  finally
    sl.Free;
  end;
end;

function PrepareFeatures(const alarms: TGorgAlarmArray): TDoubleMatrix;
var
  i: Integer;
begin
  SetLength(Result, Length(alarms));
  for i := 0 to High(alarms) do
  begin
    SetLength(Result[i], 10);
    
    // Временные признаки
    Result[i][0] := alarms[i].hour;
    Result[i][1] := alarms[i].minute / 60.0;
    if alarms[i].year > 0 then
      Result[i][2] := DayOfWeek(EncodeDate(alarms[i].year, alarms[i].month, alarms[i].day)) / 7.0
    else
      Result[i][2] := 0;

    // Признаки сообщения
    Result[i][3] := IfThen(ContainsText(alarms[i].message, 'Физкультура') or 
                          ContainsText(alarms[i].message, 'спорт'), 1, 0);
    Result[i][4] := IfThen(ContainsText(alarms[i].message, 'показания') or 
                          ContainsText(alarms[i].message, 'работа'), 1, 0);
    Result[i][5] := IfThen(ContainsText(alarms[i].message, 'Д.р.') or 
                          ContainsText(alarms[i].message, 'годовщина') or
                          ContainsText(alarms[i].message, 'день рождения'), 1, 0);

    // Мета-признаки
    Result[i][6] := IfThen(alarms[i].soundFile <> '', 1, 0);
    Result[i][7] := Length(alarms[i].message) / 50.0;
    Result[i][8] := IfThen(alarms[i].isRealtime, 1, 0);
    Result[i][9] := alarms[i].weekMask / 127.0;
  end;
end;

function AnalyzeCluster(const centroid: TDoubleArray): String;
begin
  Result := '';
  
  // Анализ времени
  if centroid[0] >= 18 then
    Result := 'Вечерние'
  else if centroid[0] <= 12 then
    Result := 'Утренние'
  else
    Result := 'Дневные';

  // Анализ типа события
  if centroid[3] > 0.5 then
    Result := Result + ' спортивные'
  else if centroid[5] > 0.5 then
    Result := Result + ' юбилейные'
  else if centroid[4] > 0.5 then
    Result := Result + ' рабочие';

  // Анализ звука
  if centroid[6] > 0.5 then
    Result := Result + ' (со звуком)';
end;

end.