unit GradientBoosting;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}
{$OPTIMIZATION LEVEL4}
{$OPTIMIZATION PEEPHOLE}
{$OPTIMIZATION REGVAR}
{$INLINE ON}
interface

uses
  SysUtils, Math, DataUtils, DecisionTreeGB;

type
  TGradientBoosting = record
    trees: array of TDecisionTree;
    learningRate: Double;
    nEstimators: Integer;
    featureSubsetSize: Integer;
  end;

procedure TrainGB(
  var model: TGradientBoosting;
  const x: TDoubleMatrix;
  const y: TDoubleArray;
  learningRate: Double = 0.1;
  nEstimators: Integer = 100;
  maxDepth: Integer = 3;
  minSamplesSplit: Integer = 10;
  featureSubsetSize: Integer = -1
);
function PredictGB(const model: TGradientBoosting; const x: TDoubleArray): Double; inline;

implementation

procedure TrainGB(
  var model: TGradientBoosting;
  const x: TDoubleMatrix;
  const y: TDoubleArray;
  learningRate: Double = 0.1;
  nEstimators: Integer = 100;
  maxDepth: Integer = 3;
  minSamplesSplit: Integer = 10;
  featureSubsetSize: Integer = -1
);
var
  i, j: Integer;
  residuals, predictions: TDoubleArray;
begin
  // Инициализация
  model.learningRate := learningRate;
  model.nEstimators := nEstimators;
  SetLength(model.trees, nEstimators);
  SetLength(predictions, Length(y));

  // Автовыбор размера подмножества признаков
  if featureSubsetSize = -1 then
    model.featureSubsetSize := Round(Sqrt(Length(x[0])))
  else
    model.featureSubsetSize := featureSubsetSize;

  // Основной цикл бустинга
  for i := 0 to nEstimators - 1 do
  begin
    // Вычисление остатков
    SetLength(residuals, Length(y));
    for j := 0 to High(y) do
      residuals[j] := y[j] - predictions[j];

    // Обучение дерева
    TrainDecisionTree(
      model.trees[i],
      x,
      residuals,
      maxDepth,
      minSamplesSplit
    );

    // Обновление предсказаний
    for j := 0 to High(x) do
      predictions[j] := predictions[j] + learningRate * PredictDecisionTree(model.trees[i], x[j]);
  end;
end;

function PredictGB(const model: TGradientBoosting; const x: TDoubleArray): Double; inline;
var
  i: Integer;
begin
  Result := 0.0;
  for i := 0 to High(model.trees) do
    Result := Result + model.learningRate * PredictDecisionTree(model.trees[i], x);
end;

end.