program MLBenchmark;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}//{$RANGECHECKS ON}
uses
  SysUtils, DateUtils,
  DataUtils, DecisionTree, RandomForest, KMeans, LinearAlgebra, BenchmarkUtils,Math;

const
  NUM_SAMPLES = 1000;
  NUM_FEATURES = 10;

procedure GenerateTestData(var x: TDoubleMatrix; var y: TDoubleArray; problemType: String);
var
  i, j: Integer;
begin
  SetLength(x, NUM_SAMPLES, NUM_FEATURES);
  SetLength(y, NUM_SAMPLES);
  
  Randomize;
  for i := 0 to High(x) do
  begin
    for j := 0 to High(x[i]) do
      x[i][j] := Random * 10;
      
    if problemType = 'regression' then
      y[i] := 2 * x[i][0] + 3 * x[i][1] - 1.5 * x[i][2] + Random
    else
      // Гарантируем, что классы будут положительными
      y[i] := IfThen(x[i][0] + x[i][1] > 10, 1, 0);
  end;
end;

procedure RunBenchmark;
var
  x: TDoubleMatrix;
  y: TDoubleArray;
  dt: TDecisionTree;
  startTime: TDateTime;
  elapsed: Double;
  i: Integer;
  prediction: Double;
begin
  try
    // Инициализация дерева
    FillChar(dt, SizeOf(dt), 0);
    // Генерация данных
    GenerateTestData(x, y, 'regression');
    
    WriteLn('=== Decision Tree Benchmark ===');
    WriteLn('Data generated: ', Length(x), ' samples with ', Length(x[0]), ' features');
    
    StartTimer;
    try
      TrainDecisionTree(dt, x, y, 5, 2);
      elapsed := StopTimer;
      WriteLn('Training time: ', elapsed:0:4, ' seconds');
      
      StartTimer;
      for i := 0 to High(x) do
      begin
        prediction := PredictDecisionTree(dt, x[i]);
        if IsNan(prediction) or IsInfinite(prediction) then
          WriteLn('Invalid prediction for sample ', i);
      end;
      elapsed := StopTimer;
      WriteLn('Prediction time (', Length(x), ' samples): ', elapsed:0:4, ' seconds');
    finally
      FreeDecisionTree(dt);
      // Проверка что память освобождена
      if dt.root <> nil then
        WriteLn('Warning: Tree root not nil after free!');
    end;
  except
    on E: Exception do
      WriteLn('Benchmark error: ', E.ClassName, ': ', E.Message);
  end;
end;

begin
  RunBenchmark;
end.