unit MachineLearning;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}
interface

uses
  DataUtils, LinearAlgebra,SysUtils,Math;

type
  TLogisticRegression = record
    weights: TDoubleArray;
  end;
  TLinearRegression = TLogisticRegression;

  TKNN = record
    k: Integer;
    x: TDoubleMatrix;
    y: TDoubleArray;
  end;

procedure TrainLinearRegression(var model: TLinearRegression; const x: TDoubleMatrix; const y: TDoubleArray; learningRate: Double; epochs: Integer; lambda: Double = 0.0);
function PredictLinearRegression(const model: TLinearRegression; const x: TDoubleArray): Double;
procedure TrainLogisticRegression(var model: TLogisticRegression; const x: TDoubleMatrix; const y: TDoubleArray; learningRate: Double; epochs: Integer);
function PredictLogisticRegression(const model: TLogisticRegression; const x: TDoubleArray): Double;
function Sigmoid(z: Double): Double;
function EuclideanDistance(const a, b: TDoubleArray): Double;
procedure TrainKNN(var model: TKNN; const x: TDoubleMatrix; const y: TDoubleArray; k: Integer);
function PredictKNN(const model: TKNN; const x: TDoubleArray): Double;

implementation

function EuclideanDistance(const a, b: TDoubleArray): Double;
var
  i: Integer;
begin
  Result := 0;
  for i := 0 to High(a) do
    Result := Result + Sqr(a[i] - b[i]);
  Result := Sqrt(Result);
end;

procedure TrainKNN(var model: TKNN; const x: TDoubleMatrix; const y: TDoubleArray; k: Integer);
begin
  if (Length(x) = 0) or (Length(y) = 0) then
    raise Exception.Create('Data is empty');

  if Length(x) <> Length(y) then
    raise Exception.Create('Input and target data sizes do not match');

  model.k := k;
  model.x := x;
  model.y := y;
end;

function PredictKNN(const model: TKNN; const x: TDoubleArray): Double;
var
  i, j: Integer;
  distances: TDoubleArray;
  nearestIndices: array of Integer;
begin
  SetLength(distances, Length(model.x));
  for i := 0 to High(model.x) do
    distances[i] := EuclideanDistance(x, model.x[i]);

  // Находим k ближайших соседей
  SetLength(nearestIndices, model.k);
  for i := 0 to model.k - 1 do
  begin
    nearestIndices[i] := 0;
    for j := 1 to High(distances) do
      if distances[j] < distances[nearestIndices[i]] then
        nearestIndices[i] := j;
    distances[nearestIndices[i]] := MaxDouble;  // Исключаем этот индекс из дальнейшего поиска
  end;

  // Вычисляем среднее значение целевых переменных для k ближайших соседей
  Result := 0;
  for i := 0 to High(nearestIndices) do
    Result := Result + model.y[nearestIndices[i]];
  Result := Result / model.k;
end;

procedure TrainLinearRegression(var model: TLinearRegression; const x: TDoubleMatrix; const y: TDoubleArray; learningRate: Double; epochs: Integer; lambda: Double = 0.0);
var
  i, j, k: Integer;
  error, gradient: Double;
begin
  if (Length(x) = 0) or (Length(y) = 0) then
    raise Exception.Create('Data is empty');

  if Length(x) <> Length(y) then
    raise Exception.Create('Input and target data sizes do not match');

  SetLength(model.weights, Length(x[0]) + 1);
  for i := 0 to High(model.weights) do
    model.weights[i] := 0.0;

  for i := 1 to epochs do
  begin
    for j := 0 to High(x) do
    begin
      if Length(x[j]) < 2 then
        raise Exception.Create('Invalid data format');

      error := PredictLinearRegression(model, x[j]) - y[j];

      // Обновляем bias (свободный член)
      model.weights[0] := model.weights[0] - learningRate * error;

      // Обновляем веса с учётом L2-регуляризации
      for k := 1 to High(model.weights) do
      begin
        gradient := error * x[j][k - 1] + lambda * model.weights[k];
        model.weights[k] := model.weights[k] - learningRate * gradient;
      end;
    end;
  end;
end;

function PredictLinearRegression(const model: TLinearRegression; const x: TDoubleArray): Double;
begin
  if Length(model.weights) = 0 then
    raise Exception.Create('Model is not trained');

  if Length(x) <> Length(model.weights) - 1 then
    raise Exception.Create('Input data size does not match model weights');

  Result := model.weights[0] + DotProduct(model.weights, x); // model.weights[0] — это bias
end;

function Sigmoid(z: Double): Double;
begin
  Result := 1.0 / (1.0 + Exp(-z));
end;

procedure TrainLogisticRegression(var model: TLogisticRegression; const x: TDoubleMatrix; const y: TDoubleArray; learningRate: Double; epochs: Integer);
var
  i, j, k: Integer;
  prediction, error: Double;
begin
  if (Length(x) = 0) or (Length(y) = 0) then
    raise Exception.Create('Data is empty');

  if Length(x) <> Length(y) then
    raise Exception.Create('Input and target data sizes do not match');

  SetLength(model.weights, Length(x[0]) + 1);
  for i := 0 to High(model.weights) do
    model.weights[i] := 0.0;

  for i := 1 to epochs do
  begin
    for j := 0 to High(x) do
    begin
      prediction := Sigmoid(PredictLinearRegression(model, x[j]));
      error := prediction - y[j];

      // Обновляем bias (свободный член)
      model.weights[0] := model.weights[0] - learningRate * error;

      // Обновляем веса
      for k := 1 to High(model.weights) do
        model.weights[k] := model.weights[k] - learningRate * error * x[j][k - 1];
    end;
  end;
end;

function PredictLogisticRegression(const model: TLogisticRegression; const x: TDoubleArray): Double;
begin
  Result := Sigmoid(PredictLinearRegression(model, x));
end;

end.