unit NLPCache;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}

interface

uses
  SysUtils, Classes, StrUtils;

type
  TNLPCache = class
  private
    FCache: TStringList;
    FMaxSize: Integer;
    FCacheFile: string;
    procedure LoadFromFile;
    procedure SaveToFile;
  public
    constructor Create(const CacheFile: string; MaxSize: Integer = 1000);
    destructor Destroy; override;
    procedure Add(const Key, Value: string);
    function Get(const Key: string; out Value: string): Boolean;
  end;

implementation

constructor TNLPCache.Create(const CacheFile: string; MaxSize: Integer);
begin
  FCache := TStringList.Create;
  FCache.Sorted := True; // Для быстрого поиска
  FCache.Duplicates := dupIgnore; // Игнорировать дубликаты
  FCache.CaseSensitive := False; // Регистронезависимый поиск
  FMaxSize := MaxSize;
  FCacheFile := CacheFile;
  LoadFromFile;
end;

destructor TNLPCache.Destroy;
begin
  SaveToFile;
  FCache.Free;
  inherited;
end;

procedure TNLPCache.LoadFromFile;
var
  List: TStringList;
  I: Integer;
  Parts: TStringArray;
begin
  if not FileExists(FCacheFile) then Exit;
  
  List := TStringList.Create;
  try
    List.LoadFromFile(FCacheFile);
    for I := 0 to List.Count - 1 do
    begin
      Parts := SplitString(List[I], '|');
      if Length(Parts) = 2 then
        FCache.Values[Parts[0]] := Parts[1];
    end;
  finally
    List.Free;
  end;
end;

procedure TNLPCache.SaveToFile;
var
  List: TStringList;
  I: Integer;
begin
  List := TStringList.Create;
  try
    for I := 0 to FCache.Count - 1 do
      List.Add(FCache.Names[I] + '|' + FCache.ValueFromIndex[I]);
    List.SaveToFile(FCacheFile);
  finally
    List.Free;
  end;
end;

procedure TNLPCache.Add(const Key, Value: string);
begin
  if FCache.Count >= FMaxSize then
    FCache.Clear;
  FCache.Values[Key] := Value;
end;

function TNLPCache.Get(const Key: string; out Value: string): Boolean;
var
  Index: Integer;
begin
  Index := FCache.IndexOfName(Key);
  Result := Index >= 0;
  if Result then
    Value := FCache.ValueFromIndex[Index]
  else
    Value := '';
end;

end.