unit NeuralNetwork;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}

interface

uses
  SysUtils, Math, DataUtils;

type
  TLayer = record
    weights: TDoubleMatrix; // Веса слоя
    biases: TDoubleArray;   // Смещения (bias)
    output: TDoubleArray;   // Выходные значения нейронов
  end;

  TNeuralNetwork = record
    layers: array of TLayer; // Слои сети
    learningRate: Double;    // Скорость обучения
    lambda: Double;          // Параметр L2-регуляризации
  end;

procedure InitializeNetwork(var network: TNeuralNetwork; const layerSizes: array of Integer; learningRate: Double; lambda: Double = 0.0);
procedure ForwardPropagation(var network: TNeuralNetwork; const input: TDoubleArray);
procedure BackwardPropagation(var network: TNeuralNetwork; const input: TDoubleArray; target: Double); // Изменён тип target на Double
procedure TrainNetwork(var network: TNeuralNetwork; const x: TDoubleMatrix; const y: TDoubleArray; epochs: Integer);
function PredictNetwork(const network: TNeuralNetwork; const input: TDoubleArray): TDoubleArray;
procedure FreeNetwork(var network: TNeuralNetwork);

implementation

function ReLU(x: Double): Double;
begin
  if x > 0 then
    Result := x
  else
    Result := 0;
end;

function LeakyReLU(x: Double): Double;
begin
  if x > 0 then
    Result := x
  else
    Result := 0.01 * x; // Малый наклон для отрицательных значений
end;

// Функция для генерации массива случайных чисел
function RandomArray(size: Integer; minVal, maxVal: Double): TDoubleArray;
var
  i: Integer;
begin
  SetLength(Result, size); // Явная инициализация массива
  for i := 0 to High(Result) do
    Result[i] := minVal + (maxVal - minVal) * Random;
end;

// Сигмоида
function Sigmoid(x: Double): Double;
begin
  Result := 1.0 / (1.0 + Exp(-x));
end;

// Гиперболический тангенс
function Tanh(x: Double): Double;
begin
  Result := (Exp(2 * x) - 1) / (Exp(2 * x) + 1);
end;

// Softmax
function Softmax(const x: TDoubleArray): TDoubleArray;
var
  i: Integer;
  maxVal, sum: Double;
begin
  SetLength(Result, Length(x)); // Явная инициализация массива
  maxVal := MaxValue(x);
  sum := 0.0;

  for i := 0 to High(x) do
  begin
    Result[i] := Exp(x[i] - maxVal); // Для численной стабильности
    sum := sum + Result[i];
  end;

  for i := 0 to High(Result) do
    Result[i] := Result[i] / sum;
end;

// Инициализация сети
procedure InitializeNetwork(var network: TNeuralNetwork; const layerSizes: array of Integer; learningRate: Double; lambda: Double = 0.0);
var
  i, j: Integer;
begin
  network.learningRate := learningRate;
  network.lambda := lambda;
  SetLength(network.layers, Length(layerSizes) - 1);

  for i := 0 to High(network.layers) do
  begin
    // Инициализация весов и смещений
    SetLength(network.layers[i].weights, layerSizes[i + 1], layerSizes[i]);
    SetLength(network.layers[i].biases, layerSizes[i + 1]);
    SetLength(network.layers[i].output, layerSizes[i + 1]);

    // Инициализация весов случайными значениями
    for j := 0 to High(network.layers[i].weights) do
    begin
      network.layers[i].weights[j] := RandomArray(layerSizes[i], -1.0, 1.0);
      network.layers[i].biases[j] := -1.0 + 2.0 * Random; // Случайное число от -1 до 1
    end;
  end;
end;

// Прямое распространение (Forward Propagation)
procedure ForwardPropagation(var network: TNeuralNetwork; const input: TDoubleArray);
var
  i, j, k: Integer;
  sum: Double;
begin
  if Length(input) <> Length(network.layers[0].weights[0]) then
    raise Exception.Create('Input size does not match network input layer size');

  // Входной слой
  for i := 0 to High(network.layers[0].weights) do
  begin
    sum := 0.0;
    for j := 0 to High(input) do
      sum := sum + network.layers[0].weights[i][j] * input[j];
    sum := sum + network.layers[0].biases[i];
//    network.layers[0].output[i] := ReLU(sum); // Используем ReLU для активации
network.layers[0].output[i] := LeakyReLU(sum);
  end;

  // Скрытые слои
  for i := 1 to High(network.layers) do
  begin
    for j := 0 to High(network.layers[i].weights) do
    begin
      sum := 0.0;
      for k := 0 to High(network.layers[i - 1].output) do
        sum := sum + network.layers[i].weights[j][k] * network.layers[i - 1].output[k];
      sum := sum + network.layers[i].biases[j];
//      network.layers[i].output[j] := LeakyReLU(sum); // Используем LeakyReLU для активации
network.layers[i].output[j] := LeakyReLU(sum);
    end;
  end;

  // Выходной слой (Softmax для классификации)
  network.layers[High(network.layers)].output := Softmax(network.layers[High(network.layers)].output);
end;

// Обратное распространение (Backward Propagation)
procedure BackwardPropagation(var network: TNeuralNetwork; const input: TDoubleArray; target: Double);
var
  i, j, k: Integer;
  error, derivative: Double;
  deltas: array of TDoubleArray;
begin
  // Инициализация массива deltas
  SetLength(deltas, Length(network.layers));
  for i := 0 to High(deltas) do
    SetLength(deltas[i], Length(network.layers[i].output));

  // Вычисляем ошибку на выходном слое
  for i := 0 to High(network.layers[High(network.layers)].output) do
  begin
    error := network.layers[High(network.layers)].output[i] - target;
    deltas[High(deltas)][i] := error;
  end;

  // Распространяем ошибку назад
  for i := High(network.layers) downto 1 do
  begin
    for j := 0 to High(network.layers[i].weights) do
    begin
      derivative := 1.0 - Sqr(network.layers[i].output[j]); // Производная Tanh
      deltas[i - 1][j] := 0.0;
      for k := 0 to High(network.layers[i - 1].output) do
      begin
        // Проверка, чтобы не выйти за пределы массива
        if (k <= High(deltas[i])) and (k <= High(network.layers[i].weights[j])) then
          deltas[i - 1][j] := deltas[i - 1][j] + network.layers[i].weights[j][k] * deltas[i][k] * derivative;
      end;
    end;
  end;

  // Обновляем веса и смещения с учётом L2-регуляризации
  for i := 0 to High(network.layers) do
  begin
    for j := 0 to High(network.layers[i].weights) do
    begin
      for k := 0 to High(network.layers[i].weights[j]) do
      begin
        // Проверка, чтобы не выйти за пределы массива
        if (k <= High(network.layers[i].output)) then
        begin
          network.layers[i].weights[j][k] := network.layers[i].weights[j][k] - network.learningRate * (deltas[i][j] * network.layers[i].output[k] + network.lambda * network.layers[i].weights[j][k]);
        end;
      end;
      network.layers[i].biases[j] := network.layers[i].biases[j] - network.learningRate * deltas[i][j];
    end;
  end;
end;

// Обучение сети
procedure TrainNetwork(var network: TNeuralNetwork; const x: TDoubleMatrix; const y: TDoubleArray; epochs: Integer);
var
  i, j: Integer;
begin
  for i := 1 to epochs do
  begin
    for j := 0 to High(x) do
    begin
      ForwardPropagation(network, x[j]);
      BackwardPropagation(network, x[j], y[j]); // Передаём y[j] как Double
    end;
  end;
end;

// Предсказание
function PredictNetwork(const network: TNeuralNetwork; const input: TDoubleArray): TDoubleArray;
var
  networkCopy: TNeuralNetwork;
begin
  // Создаём копию сети, чтобы не изменять оригинальную
  networkCopy := network;
  ForwardPropagation(networkCopy, input);
  Result := networkCopy.layers[High(networkCopy.layers)].output;
end;

// Освобождение памяти
procedure FreeNetwork(var network: TNeuralNetwork);
var
  i: Integer;
begin
  for i := 0 to High(network.layers) do
  begin
    SetLength(network.layers[i].weights, 0);
    SetLength(network.layers[i].biases, 0);
    SetLength(network.layers[i].output, 0);
  end;
  SetLength(network.layers, 0);
end;

end.