unit SVM;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}

interface

uses
  DataUtils, LinearAlgebra, SysUtils, Math, CommonUnit, ColorizerUnit;

type
  TSVM = record
    weights: TDoubleArray;
    bias: Double;
  end;

procedure TrainSVM(var model: TSVM; const x: TDoubleMatrix; const y: TDoubleArray; 
                  learningRate: Double; epochs: Integer; lambda: Double = 0.0);
function PredictSVM(const model: TSVM; const x: TDoubleArray): Double;

implementation

procedure TrainSVM(var model: TSVM; const x: TDoubleMatrix; const y: TDoubleArray; 
                  learningRate: Double; epochs: Integer; lambda: Double = 0.0);
var
  i, j, k: Integer;
  margin, gradient: Double;
begin
  if Length(x) = 0 then
  begin
    ShowError('Ошибка: Данные не загружены.');
    Log('Ошибка: Данные не загружены.', LOG_ERROR);
    Exit;
  end;

  try
    // Инициализация весов
    SetLength(model.weights, Length(x[0]));
    for i := 0 to High(model.weights) do
      model.weights[i] := 0;
    model.bias := 0;

    // Обучение модели
    for k := 1 to epochs do
    begin
      for i := 0 to High(x) do
      begin
        margin := y[i] * (DotProduct(model.weights, x[i]) + model.bias);
        if margin < 1 then
        begin
          for j := 0 to High(model.weights) do
          begin
            gradient := model.weights[j] - lambda * y[i] * x[i][j];
            model.weights[j] := model.weights[j] - learningRate * gradient;
          end;
          model.bias := model.bias + learningRate * y[i];
        end
        else
        begin
          for j := 0 to High(model.weights) do
            model.weights[j] := model.weights[j] * (1 - learningRate * lambda);
        end;
      end;
    end;

    ShowSuccess('Модель SVM успешно обучена.');
    Log('Модель SVM обучена.', LOG_INFO);
  except
    on E: Exception do
    begin
      ShowError('Ошибка: ' + E.Message);
      Log('Ошибка при обучении модели SVM: ' + E.Message, LOG_ERROR);
    end;
  end;
end;

function PredictSVM(const model: TSVM; const x: TDoubleArray): Double;
begin
  if Length(model.weights) = 0 then
  begin
    ShowError('Ошибка: Модель не обучена.');
    Log('Ошибка: Модель SVM не обучена.', LOG_ERROR);
    Exit(0.0);
  end;

  if Length(x) <> Length(model.weights) then
  begin
    ShowError('Ошибка: Размер входных данных не совпадает с размером модели.');
    Log('Ошибка: Несоответствие размеров данных в PredictSVM', LOG_ERROR);
    Exit(0.0);
  end;

  try
    Result := DotProduct(model.weights, x) + model.bias;
    if Result >= 0 then
      Result := 1
    else
      Result := -1;
  except
    on E: Exception do
    begin
      ShowError('Ошибка при предсказании: ' + E.Message);
      Log('Ошибка в PredictSVM: ' + E.Message, LOG_ERROR);
      Result := 0.0;
    end;
  end;
end;

end.