program TestUniversalFileReader;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}

uses
  SysUtils, UniversalFileReader,Math;

function FormatSize(Size: Int64): String;
begin
  if Size < 1024 then
    Result := IntToStr(Size) + ' B'
  else if Size < 1024*1024 then
    Result := IntToStr(Size div 1024) + ' KB'
  else
    Result := Format('%.2f MB', [Size/(1024*1024)]);
end;

procedure TestFile(const Filename: string; LoadInRAM: Boolean);
var
  reader: TUniversalFileReader;
  content: TFileContent;
  bytes: TBytes;
  i: Integer;
begin
  WriteLn('=== Testing file: "', Filename, '" ===');
  WriteLn('LoadInRAM: ', LoadInRAM);
  
  try
    reader := TUniversalFileReader.Create(Filename, LoadInRAM);
    try
      content := reader.ReadContent;
      
      WriteLn('File size: ', FormatSize(content.Size));
      WriteLn('Is text: ', reader.IsTextFile);
      WriteLn('Is in RAM: ', reader.IsInRAM);
      
      if reader.IsTextFile then
      begin
        WriteLn('--- Text preview (first 200 chars) ---');
        WriteLn(Copy(reader.ReadAsText, 1, 200));
      end
      else
      begin
        WriteLn('--- Binary signature ---');
        bytes := reader.ReadAsBytes;
        for i := 0 to Min(15, High(bytes)) do
          Write(IntToHex(bytes[i], 2), ' ');
        WriteLn;
      end;
    finally
      reader.Free;
    end;
  except
    on E: Exception do
      WriteLn('ERROR: ', E.Message);
  end;
  
  WriteLn;
end;

const VideoFile = 'test_video.avi';

begin
  WriteLn('===== UniversalFileReader Test =====');
  WriteLn('Platform: ', {$I %FPCTARGETOS%}, ' ', {$I %FPCTARGETCPU%});
  WriteLn;

  WriteLn('=== Text File Test ===');
  TestFile('TestUniversalFileReader.pas', False);
  TestFile('TestUniversalFileReader.pas', True);

  WriteLn('=== Binary File Test ===');
  if FileExists(VideoFile) then
  begin
    TestFile(VideoFile, False);
    TestFile(VideoFile, True);
  end
  else
    WriteLn('Warning: Test video file not found (', VideoFile, ')');

  WriteLn('=== Special Cases ===');
  TestFile('/proc/cpuinfo', False);
  TestFile('/dev/zero', False);
  TestFile('non_existent_file', False);

  WriteLn('=== Testing completed ===');
end.