unit SVM;
{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}

interface

uses
  DataUtils, LinearAlgebra, SysUtils, Math, CommonUnit, ColorizerUnit;

type
  TSVM = record
    weights: TDoubleArray;
    bias: Double;
  end;

procedure TrainSVM(var model: TSVM; const x: TDoubleMatrix; const y: TDoubleArray; 
                  learningRate: Double; epochs: Integer; lambda: Double = 0.0);
function PredictSVM(const model: TSVM; const x: TDoubleArray): Double;

implementation

{
function DotProduct(const a, b: TDoubleArray): Double;
var
  i: Integer;
begin
  if Length(a) <> Length(b) then
    raise Exception.Create('Arrays must have compatible lengths for dot product');
  Result := 0;
  for i := 0 to High(a) do
    Result := Result + a[i] * b[i];
end;
}


procedure TrainSVM(var model: TSVM; const x: TDoubleMatrix; const y: TDoubleArray; 
                  learningRate: Double; epochs: Integer; lambda: Double = 0.0);
var
  i, j, k: Integer;
  margin, gradient: Double;
begin
  if Length(x) = 0 then
  begin
    ShowError('Ошибка: Данные не загружены.');
    Log('Ошибка: Данные не загружены.', LOG_ERROR);
    Exit;
  end;

  try
    // Инициализация весов
    SetLength(model.weights, Length(x[0]));
    for i := 0 to High(model.weights) do
      model.weights[i] := Random * 0.01; // Небольшая случайная инициализация
    model.bias := 0;

    // Обучение модели
    for k := 1 to epochs do
    begin
      for i := 0 to High(x) do
      begin
        margin := y[i] * (DotProduct(model.weights, x[i]) + model.bias);
        if margin < 1 then
        begin
          for j := 0 to High(model.weights) do
          begin
            gradient := -y[i] * x[i][j] + lambda * model.weights[j];
            model.weights[j] := model.weights[j] - learningRate * gradient;
          end;
          model.bias := model.bias - learningRate * lambda * y[i];
        end
        else
        begin
          for j := 0 to High(model.weights) do
            model.weights[j] := model.weights[j] - learningRate * lambda * model.weights[j];
        end;
      end;
    end;

    ShowSuccess('Модель SVM успешно обучена.');
    Log('Модель SVM обучена.', LOG_INFO);
  except
    on E: Exception do
    begin
      ShowError('Ошибка: ' + E.Message);
      Log('Ошибка при обучении модели SVM: ' + E.Message, LOG_ERROR);
    end;
  end;
end;

function PredictSVM(const model: TSVM; const x: TDoubleArray): Double;
begin
  if Length(model.weights) = 0 then
  begin
    ShowError('Ошибка: Модель не обучена.');
    Log('Ошибка: Модель SVM не обучена.', LOG_ERROR);
    Exit(0.0);
  end;

  if Length(x) <> Length(model.weights) then
  begin
    ShowError('Ошибка: Размер входных данных не совпадает с размером модели.');
    Log('Ошибка: Несоответствие размеров данных в PredictSVM', LOG_ERROR);
    Exit(0.0);
  end;

  try
    Result := DotProduct(model.weights, x) + model.bias;
  except
    on E: Exception do
    begin
      ShowError('Ошибка при предсказании: ' + E.Message);
      Log('Ошибка в PredictSVM: ' + E.Message, LOG_ERROR);
      Result := 0.0;
    end;
  end;
end;

end.