unit RandomForest;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}

interface

uses
  SysUtils, DecisionTree, DataUtils,Math;

type
  TRandomForest = record
    trees: array of TDecisionTree;
    numTrees: Integer;
    maxDepth: Integer;
    minSamplesSplit: Integer;
  end;

procedure TrainRandomForest(var forest: TRandomForest; const x: TDoubleMatrix; const y: TDoubleArray; numTrees: Integer; maxDepth: Integer; minSamplesSplit: Integer);
function PredictRandomForest(const forest: TRandomForest; const x: TDoubleArray): Double;
procedure FreeRandomForest(var forest: TRandomForest);

implementation

// Функция для создания бутстрэп-выборки
procedure BootstrapSample(const x: TDoubleMatrix; const y: TDoubleArray; out bootstrapX: TDoubleMatrix; out bootstrapY: TDoubleArray);
var
  i, index: Integer;
begin
  SetLength(bootstrapX, Length(x));
  SetLength(bootstrapY, Length(y));

  for i := 0 to High(x) do
  begin
    // Случайный индекс с повторениями
    index := Random(Length(x));
    bootstrapX[i] := x[index];
    bootstrapY[i] := y[index];
  end;
end;

procedure TrainRandomForest(var forest: TRandomForest; const x: TDoubleMatrix; const y: TDoubleArray; numTrees: Integer; maxDepth: Integer; minSamplesSplit: Integer);
var
  i: Integer;
  bootstrapX: TDoubleMatrix;
  bootstrapY: TDoubleArray;
begin
  forest.numTrees := numTrees;
  forest.maxDepth := maxDepth;
  forest.minSamplesSplit := minSamplesSplit;
  SetLength(forest.trees, numTrees);

  Randomize; // Инициализация генератора случайных чисел

  for i := 0 to numTrees - 1 do
  begin
    // Создаём бутстрэп-выборку
    BootstrapSample(x, y, bootstrapX, bootstrapY);

    // Обучаем дерево на бутстрэп-выборке
    TrainDecisionTree(forest.trees[i], bootstrapX, bootstrapY, maxDepth, minSamplesSplit);
  end;
end;

function PredictRandomForest(const forest: TRandomForest; const x: TDoubleArray): Double;
var
  i: Integer;
  predictions: TDoubleArray;
begin
  SetLength(predictions, forest.numTrees);
  for i := 0 to forest.numTrees - 1 do
    predictions[i] := PredictDecisionTree(forest.trees[i], x);

  // Для классификации: мода (наиболее частое значение)
  // Для регрессии: среднее значение
  Result := Sum(predictions) / Length(predictions);
end;

procedure FreeRandomForest(var forest: TRandomForest);
var
  i: Integer;
begin
  for i := 0 to High(forest.trees) do
    FreeDecisionTree(forest.trees[i]);
  SetLength(forest.trees, 0);
end;

end.