program ii3_lemmatize;

{$MODE OBJFPC}{$H+}

uses
  SysUtils, Classes, StrUtils, Lemmatizer;

procedure PrintHelp;
begin
  WriteLn('Usage: ii3_lemmatize [options]');
  WriteLn('Options:');
  WriteLn('  --input <file>       Input file to process');
  WriteLn('  --output <file>      Output file (default: stdout)');
  WriteLn('  --rules <dir>        Directory with rule files');
  WriteLn('  --dict <file>        Custom dictionary file');
  WriteLn('  --exceptions <file>  Custom exceptions file');
  WriteLn('  --suffixes <file>    Custom suffixes rules file');
  WriteLn('  --prefixes <file>    Custom prefixes rules file');
  WriteLn('  --help               Show this help message');
  Halt(0);
end;

type
  TAppOptions = record
    InputFile: string;
    OutputFile: string;
    RulesDir: string;
    DictFile: string;
    ExceptionsFile: string;
    SuffixesFile: string;
    PrefixesFile: string;
  end;

function ParseOptions: TAppOptions;
var
  i: Integer;
begin
  Result.InputFile := '';
  Result.OutputFile := '';
  Result.RulesDir := 'rules/';
  Result.DictFile := '';
  Result.ExceptionsFile := '';
  Result.SuffixesFile := '';
  Result.PrefixesFile := '';

  i := 1;
  while i <= ParamCount do
  begin
    if ParamStr(i) = '--input' then
    begin
      Inc(i);
      Result.InputFile := ParamStr(i);
    end
    else if ParamStr(i) = '--output' then
    begin
      Inc(i);
      Result.OutputFile := ParamStr(i);
    end
    else if ParamStr(i) = '--rules' then
    begin
      Inc(i);
      Result.RulesDir := IncludeTrailingPathDelimiter(ParamStr(i));
    end
    else if ParamStr(i) = '--dict' then
    begin
      Inc(i);
      Result.DictFile := ParamStr(i);
    end
    else if ParamStr(i) = '--exceptions' then
    begin
      Inc(i);
      Result.ExceptionsFile := ParamStr(i);
    end
    else if ParamStr(i) = '--suffixes' then
    begin
      Inc(i);
      Result.SuffixesFile := ParamStr(i);
    end
    else if ParamStr(i) = '--prefixes' then
    begin
      Inc(i);
      Result.PrefixesFile := ParamStr(i);
    end
    else if ParamStr(i) = '--help' then
    begin
      PrintHelp;
    end;
    Inc(i);
  end;

  if Result.InputFile = '' then
  begin
    WriteLn('Error: Input file must be specified');
    PrintHelp;
  end;
end;

function Tokenize(const Text: string): TStringArray;
begin
  Result := SplitString(Text, ' .,;:!?()[]{}"''');
end;

procedure ProcessText(LemmatizerObj: TLemmatizer; const InputFile, OutputFile: string);
var
  InputList, OutputList: TStringList;
  i, j: Integer;
  Tokens: TStringArray;
  Line: string;
begin
  InputList := TStringList.Create;
  OutputList := TStringList.Create;
  try
    InputList.LoadFromFile(InputFile);

    for i := 0 to InputList.Count - 1 do
    begin
      Line := '';
      Tokens := Tokenize(InputList[i]);
      for j := 0 to High(Tokens) do
      begin
        if Tokens[j] <> '' then
        begin
          if Line <> '' then
            Line := Line + ' ';
          Line := Line + LemmatizerObj.Lemmatize(Tokens[j]);
        end;
      end;
      OutputList.Add(Line);
    end;

    if OutputFile = '' then
      Write(OutputList.Text)
    else
      OutputList.SaveToFile(OutputFile);
  finally
    InputList.Free;
    OutputList.Free;
  end;
end;

var
  Options: TAppOptions;
  LemmatizerObj: TLemmatizer;
begin
  try
    Options := ParseOptions;
    
    // Initialize lemmatizer with rules directory
    LemmatizerObj := TLemmatizer.Create(Options.RulesDir);
    try
      // Override default files if specified
      if Options.DictFile <> '' then
        LemmatizerObj.LoadDictionary(Options.DictFile);
      if Options.ExceptionsFile <> '' then
        LemmatizerObj.LoadRules(Options.ExceptionsFile, LemmatizerObj.FExceptions);
      if Options.SuffixesFile <> '' then
        LemmatizerObj.LoadRules(Options.SuffixesFile, LemmatizerObj.FSuffixRules);
      if Options.PrefixesFile <> '' then
        LemmatizerObj.LoadRules(Options.PrefixesFile, LemmatizerObj.FPrefixRules);

      // Process the input file
      ProcessText(LemmatizerObj, Options.InputFile, Options.OutputFile);
    finally
      LemmatizerObj.Free;
    end;
  except
    on E: Exception do
    begin
      WriteLn('Error: ', E.Message);
      Halt(1);
    end;
  end;
end.