unit MusicAnalyzer;
{$MODE OBJFPC}{$H+}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

interface

uses
  SysUtils, Classes, NeuralNetwork, MusicUtils, DataUtils, StrUtils;

type
  TAnalyzerMode = (amAlgorithmic, amNeuralNetwork, amHybrid);

  TMusicAnalyzer = class
  private
    FMode: TAnalyzerMode;
    FNetwork: TNeuralNetwork;
    function SafeAnalyze(const Notes: TNoteArray): TMusicFeatures;
    function CalculateMood(Tempo: Integer; MoodValue: Double): String;
  public
    constructor Create(Mode: TAnalyzerMode);
    destructor Destroy; override;
    function Analyze(const FileName: String): TMusicFeatures;
    property Mode: TAnalyzerMode read FMode write FMode;
  end;

implementation

constructor TMusicAnalyzer.Create(Mode: TAnalyzerMode);
begin
  FMode := Mode;
  if Mode in [amNeuralNetwork, amHybrid] then
    FNetwork := TNeuralNetwork.Create;
end;

destructor TMusicAnalyzer.Destroy;
begin
  if Assigned(FNetwork) then
    FNetwork.Free;
  inherited;
end;

function TMusicAnalyzer.CalculateMood(Tempo: Integer; MoodValue: Double): String;
begin
  if MoodValue > 0.7 then
    Result := 'happy'
  else if MoodValue < 0.3 then
    Result := 'sad'
  else if Tempo > 140 then
    Result := 'energetic'
  else
    Result := 'neutral';
end;

function TMusicAnalyzer.SafeAnalyze(const Notes: TNoteArray): TMusicFeatures;
begin
  // Пока используем только алгоритмический анализ
  Result := SimpleMusicAnalysis(Notes);
  
  // Гарантируем допустимые диапазоны
  Result.Tempo := Round(EnsureRange(Result.Tempo, 40, 220));
  Result.Complexity := EnsureRange(Result.Complexity, 0.1, 1.0);
end;

function TMusicAnalyzer.Analyze(const FileName: String): TMusicFeatures;
var
  Notes: TNoteArray;
begin
  // Проверка файла
  if not FileExists(FileName) then
  begin
    WriteLn('Error: File not found - ', FileName);
    Exit;
  end;

  // Загрузка нот
  Notes := LoadNotesFromFile(FileName);
  if Length(Notes) < 3 then
  begin
    WriteLn('Error: Not enough notes in file (min 3 required)');
    Exit;
  end;

  // Анализ
  Result := SafeAnalyze(Notes);
 
end;

end.