unit MusicUtils;
{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

interface

uses
  SysUtils, Classes, NeuralNetwork, Math, FileSystem,DataUtils;

const
  Keys: array[0..11] of String = ('C', 'C#', 'D', 'D#', 'E', 'F', 
                                 'F#', 'G', 'G#', 'A', 'A#', 'B');

type
  TMusicEvent = (
    meSuccess, meError, meWarning, 
    meAlarm, meQuestion, meResolution,
    meRandom
  );

  TMusicFeatures = record
    Tempo: Integer;
    Key: String;
    Mood: String;
    Complexity: Double;
  end;

function GenerateEventSound(Event: TMusicEvent): TNoteArray;
function HzToTone(FreqHz: Word): Word; inline;
function GetRandomComposition: String;
function AnalyzeMusicFeatures(const Notes: TNoteArray): TMusicFeatures;
function LoadNotesFromFile(const FileName: String): TNoteArray;
function SimpleMusicAnalysis(const Notes: TNoteArray): TMusicFeatures;
function CalculateTempo(const Notes: TNoteArray): Integer;
function CalculateMood(Tempo: Integer; MoodValue: Double): String;
function AnalyzeChord(const Note1, Note2, Note3: TNote): Integer;
function CountUniqueDurations(const Notes: TNoteArray): Integer;
function EnsureRange(Value, MinVal, MaxVal: Double): Double;
function Lerp(A, B, T: Double): Double;

implementation

function HzToTone(FreqHz: Word): Word;
begin
  if FreqHz = 0 then FreqHz := 1;
  Result := 1193182 div FreqHz;
end;

function GenerateEventSound(Event: TMusicEvent): TNoteArray;
var I:integer;
begin
  case Event of
    meSuccess:
      begin
        SetLength(Result, 3);
        Result[0].tone := HzToTone(784);
        Result[0].duration := 150;
        Result[1].tone := HzToTone(1046);
        Result[1].duration := 150;
        Result[2].tone := HzToTone(1318);
        Result[2].duration := 300;
      end;
    meError:
      begin
        SetLength(Result, 4);
        Result[0].tone := HzToTone(523);
        Result[0].duration := 200;
        Result[1].tone := HzToTone(466);
        Result[1].duration := 200;
        Result[2].tone := HzToTone(392);
        Result[2].duration := 200;
        Result[3].tone := HzToTone(349);
        Result[3].duration := 400;
      end;
    meAlarm:
      begin
        SetLength(Result, 8);
        for i := 0 to 7 do
        begin
          Result[i].tone := HzToTone(880 + (i mod 2) * 440);
          Result[i].duration := 100;
        end;
      end;
    meRandom:
      begin
        Result := LoadNotesFromFile(GetRandomComposition);
      end;
    else
      SetLength(Result, 0);
  end;
end;

function GetRandomComposition: String;
var
  Files: TStringList;
  SearchRec: TSearchRec;
begin
  Files := TStringList.Create;
  try
    if FindFirst(GetMusicDir + '/compositions/*.speaker', faAnyFile, SearchRec) = 0 then
    begin
      repeat
        Files.Add(GetMusicDir + '/compositions/' + SearchRec.Name);
      until FindNext(SearchRec) <> 0;
      FindClose(SearchRec);
    end;

    if Files.Count = 0 then
      raise Exception.Create('No compositions found in ' + GetMusicDir + '/compositions/');

    Randomize;
    Result := Files[Random(Files.Count)];
  finally
    Files.Free;
  end;
end;

function LoadNotesFromFile(const FileName: String): TNoteArray;
var
  F: File;
  FileSizeBytes, NotesCount: Integer;
  RawNote: TNote;
  i: Integer;
begin
WriteLn('Загрузка файла: ', FileName);
  SetLength(Result, 0);
  
  if not FileExists(FileName) then
  begin
    WriteLn('Ошибка: Файл не найден - ', FileName);
    Exit;
  end;

  Assign(F, FileName);
  try
    Reset(F, 1);
    FileSizeBytes := System.FileSize(F);
    
    // Проверяем что размер файла кратен размеру TNote (4 байта)
    if (FileSizeBytes mod SizeOf(TNote)) <> 0 then
    begin
      WriteLn('Предупреждение: Размер файла ', FileSizeBytes, 
              ' не кратен ', SizeOf(TNote), ' байтам. Возможно неполная запись.');
    end;
    
    // Вычисляем максимальное количество нот, которые можно прочитать
    NotesCount := FileSizeBytes div SizeOf(TNote);
    if NotesCount = 0 then
    begin
      WriteLn('Ошибка: Файл не содержит ни одной ноты');
      Exit;
    end;
    
    // Читаем ноты по одной и фильтруем только по duration > 0
    SetLength(Result, NotesCount); // Максимально возможный размер
    i := 0;
    while not Eof(F) do
    begin
      BlockRead(F, RawNote, SizeOf(TNote));
      
      // Отбрасываем только ноты с нулевой длительностью
      if RawNote.duration > 0 then
      begin
        // Корректируем слишком короткие ноты
        if RawNote.duration < 10 then 
          RawNote.duration := 10;
          
        Result[i] := RawNote;
        Inc(i);
      end
      else
      begin
        WriteLn('Пропущена нота с нулевой длительностью: tone=', RawNote.tone);
      end;
    end;
    
    // Устанавливаем фактическую длину массива
    SetLength(Result, i);
    
    if Length(Result) = 0 then
    begin
      WriteLn('Ошибка: После фильтрации не осталось ни одной ноты');
    end
    else
    begin
      WriteLn('Успешно загружено нот: ', Length(Result), ' из ', NotesCount);
      WriteLn('Первая нота: tone=', Result[0].tone, 
              ' (~', Round(1193182/Result[0].tone), ' Гц), ',
              'duration=', Result[0].duration, ' мс');
    end;
  except
    on E: Exception do
      WriteLn('Ошибка чтения файла: ', E.Message);
  end;
  
  Close(F);
end;

function SimpleMusicAnalysis(const Notes: TNoteArray): TMusicFeatures;
var
  i, NoteCount: Integer;
  TotalDuration, PeakCount: Integer;
  CurrentTime, LastPeakTime: Integer;
  Intervals: array of Integer;
  MoodScore: Double;
  NoteFreq: Double;
  NoteIndex, MajorCount, MinorCount: Integer;
  NoteHistogram: array[0..11] of Integer;
  Durations: array of Integer;
  UniqueDurations: Integer;
  MaxCount: Integer;
  KeyIndex: Integer;
begin
  // Инициализация
  Result.Tempo := 120;
  Result.Key := 'C major';
  Result.Mood := 'neutral';
  Result.Complexity := 0.5;
  
  NoteCount := Length(Notes);
  if NoteCount < 10 then Exit; // Минимум 10 нот для анализа

  // Обнуляем гистограмму
  for i := 0 to 11 do
    NoteHistogram[i] := 0;

  // Анализ ритма и высоты тона
  CurrentTime := 0;
  PeakCount := 0;
  MoodScore := 0;
  SetLength(Intervals, 0);
  SetLength(Durations, NoteCount);
  MajorCount := 0;
  MinorCount := 0;

  for i := 0 to High(Notes) do
  begin
    if Notes[i].tone = 0 then Continue;
    
    CurrentTime := CurrentTime + Notes[i].duration;
    Durations[i] := Notes[i].duration;
    NoteFreq := 1193182 / Notes[i].tone;
    
    // Заполняем гистограмму нот
    NoteIndex := Round(12 * Log2(NoteFreq / 440.0)) mod 12;
    if NoteIndex < 0 then NoteIndex := 0;
    if NoteIndex > 11 then NoteIndex := 11;
    Inc(NoteHistogram[NoteIndex]);
    
    // Определение мажорных/минорных нот
    if NoteIndex in [0, 4, 7] then Inc(MajorCount);  // Мажорное трезвучие
    if NoteIndex in [0, 3, 7] then Inc(MinorCount);  // Минорное трезвучие
    
    // Определяем ритмические пики
    if (Notes[i].duration < 200) and (NoteFreq > 220) then
    begin
      if PeakCount > 0 then
      begin
        SetLength(Intervals, Length(Intervals) + 1);
        Intervals[High(Intervals)] := CurrentTime - LastPeakTime;
      end;
      LastPeakTime := CurrentTime;
      Inc(PeakCount);
    end;
    
    // Оцениваем настроение
    if NoteFreq > 880 then MoodScore := MoodScore + 0.5    // Высокие ноты
    else if NoteFreq < 220 then MoodScore := MoodScore - 0.5; // Низкие ноты
  end;

  // Расчет темпа
  if Length(Intervals) > 2 then
  begin
    TotalDuration := 0;
    for i := 0 to High(Intervals) do
      TotalDuration := TotalDuration + Intervals[i];
    Result.Tempo := Round(60000 / (TotalDuration / Length(Intervals)));
    Result.Tempo := Min(220, Max(40, Result.Tempo));
  end;

  // Определение тональности
  MaxCount := 0;
  KeyIndex := 0;
  for i := 0 to 11 do
  begin
    if NoteHistogram[i] > MaxCount then
    begin
      MaxCount := NoteHistogram[i];
      KeyIndex := i;
    end;
  end;

  // Определяем мажор/минор
  if MajorCount > MinorCount * 1.3 then
    Result.Key := Keys[KeyIndex] + ' major'
  else if MinorCount > MajorCount * 1.3 then
    Result.Key := Keys[KeyIndex] + ' minor'
  else
    Result.Key := Keys[KeyIndex] + ' (mixed)';

  // Оценка настроения
  MoodScore := MoodScore / NoteCount * 10; // Нормализация
  MoodScore := MoodScore + (Result.Tempo - 100) * 0.02; // Учет темпа
  
  if MoodScore > 1.5 then
    Result.Mood := 'happy'
  else if MoodScore < -1.5 then
    Result.Mood := 'sad'
  else if Result.Tempo > 140 then
    Result.Mood := 'energetic'
  else if Result.Tempo < 80 then
    Result.Mood := 'calm'
  else
    Result.Mood := 'neutral';

  // Оценка сложности
  UniqueDurations := 1;
  for i := 1 to High(Durations) do
    if Durations[i] <> Durations[i-1] then
      Inc(UniqueDurations);

  Result.Complexity := Min(1.0, 
    (NoteCount / 200.0) + 
    (UniqueDurations / 15.0) + 
    (Abs(MoodScore) * 0.3));
end;

function CalculateTempo(const Notes: TNoteArray): Integer;
var
  TotalDuration: Integer;
  i, ValidNotes: Integer;
begin
  Result := 120; // Значение по умолчанию
  ValidNotes := 0;
  TotalDuration := 0;

  for i := 0 to High(Notes) do
  begin
    if Notes[i].duration > 0 then
    begin
      Inc(ValidNotes);
      TotalDuration := TotalDuration + Notes[i].duration;
    end;
  end;

  if (ValidNotes > 1) and (TotalDuration > 0) then
  begin
    Result := Round(60000 * (ValidNotes-1) / TotalDuration);
    // Ограничиваем диапазон
    Result := Min(240, Max(40, Result));
  end;
end;

function CalculateMood(Tempo: Integer; MoodValue: Double): String;
begin
  if MoodValue > 0.7 then
    Result := 'happy'
  else if MoodValue < 0.3 then
    Result := 'sad'
  else if Tempo > 140 then
    Result := 'energetic'
  else if Tempo < 80 then
    Result := 'calm'
  else
    Result := 'neutral';
end;

function AnalyzeChord(const Note1, Note2, Note3: TNote): Integer;
var
  freq1, freq2, freq3, avgFreq, freqRatio: Double;
begin
  Result := -1; // Значение по умолчанию при ошибке
  // Проверяем валидность нот
  if (Note1.tone = 0) or (Note2.tone = 0) or (Note3.tone = 0) then
    Exit;
  try
    // Вычисляем частоты
    freq1 := 1193182 / Note1.tone;
    freq2 := 1193182 / Note2.tone;
    freq3 := 1193182 / Note3.tone;
    // Средняя частота аккорда
    avgFreq := (freq1 + freq2 + freq3) / 3;
    // Проверяем допустимый диапазон частот
    if (avgFreq < 20) or (avgFreq > 5000) then
      Exit;
    // Вычисляем отношение к эталонной частоте
    freqRatio := avgFreq / 440.0;
    if freqRatio <= 0 then
      Exit;
    // Определяем индекс ноты
    Result := Round(12 * Log2(freqRatio)) mod 12;
    if Result < 0 then 
      Result := 0;
  except
    on E: Exception do
    begin
      WriteLn('Chord analysis error: ', E.Message);
      Result := -1;
    end;
  end;
end;

function CountUniqueDurations(const Notes: TNoteArray): Integer;
var
  i: Integer;
  LastDuration: Word;
begin
  Result := 0;
  if Length(Notes) = 0 then Exit;
  
  LastDuration := Notes[0].duration;
  Result := 1;
  
  for i := 1 to High(Notes) do
  begin
    if Notes[i].duration <> LastDuration then
    begin
      Inc(Result);
      LastDuration := Notes[i].duration;
    end;
  end;
end;

// Линейная интерполяция
function Lerp(A, B, T: Double): Double;
begin
  Result := A + (B - A) * T;
end;

// Ограничение диапазона
function EnsureRange(Value, MinVal, MaxVal: Double): Double;
begin
  Result := Value;
  if Result < MinVal then Result := MinVal;
  if Result > MaxVal then Result := MaxVal;
end;

function AnalyzeMusicFeatures(const Notes: TNoteArray): TMusicFeatures;
var
  Network: TNeuralNetwork;
  NetOutputs: TDoubleArray;
  KeyIndex: Integer;
  KeySuffix: String;
const
  Keys: array[0..11] of String = ('C', 'C#', 'D', 'D#', 'E', 'F', 
                                 'F#', 'G', 'G#', 'A', 'A#', 'B');
begin
  // Инициализация по умолчанию
  Result.Tempo := 120;
  Result.Key := 'C major';
  Result.Mood := 'neutral';
  Result.Complexity := 0.5;

  if Length(Notes) < 3 then Exit;

  // Алгоритмический анализ (резервный вариант)
  Result.Tempo := CalculateTempo(Notes);
  if Result.Tempo > 140 then Result.Mood := 'energetic' else
  if Result.Tempo > 100 then Result.Mood := 'happy' else
  if Result.Tempo < 80 then Result.Mood := 'calm' else Result.Mood := 'neutral';

  // Нейросетевой анализ
  Network := TNeuralNetwork.Create;
  try
    if Network.LoadFromFile(GetMusicDir + '/models/music_analyzer.nn') then
    begin
      NetOutputs := Network.Predict(Notes);

      if Length(NetOutputs) >= 5 then
      begin
        // Комбинируем результаты
        Result.Tempo := Round(Lerp(Result.Tempo, 40 + 200*NetOutputs[0], 0.7));
        
        KeyIndex := Round(NetOutputs[1]*11) mod 12;
        if NetOutputs[2] > 0.5 then KeySuffix := ' major' else KeySuffix := ' minor';
        Result.Key := Keys[KeyIndex] + KeySuffix;

        if NetOutputs[3] > 0.7 then
          Result.Mood := 'happy'
        else if NetOutputs[3] < 0.3 then
          Result.Mood := 'sad'
        else if Result.Tempo > 140 then
          Result.Mood := 'energetic';

        Result.Complexity := EnsureRange(NetOutputs[4], 0.1, 1.0);
      end;
    end;
  finally
    Network.Free;
  end;
end;

end.