unit aifourier;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}
{$OPTIMIZATION LEVEL3}
{$OPTIMIZATION PEEPHOLE}
{$OPTIMIZATION REGVAR}
{$OPTIMIZATION LOOPUNROLL}
{$OPTIMIZATION CSE}
{$OPTIMIZATION DFA}
{$RANGECHECKS OFF}
{$OVERFLOWCHECKS OFF}
{$STACKFRAMES OFF}

interface

type
  TDoubleArray = array of Double;

procedure fft(NumSamples: Integer;
              var RealIn, ImagIn: TDoubleArray;
              var RealOut, ImagOut: TDoubleArray);

implementation

function IsPowerOfTwo(x: Integer): Boolean; inline;
begin
  Exit( (x > 0) and ((x and (x - 1)) = 0) );
end;

function NumberOfBitsNeeded(PowerOfTwo: Integer): byte{Integer}; inline;
var
  i: Integer;
begin
  for i := 0 to 31 do
    if (PowerOfTwo and (1 shl i)) <> 0 then
      Exit(i);
  Exit( 0 );
end;

function ReverseBits(Index, NumBits: Integer): Integer; inline;
var
  i: Integer;
begin
Result := 0;
  for i := 0 to NumBits-1 do
  begin
    Result := (Result shl 1) or (Index and 1);
    Index := Index shr 1;
  end;
end;

procedure FourierTransform(AngleNumerator: Double;
                          NumSamples: Integer;
                          var RealIn, ImagIn: TDoubleArray;
                          var RealOut, ImagOut: TDoubleArray);
var
  NumBits, i, j, k, n, BlockSize, BlockEnd: Integer;
  delta_angle, delta_ar: Double;
  alpha, beta: Double;
  tr, ti, ar, ai: Double;
begin
  if not IsPowerOfTwo(NumSamples) or (NumSamples < 2) then
    Exit;

  NumBits := NumberOfBitsNeeded(NumSamples);
  
  // Bit-reversal permutation
  for i := 0 to NumSamples-1 do
  begin
    j := ReverseBits(i, NumBits);
    RealOut[j] := RealIn[i];
    ImagOut[j] := ImagIn[i];
  end;

  // Основной алгоритм FFT
  BlockEnd := 1;
  BlockSize := 2;
  while BlockSize <= NumSamples do
  begin
    delta_angle := AngleNumerator / BlockSize;
    alpha := Sin(0.5 * delta_angle);
    alpha := 2.0 * alpha * alpha;
    beta := Sin(delta_angle);

    i := 0;
    while i < NumSamples do
    begin
      ar := 1.0;
      ai := 0.0;

      j := i;
      for n := 0 to BlockEnd-1 do
      begin
        k := j + BlockEnd;
        tr := ar * RealOut[k] - ai * ImagOut[k];
        ti := ar * ImagOut[k] + ai * RealOut[k];
        RealOut[k] := RealOut[j] - tr;
        ImagOut[k] := ImagOut[j] - ti;
        RealOut[j] := RealOut[j] + tr;
        ImagOut[j] := ImagOut[j] + ti;
        
        delta_ar := alpha * ar + beta * ai;
        ai := ai - (alpha * ai - beta * ar);
        ar := ar - delta_ar;
        Inc(j);
      end;
      Inc(i, BlockSize);
    end;
    BlockEnd := BlockSize;
    BlockSize := BlockSize shl 1;
  end;
end;

procedure fft(NumSamples: Integer;
              var RealIn, ImagIn: TDoubleArray;
              var RealOut, ImagOut: TDoubleArray);
begin
  if (NumSamples <= 0) or (Length(RealIn) < NumSamples) or 
     (Length(ImagIn) < NumSamples) then
    Exit;

  SetLength(RealOut, NumSamples);
  SetLength(ImagOut, NumSamples);
  FourierTransform(2 * Pi, NumSamples, RealIn, ImagIn, RealOut, ImagOut);
end;

end.