program gorg64_musician;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}

uses
  SysUtils, BaseUnix, Unix, FileSystem, MusicUtils,process,DataUtils,MusicAnalyzer,NeuralNetwork;

const
  SPEAKER_PLAYER = 'gorg64_spkplay';

var
  EventType: String = '';
  OutputFile: String = '';
  PlayRandom: Boolean = False;
  i: Integer;
  Notes: array of TNote;
  Features : TMusicFeatures;
  Analyzer: TMusicAnalyzer;
  NeuralNet:TNeuralNetwork;
NetOut : TDoubleArray;
tmps:string;
MinFreq,MaxFreq,Freq:Double;

procedure SaveToSpeakerFile(const Notes: array of TNote; const FileName: String);
var
  F: File;
begin
  Assign(F, FileName);
  Rewrite(F, 1);
  try
    BlockWrite(F, Notes[0], Length(Notes) * SizeOf(TNote));
  finally
    Close(F);
  end;
end;

procedure PlayNotes(const Notes: array of TNote);
var
  TempFile: String;
  Process: TProcess;
begin
  TempFile := GetTempDir + 'temp_sound.speaker';
  SaveToSpeakerFile(Notes, TempFile);

  Process := TProcess.Create(nil);
  try
    Process.Executable := SPEAKER_PLAYER;
    Process.Parameters.Add(TempFile);
    Process.Options := [poWaitOnExit];
    Process.Execute;
  finally
    Process.Free;
    DeleteFile(TempFile);
  end;
end;

procedure PrintHelp;
begin
  WriteLn('Gorg64 Musician - PC Speaker Sound Generator');
  WriteLn('Usage:');
  WriteLn('  Play event sound:');
  WriteLn('    gorg64_musician --event <type>');
  WriteLn('  Play random composition:');
  WriteLn('    gorg64_musician --random');
  WriteLn('  Save to file:');
  WriteLn('    gorg64_musician --event <type> --output <file.speaker>');
  WriteLn;
  WriteLn('Available event types: success, error, warning, alarm, question, resolution');
  WriteLn;
  WriteLn('Note: Place your .speaker files in ~/.musician/compositions/ for random play');
end;

begin
  InitMusicEnvironment;
  WriteLn('Music directories initialized in: ', GetMusicDir);

  // Парсинг аргументов
  if ParamCount = 0 then
  begin
    PrintHelp;
    Exit;
  end;

  i := 1;
  while i <= ParamCount do
  begin
    if ParamStr(i) = '--help' then
    begin
      PrintHelp;
      Exit;
    end
    else if ParamStr(1) = '--analyze' then
    begin
//      Features := AnalyzeMusicFeatures(LoadNotesFromFile(ParamStr(2)));
//      WriteLn('Music analysis:');
//      WriteLn('  Tempo: ', Features.Tempo, ' BPM');
//      WriteLn('  Key: ', Features.Key);
//      WriteLn('  Mood: ', Features.Mood);
//      WriteLn('  Complexity: ', Features.Complexity:0:2);
//  Analyzer := TMusicAnalyzer.Create(amHybrid);
  NeuralNet := TNeuralNetwork.Create;
  // Создаем анализатор в гибридном режиме
  Analyzer := TMusicAnalyzer.Create(amHybrid);
  Analyzer.Mode := amHybrid;
  try
    if not NeuralNet.LoadFromFile(GetMusicDir + '/models/music_analyzer.nn') then
      WriteLn('Предупреждение: Не удалось загрузить модель нейросети');
    Features := Analyzer.Analyze(ParamStr(2));
//    WriteLn('Tempo: ', Features.Tempo);
//    WriteLn('Key: ', Features.Key);
//    WriteLn('Mood: ', Features.Mood);
//    WriteLn('Complexity: ', Features.Complexity);
// Вместо текущего вывода Analysis details добавим:

// После получения Features:
WriteLn('=== Результаты анализа ===');
WriteLn('  Tempo: ', Features.Tempo);
WriteLn('  Key: ', Features.Key);
WriteLn('  Mood: ', Features.Mood);
WriteLn('  Complexity: ', Features.Complexity:0:2);

// Замените блок вывода статистики на:
if Length(Notes) > 0 then
begin
  WriteLn('Статистика:');
  WriteLn('  Всего нот: ', Length(Notes));
  WriteLn('  Различных длительностей: ', CountUniqueDurations(Notes));
  
  // Безопасное вычисление диапазона частот
  MinFreq := 0;
  MaxFreq := 0;
  if Notes[0].tone > 0 then
  begin
    MinFreq := Round(1193182/Notes[0].tone);
    MaxFreq := MinFreq;
  end;
  
  for i := 1 to High(Notes) do
  begin
    if Notes[i].tone > 0 then
    begin
      Freq := Round(1193182/Notes[i].tone);
      if Freq < MinFreq then MinFreq := Freq;
      if Freq > MaxFreq then MaxFreq := Freq;
    end;
  end;
  
  if (MinFreq > 0) and (MaxFreq > 0) then
    WriteLn('  Диапазон частот: ', MinFreq, '-', MaxFreq, ' Гц')
  else
    WriteLn('  Диапазон частот: не определен');
end
else
begin
  WriteLn('Статистика: нет данных (массив нот пуст)');
end;


{
WriteLn('=== Результаты анализа ===');
WriteLn('Алгоритмический анализ:');
WriteLn('  Tempo: ', SimpleMusicAnalysis(Notes).Tempo);
WriteLn('  Key: ', SimpleMusicAnalysis(Notes).Key);
WriteLn('  Mood: ', SimpleMusicAnalysis(Notes).Mood);
WriteLn('  Complexity: ', SimpleMusicAnalysis(Notes).Complexity:0:2);

if (Analyzer.Mode in [amHybrid, amNeuralNetwork]) and Assigned(NeuralNet) then
begin
  NetOut := NeuralNet.Predict(Notes);
  WriteLn('Нейросетевой анализ:');
  WriteLn('  Tempo: ', Round(40 + 200*NetOut[0]));
if NetOut[2] > 0.5 then tmps := ' major' else tmps := ' minor';
  WriteLn('  Key: ', Keys[Round(NetOut[1]*11)] + tmps);
if NetOut[3] > 0.7 then tmps := 'happy' else
 if NetOut[3] < 0.3 then tmps := 'sad' else tmps := 'neutral';
  WriteLn('  Mood: ', tmps);
  WriteLn('  Complexity: ', NetOut[4]:0:2);
end;
}

WriteLn('Финальные результаты:');
WriteLn('  Tempo: ', Features.Tempo);
WriteLn('  Key: ', Features.Key);
WriteLn('  Mood: ', Features.Mood);
WriteLn('  Complexity: ', Features.Complexity:0:2);

  finally
    Analyzer.Free;
    NeuralNet.Free;
  end;
      Exit;
    end
    else if ParamStr(i) = '--event' then
    begin
      Inc(i);
      if i <= ParamCount then
        EventType := LowerCase(ParamStr(i));
    end
    else if ParamStr(i) = '--random' then
    begin
      PlayRandom := True;
    end
    else if ParamStr(i) = '--output' then
    begin
      Inc(i);
      if i <= ParamCount then
        OutputFile := ExpandFileName(ParamStr(i));
    end;
    Inc(i);
  end;

  // Генерация звука
  try
    if PlayRandom then
    begin
      Notes := GenerateEventSound(meRandom);
      WriteLn('Playing random composition');
    end
    else if EventType <> '' then
    begin
      case EventType of
        'success':    Notes := GenerateEventSound(meSuccess);
        'error':      Notes := GenerateEventSound(meError);
        'warning':    Notes := GenerateEventSound(meWarning);
        'alarm':      Notes := GenerateEventSound(meAlarm);
        'question':   Notes := GenerateEventSound(meQuestion);
        'resolution': Notes := GenerateEventSound(meResolution);
      else
        begin
          WriteLn('Error: Unknown event type "', EventType, '"');
          PrintHelp;
          Halt(1);
        end;
      end;
      WriteLn('Playing sound for event: ', EventType);
    end
    else
    begin
      WriteLn('Error: No action specified');
      PrintHelp;
      Halt(1);
    end;

    // Сохранение или воспроизведение
    if OutputFile <> '' then
    begin
      SaveToSpeakerFile(Notes, OutputFile);
      WriteLn('Sound saved to: ', OutputFile);
    end
    else
    begin
      PlayNotes(Notes);
    end;
  except
    on E: Exception do
    begin
      WriteLn('Error: ', E.Message);
      Halt(1);
    end;
  end;
end.