program test_fft;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}
uses
  SysUtils, DateUtils, Math, fourier, aifourier;

const
  FFT_SIZE = 4096;
  TEST_ITERATIONS = 1000;

function Max(a, b: Double): Double; inline;
begin
  if a > b then Result := a else Result := b;
end;

procedure TestCorrectness;
var
  i: Integer;
  RealIn, ImagIn, RealOut1, ImagOut1, RealOut2, ImagOut2: array of Double;
  MaxError: Double;
begin
  WriteLn('Testing correctness...');

  // Инициализация массивов
  SetLength(RealIn, FFT_SIZE);
  SetLength(ImagIn, FFT_SIZE);
  SetLength(RealOut1, FFT_SIZE);
  SetLength(ImagOut1, FFT_SIZE);
  SetLength(RealOut2, FFT_SIZE);
  SetLength(ImagOut2, FFT_SIZE);

  // Заполнение входных данных
  for i := 0 to FFT_SIZE-1 do
  begin
    RealIn[i] := Sin(2 * Pi * i / FFT_SIZE);
    ImagIn[i] := 0;
  end;

  // Оригинальная FFT
  fourier.fft(FFT_SIZE, RealIn, ImagIn, RealOut1, ImagOut1);

  // Оптимизированная FFT
  aifourier.fft(FFT_SIZE, RealIn, ImagIn, RealOut2, ImagOut2);

  // Проверка результатов
  MaxError := 0;
  for i := 0 to FFT_SIZE-1 do
  begin
    MaxError := Max(MaxError, Abs(RealOut1[i] - RealOut2[i]));
    MaxError := Max(MaxError, Abs(ImagOut1[i] - ImagOut2[i]));
  end;

  WriteLn('Max error: ', MaxError);
  if MaxError < 1e-10 then
    WriteLn('PASS: Results match')
  else
    WriteLn('FAIL: Significant differences found');
end;

procedure TestPerformance;
var
  i, j: Integer;
  StartTime: TDateTime;
  Duration1, Duration2: Double;
  RealIn, ImagIn, RealOut1, ImagOut1, RealOut2, ImagOut2: array of Double;
begin
  WriteLn('Testing performance...');

  // Инициализация массивов
  SetLength(RealIn, FFT_SIZE);
  SetLength(ImagIn, FFT_SIZE);
  SetLength(RealOut1, FFT_SIZE);
  SetLength(ImagOut1, FFT_SIZE);
  SetLength(RealOut2, FFT_SIZE);
  SetLength(ImagOut2, FFT_SIZE);

  // Заполнение входных данных
  for i := 0 to FFT_SIZE-1 do
  begin
    RealIn[i] := Random;
    ImagIn[i] := 0;
  end;

  // Тест оригинальной реализации
  StartTime := Now;
  for j := 1 to TEST_ITERATIONS do
    fourier.fft(FFT_SIZE, RealIn, ImagIn, RealOut1, ImagOut1);
  Duration1 := MilliSecondsBetween(Now, StartTime) / TEST_ITERATIONS;

  // Тест оптимизированной реализации
  StartTime := Now;
  for j := 1 to TEST_ITERATIONS do
    aifourier.fft(FFT_SIZE, RealIn, ImagIn, RealOut2, ImagOut2);
  Duration2 := MilliSecondsBetween(Now, StartTime) / TEST_ITERATIONS;

  WriteLn('Original FFT: ', Duration1:0:3, ' ms/iteration');
  WriteLn('Optimized FFT: ', Duration2:0:3, ' ms/iteration');
  WriteLn('Speedup: ', (Duration1/Duration2):0:2, 'x');
end;

begin
  Randomize;
  TestCorrectness;
  TestPerformance;
end.