program wav2speaker_old;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}

{$MODE OBJFPC}{$H+}
uses
  SysUtils, Math, wavfileunit, Fourier;

const
  SAMPLE_RATE = 44100;
  FFT_SIZE = 4096;  // Увеличили размер FFT для лучшего разрешения низких частот
  MIN_FREQ = 65;    // Минимальная частота (C2)
  MAX_FREQ = 1046;  // Максимальная частота (C6)
  MIN_NOTE_DURATION = 30; // Более короткие ноты для вокала
  SILENCE_THRESHOLD = 0.005; // Более чувствительный порог тишины
  NOTE_TOLERANCE = 0.02; // Более строгий допуск (2%)
  OVERLAP_FACTOR = 8; // Большее перекрытие окон для плавности
  VIBRATO_TOLERANCE = 0.05; // Допуск для вибрато
  VIBRATO_RATE = 5.0; // Частота вибрато (Гц)
  GLISSANDO_THRESHOLD = 0.15; // Порог для плавных переходов

// Для рок-музыки (как "Пикник")
//  FFT_SIZE = 1024;
//  SILENCE_THRESHOLD = 0.1;
//  NOTE_TOLERANCE = 0.03;

// Для классического вокала (Шаляпин)
//  FFT_SIZE = 2048;
//  SILENCE_THRESHOLD = 0.02;
//  NOTE_TOLERANCE = 0.05;
//  MIN_FREQ = 80;  // Более высокий порог для баритона

type
  TNote = packed record
    tone, duration: Word;
  end;
  TFFTArray = array[0..FFT_SIZE-1] of Double;

// Расширенная таблица частот нот для баритона/баса
const
  NOTE_FREQUENCIES: array[0..48] of Double = (
    // Октава 2 (бас)
    65.41, 69.30, 73.42, 77.78, 82.41, 87.31, 92.50, 98.00, 103.83, 110.00, 116.54, 123.47,
    // Октава 3
    130.81, 138.59, 146.83, 155.56, 164.81, 174.61, 185.00, 196.00, 207.65, 220.00, 233.08, 246.94,
    // Октава 4
    261.63, 277.18, 293.66, 311.13, 329.63, 349.23, 369.99, 392.00, 415.30, 440.00, 466.16, 493.88,
    // Октава 5
    523.25, 554.37, 587.33, 622.25, 659.26, 698.46, 739.99, 783.99, 830.61, 880.00, 932.33, 987.77,
    // Октава 6 (для высоких нот)
    1046.50
  );

function HzToTone(FreqHz: Double): Word;
var
  i: Integer;
  minDiff, diff: Double;
begin
  if FreqHz < MIN_FREQ then
    Exit(0);
    
  if FreqHz > MAX_FREQ then
    FreqHz := MAX_FREQ;

  // Находим ближайшую ноту с учетом логарифмической шкалы
  minDiff := MaxDouble;
  Result := 0;
  
  for i := 0 to High(NOTE_FREQUENCIES) do
  begin
    diff := Abs(Ln(FreqHz / NOTE_FREQUENCIES[i]));
    if diff < minDiff then
    begin
      minDiff := diff;
      // Добавляем небольшую коррекцию для точного соответствия нотам
      Result := Round(1193182 / (NOTE_FREQUENCIES[i] * Power(2, diff/2)));
    end;
  end;
end;

// Новая функция для детекции глиссандо
function IsGlissando(CurrentFreq, LastFreq: Double): Boolean;
begin
  Result := (LastFreq > 0) and 
            (Abs(CurrentFreq - LastFreq) / LastFreq > GLISSANDO_THRESHOLD) and
            (Abs(CurrentFreq - LastFreq) / LastFreq < 0.5);
end;

procedure ApplyWindow(var Buffer: TFFTArray);
var
  i: Integer;
  N, window: Double;
begin
  // Улучшенное окно Блэкмана-Харриса для музыкального анализа
  N := FFT_SIZE - 1;
  for i := 0 to FFT_SIZE - 1 do
  begin
    window := 0.35875 - 
              0.48829 * Cos(2 * Pi * i / N) + 
              0.14128 * Cos(4 * Pi * i / N) - 
              0.01168 * Cos(6 * Pi * i / N);
    Buffer[i] := Buffer[i] * window;
  end;
end;

function FindDominantFrequency(var RealIn: TFFTArray): Double;
var
  i, MaxIndex: Integer;
  MaxValue, CurrentMagnitude: Double;
  RealOut, ImagOut, ImagIn: TFFTArray;
begin
  FillChar(ImagIn, SizeOf(ImagIn), 0);
  
  // Выполняем FFT
  fft(FFT_SIZE, RealIn, ImagIn, RealOut, ImagOut);

  MaxValue := 0;
  MaxIndex := 0;

  // Ищем пик в частотном диапазоне
  for i := Round(MIN_FREQ * FFT_SIZE / SAMPLE_RATE) to 
            Round(MAX_FREQ * FFT_SIZE / SAMPLE_RATE) do
  begin
    CurrentMagnitude := Sqrt(RealOut[i]*RealOut[i] + ImagOut[i]*ImagOut[i]);
    
    // Усиливаем основные частоты вокала (100-1000 Гц)
    if (i >= Round(100 * FFT_SIZE / SAMPLE_RATE)) and 
       (i <= Round(1000 * FFT_SIZE / SAMPLE_RATE)) then
      CurrentMagnitude := CurrentMagnitude * 1.5;
    
    if CurrentMagnitude > MaxValue then
    begin
      MaxValue := CurrentMagnitude;
      MaxIndex := i;
    end;
  end;

  if MaxValue > SILENCE_THRESHOLD then
    Result := MaxIndex * SAMPLE_RATE / FFT_SIZE
  else
    Result := 0;
end;

function IsSilence(const Buffer: TFFTArray): Boolean;
var
  i: Integer;
  Sum, Avg, MaxAmp: Double;
begin
  Sum := 0;
  MaxAmp := 0;
  for i := 0 to FFT_SIZE - 1 do
  begin
    Sum := Sum + Abs(Buffer[i]);
    if Abs(Buffer[i]) > MaxAmp then
      MaxAmp := Abs(Buffer[i]);
  end;
  
  // Используем и среднее значение, и максимальную амплитуду
  Avg := Sum / FFT_SIZE;
  Result := (Avg < SILENCE_THRESHOLD) or (MaxAmp < SILENCE_THRESHOLD * 3);
end;

procedure ConvertWavToSpeaker(const InputFile, OutputFile: string);
var
  WavFile: TWAVFile;
  Amplitudes: array of TAmplitudeArray;
  SpeakerNotes: array of TNote;
  i, NumSamples, NoteIndex: Integer;
  CurrentFreq, LastFreq: Double;
  CurrentDuration: Word;
  RealIn: TFFTArray;
  SamplesProcessed, StepSize: Integer;
  F: File;
  SameFreqCount: Integer;
begin
  WavFile := TWAVFile.Create(InputFile);
  try
    WavFile.LoadFromFile(InputFile);
    
    if WavFile.GetNumChannels <> 1 then
      raise Exception.Create('Только монофонические WAV-файлы поддерживаются');
      
    if WavFile.GetBitsPerSample <> 16 then
      raise Exception.Create('Только 16-битные WAV-файлы поддерживаются');
    
    // Чтение амплитуд
    SetLength(Amplitudes, 1);
    NumSamples := WavFile.ReadAmplitudes(Amplitudes);
    
    SetLength(SpeakerNotes, NumSamples div (FFT_SIZE div OVERLAP_FACTOR) + 100);
    NoteIndex := 0;
    LastFreq := 0;
    CurrentDuration := 0;
    SamplesProcessed := 0;
    StepSize := FFT_SIZE div OVERLAP_FACTOR;
    SameFreqCount := 0;

    while SamplesProcessed + FFT_SIZE <= NumSamples do
    begin
      // Заполняем буфер для БПФ
      for i := 0 to FFT_SIZE - 1 do
        RealIn[i] := Amplitudes[0][SamplesProcessed + i];
      
      ApplyWindow(RealIn);
      
      if IsSilence(RealIn) then
      begin
        if (LastFreq > 0) and (CurrentDuration >= MIN_NOTE_DURATION) then
        begin
          SpeakerNotes[NoteIndex].tone := HzToTone(LastFreq);
          SpeakerNotes[NoteIndex].duration := CurrentDuration;
          Inc(NoteIndex);
        end;
        LastFreq := 0;
        CurrentDuration := 0;
        SameFreqCount := 0;
      end
      else
      begin
        CurrentFreq := FindDominantFrequency(RealIn);
        
        if CurrentFreq > 0 then
        begin
          // Простое сравнение частот без сложной логики
          if (LastFreq = 0) or 
             (Abs(CurrentFreq - LastFreq) > LastFreq * NOTE_TOLERANCE) then
          begin
            if (LastFreq > 0) and (CurrentDuration >= MIN_NOTE_DURATION) and
               (SameFreqCount >= 3) then
            begin
              SpeakerNotes[NoteIndex].tone := HzToTone(LastFreq);
              SpeakerNotes[NoteIndex].duration := CurrentDuration;
              Inc(NoteIndex);
            end;
            LastFreq := CurrentFreq;
            CurrentDuration := (1000 * StepSize) div SAMPLE_RATE;
            SameFreqCount := 1;
          end
          else
          begin
            Inc(CurrentDuration, (1000 * StepSize) div SAMPLE_RATE);
            Inc(SameFreqCount);
          end;
        end;
      end;

      Inc(SamplesProcessed, StepSize);
    end;

    // Добавляем последнюю ноту
    if (LastFreq > 0) and (CurrentDuration >= MIN_NOTE_DURATION) then
    begin
      SpeakerNotes[NoteIndex].tone := HzToTone(LastFreq);
      SpeakerNotes[NoteIndex].duration := CurrentDuration;
      Inc(NoteIndex);
    end;

    // Обрезаем массив до фактического размера
    SetLength(SpeakerNotes, NoteIndex);
    
    // Сохранение в файл
    Assign(F, OutputFile);
    Rewrite(F, 1);
    try
      BlockWrite(F, SpeakerNotes[0], Length(SpeakerNotes) * SizeOf(TNote));
      WriteLn('Успешно конвертировано:');
      WriteLn('  Нот создано: ', NoteIndex);
      WriteLn('  Размер файла: ', FileSize(F), ' байт');
    finally
      Close(F);
    end;
    
  finally
    WavFile.Free;
  end;
end;

begin
  if ParamCount < 2 then
  begin
    WriteLn('Использование: wav2speaker <input.wav> <output.speaker>');
    Exit;
  end;
  
  try
    ConvertWavToSpeaker(ParamStr(1), ParamStr(2));
  except
    on E: Exception do
      WriteLn('Ошибка: ', E.Message);
  end;
end.