unit MathEvaluator;
{$MODE OBJFPC}{$H+}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


interface

uses
  SysUtils, Math, fpExprPars;

type
  TMathEvaluator = class
  private
    class function IsMathExpression(const Input: string): Boolean;
    class function CleanExpression(Expr: string): string;
  public
    class function Evaluate(const Input: string; out Output: string): Boolean;
    class function HasNumbers(const Expr: string): Boolean;
  end;

implementation

class function TMathEvaluator.IsMathExpression(const Input: string): Boolean;
const
  MATH_SYMBOLS = ['+', '-', '*', '/', '^', '(', ')', '.', '0'..'9'];
  MATH_FUNCS: array[0..4] of string = ('sqrt', 'sin', 'cos', 'tan', 'log');
var
  i: Integer;
  LowerInput: string;
begin
  LowerInput := LowerCase(Trim(Input));
  
  // Проверка на явные математические запросы
  if (Pos('сколько будет', LowerInput) > 0) or
     (Pos('вычисли', LowerInput) > 0) or
     (Pos('посчитай', LowerInput) > 0) then
    Exit(True);

  // Проверка математических операторов
  for i := 1 to Length(LowerInput) do
    if LowerInput[i] in MATH_SYMBOLS then
      Exit(True);

  // Проверка математических функций
  for i := Low(MATH_FUNCS) to High(MATH_FUNCS) do
    if Pos(MATH_FUNCS[i], LowerInput) > 0 then
      Exit(True);

  // Проверка текстовых формулировок
  if (Pos('корень', LowerInput) > 0) or
     (Pos('степень', LowerInput) > 0) then
    Exit(True);

  Result := False;
end;

class function TMathEvaluator.CleanExpression(Expr: string): string;
begin
  Result := LowerCase(Trim(Expr));

  // Обработка русскоязычных формулировок
  Result := StringReplace(Result, 'корень из', 'sqrt(', [rfIgnoreCase]);
  if Pos('sqrt(', Result) > 0 then
    Result := Result + ')';
  
  Result := StringReplace(Result, ' в степени ', '^', [rfReplaceAll]);
  Result := StringReplace(Result, 'степень', '^', [rfReplaceAll]);
  
  // Удаление служебных слов
  Result := StringReplace(Result, 'сколько будет', '', [rfIgnoreCase]);
  Result := StringReplace(Result, 'вычисли', '', [rfIgnoreCase]);
  Result := StringReplace(Result, 'посчитай', '', [rfIgnoreCase]);
  
  // Удаление пробелов и пунктуации
  Result := StringReplace(Result, ' ', '', [rfReplaceAll]);
  Result := StringReplace(Result, '?', '', [rfReplaceAll]);
  Result := StringReplace(Result, '!', '', [rfReplaceAll]);
  
  // Конвертация градусов в радианы для тригонометрии
  Result := StringReplace(Result, 'sin(', 'sin(pi/180*', [rfReplaceAll]);
  Result := StringReplace(Result, 'cos(', 'cos(pi/180*', [rfReplaceAll]);
  Result := StringReplace(Result, 'tan(', 'tan(pi/180*', [rfReplaceAll]);
  
  Result := Trim(Result);
end;

class function TMathEvaluator.Evaluate(const Input: string; out Output: string): Boolean;
var
  Expr, CleanInput: string;
  Parser: TFPExpressionParser;
begin
  Result := False;
  Output := '';
  CleanInput := Trim(Input);

  if not IsMathExpression(CleanInput) then
    Exit;

  Expr := CleanExpression(CleanInput);
  if Expr = '' then Exit;

  try
    Parser := TFPExpressionParser.Create(nil);
    try
      Parser.BuiltIns := [bcMath];
      Parser.Identifiers.AddFloatVariable('pi', Pi);
      Parser.Expression := Expr;

      case Parser.Evaluate.ResultType of
        rtFloat:
          if SameValue(Frac(Parser.Evaluate.ResFloat), 0, 0.001) then
            Output := Format('%s = %d', [CleanInput, Trunc(Parser.Evaluate.ResFloat)])
          else
            Output := Format('%s = %.2f', [CleanInput, Parser.Evaluate.ResFloat]);
        rtInteger:
          Output := Format('%s = %d', [CleanInput, Parser.Evaluate.ResInteger]);
      else
        Exit;
      end;

      Result := True;
    finally
      Parser.Free;
    end;
  except
    on E: Exception do
      Output := 'Ошибка вычисления: ' + E.Message;
  end;

  if SameValue(Frac(Parser.Evaluate.ResFloat), 0, 0.001) then
    Output := Format('%d', [Trunc(Parser.Evaluate.ResFloat)])  // Целые числа без .00
  else
    Output := Format('%.2f', [Parser.Evaluate.ResFloat]);      // Дробные с 2 знаками
  
  // Убираем дублирование вопроса в ответе
  Output := StringReplace(Output, CleanInput + ' = ', '', []);
end;

class function TMathEvaluator.HasNumbers(const Expr: string): Boolean;
var
  i: Integer;
begin
  for i := 1 to Length(Expr) do
    if Expr[i] in ['0'..'9'] then
      Exit(True);
  Result := False;
end;

end.