unit MathProcessor;
{$MODE OBJFPC}{$H+}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


interface

uses
  SysUtils, Math, fpExprPars;

type
  TMathProcessor = class
  private
    class function ConvertDegreesToRadians(const Expr: string): string;
  public
    class function Calculate(const Input: string; out Output: string): Boolean;
  end;

function ExtractMathExpression(const Input: string): string;
function HasMathOperators(const Expr: string): Boolean;

implementation

class function TMathProcessor.ConvertDegreesToRadians(const Expr: string): string;
begin
  Result := StringReplace(Expr, 'sin(', 'sin(pi/180*', [rfReplaceAll]);
  Result := StringReplace(Result, 'cos(', 'cos(pi/180*', [rfReplaceAll]);
  Result := StringReplace(Result, 'tan(', 'tan(pi/180*', [rfReplaceAll]);
end;

class function TMathProcessor.Calculate(const Input: string; out Output: string): Boolean;
var
  Expr, CleanInput: string;
  Parser: TFPExpressionParser;
begin
  Result := False;
  CleanInput := LowerCase(Trim(Input));

  // Подготовка выражения
  Expr := CleanInput;
  Expr := StringReplace(Expr, '?', '', [rfReplaceAll]);
  Expr := StringReplace(Expr, 'сколько будет', '', [rfIgnoreCase]);
  Expr := StringReplace(Expr, 'вычисли', '', [rfIgnoreCase]);
  Expr := StringReplace(Expr, 'посчитай', '', [rfIgnoreCase]);
  Expr := StringReplace(Expr, ' в степени ', '^', [rfReplaceAll]);
  Expr := StringReplace(Expr, 'корень из', 'sqrt(', [rfReplaceAll]) + ')';
  Expr := StringReplace(Expr, ' ', '', [rfReplaceAll]);
  
  // Обработка случаев типа "корень из 16" без скобок
  if Pos('sqrt', Expr) > 0 then
  begin
    if Pos('(', Expr) = 0 then
      Insert('(', Expr, Pos('sqrt', Expr) + 4);
    if Pos(')', Expr) = 0 then
      Expr := Expr + ')';
  end;

  Expr := ConvertDegreesToRadians(Expr);

  if Expr = '' then Exit;

  try
    Parser := TFPExpressionParser.Create(nil);
    try
      Parser.BuiltIns := [bcMath];
      Parser.Identifiers.AddFloatVariable('pi', Pi);
      Parser.Expression := Expr;
      
      case Parser.Evaluate.ResultType of
        rtFloat: Output := Format('%.2f', [Parser.Evaluate.ResFloat]);
        rtInteger: Output := IntToStr(Parser.Evaluate.ResInteger);
        else Exit;
      end;
      Result := True;
    finally
      Parser.Free;
    end;
  except
    on E: Exception do
      Output := 'Ошибка вычисления: ' + E.Message;
  end;
end;

function ExtractMathExpression(const Input: string): string;
begin
  Result := Trim(StringReplace(Input, '?', '', [rfReplaceAll]));
  
  // Удаляем служебные слова, сохраняя структуру
  Result := StringReplace(Result, 'сколько будет', '', [rfIgnoreCase]);
  Result := StringReplace(Result, 'вычисли', '', [rfIgnoreCase]);
  Result := StringReplace(Result, 'посчитай', '', [rfIgnoreCase]);
  Result := Trim(Result);
  
  // Заменяем словесные операторы на символы
  Result := StringReplace(Result, ' в степени ', '^', []);
  Result := StringReplace(Result, 'корень из', '√', []);
end;

function HasMathOperators(const Expr: string): Boolean;
const
  OPERATORS = ['+', '-', '*', '/', '^', '(', ')'];
var
  i: Integer;
begin
  Result := False;
  for i := 1 to Length(Expr) do
    if Expr[i] in OPERATORS then
      Exit(True);
end;

end.