unit MindCore;
{$MODE OBJFPC}{$H+}
{$MODESWITCH ADVANCEDRECORDS}
//{$CODEPAGE UTF8}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


interface

uses
  cwstring, Classes, SysUtils, DecisionTree, NLPUtils, StrUtils, LazUTF8, DateUtils, RegExpr, fpExprPars, MathProcessor,MathEvaluator,
  process, fpjson, jsonparser;

type
  TStringConverter = class
    class function RussianLowerCase(const S: string): string;
  end;

  TThought = record
    Content: String;
    Confidence: Double;
    Source: String;
    function ToString: String;
  end;

  TMindEngine = class
  private
    FDecisionTree: TDecisionTree;
    FKnowledgeBase: TStringList;
    FExamples: TStringList;
    procedure LoadKnowledgeBase;
    procedure LoadExamples;
    procedure TrainDecisionTree;
    function GetFeatures(const Input: String): TDoubleArray;
    function GetContextualResponse(const Input: String): String;
    function FindBestMatch(const Input: String): String;
    function ProcessMathQuestion(const Input: String): TThought;
    function TryCalculateMath(const Question: string; out Answer: string): Boolean;
  public
    constructor Create;
    destructor Destroy; override;
    function ProcessInput(const Input: String): TThought;
    procedure Learn(const Input, Response: String);
    function ProcessMathExpression(const Input: String): TThought;
  end;

function GetCurrentTimeString: String;
function IsMathQuestion(const Question: string): Boolean;

implementation

function IsMathQuestion(const Question: string): Boolean;
begin
  Result := ContainsText(Question, 'сколько будет') or 
            ContainsText(Question, 'вычисли') or 
            ContainsText(Question, 'посчитай') or
            (Pos('(', Question) > 0) or 
            (Pos(')', Question) > 0);
end;

function GetCurrentTimeString: String;
var
  Hour, Minute, Second, MilliSecond: Word;
  TimeStr: String;
begin
  DecodeTime(Now, Hour, Minute, Second, MilliSecond);
  
  // Разные формулировки для разнообразия
  case Random(3) of
    0: TimeStr := Format('На часах %d:%2.2d', [Hour, Minute]);
    1: TimeStr := Format('Точное время: %d:%2.2d', [Hour, Minute]);
    2: TimeStr := Format('Сейчас %d:%2.2d', [Hour, Minute]);
  end;

  // Добавляем приветствие по времени суток
  if Hour < 5 then
    TimeStr := TimeStr + '. Ночь на дворе!'
  else if Hour < 12 then
    TimeStr := TimeStr + '. Утро доброе!'
  else if Hour < 17 then 
    TimeStr := TimeStr + '. День в разгаре!'
  else
    TimeStr := TimeStr + '. Вечер наступил!';

  Result := TimeStr;
end;

class function TStringConverter.RussianLowerCase(const S: string): string;
const
  UpperRus = 'АБВГДЕЁЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯ';
  LowerRus = 'абвгдеёжзийклмнопрстуфхцчшщъыьэюя';
var
  i, p: Integer;
  uStr: string;
begin
  Result := UTF8LowerCase(S); // Сначала обрабатываем стандартные символы
  uStr := UTF8Encode(S); // Явное преобразование в UTF-8
  
  for i := 1 to UTF8Length(uStr) do
  begin
    p := UTF8Pos(UTF8Copy(uStr, i, 1), UpperRus);
    if p > 0 then
      UTF8Delete(Result, i, 1);
      UTF8Insert(UTF8Copy(LowerRus, p, 1), Result, i);
  end;
end;

{ TThought }
function TThought.ToString: String;
begin
  Result := Content + ' [' + FloatToStrF(Confidence*100, ffFixed, 5, 2) + '%]';
  Result := Result + ' (Источник: ' + Source + ')';
end;

{ TMindEngine }
constructor TMindEngine.Create;
begin
  FDecisionTree := TDecisionTree.Create(10, 3);
  FKnowledgeBase := TStringList.Create;
  FExamples := TStringList.Create;
  LoadKnowledgeBase;
  LoadExamples;
  TrainDecisionTree;
end;

destructor TMindEngine.Destroy;
begin
  FreeAndNil(FDecisionTree);
  FreeAndNil(FKnowledgeBase);
  FreeAndNil(FExamples);
  inherited;
end;

function TMindEngine.GetFeatures(const Input: String): TDoubleArray;
begin
  Result := NLPUtils.TextToFeatures(Input);
end;

procedure TMindEngine.LoadKnowledgeBase;
begin
  FKnowledgeBase.Values['привет'] := 'Здравствуйте!';
  FKnowledgeBase.Values['здравствуй'] := 'Приветствую вас!';
  FKnowledgeBase.Values['как дела'] := 'Всё отлично, спасибо!';
  FKnowledgeBase.Values['пока'] := 'До свидания!';
  FKnowledgeBase.Values['сколько будет 2+2'] := 'Два плюс два равно четыре';
end;

procedure TMindEngine.LoadExamples;
begin
  // Основные вопросы и ответы
  // Приветствия (все ключи в нижнем регистре)
  FExamples.Values['привет'] := 'Привет! Как ваши дела?';
  FExamples.Values['привет!'] := 'Привет!';
  FExamples.Values['здравствуй'] := 'Здравствуй!';
  FExamples.Values['здравствуйте'] := 'Здравствуйте!';
  FExamples.Values['добрый день'] := 'Добрый день!';
  FExamples.Values['доброе утро'] := 'Доброе утро!';
  FExamples.Values['добрый вечер'] := 'Добрый вечер!';
  
  FExamples.Values['кто ты'] := 'Я - ваш цифровой помощник';
  FExamples.Values['что ты умеешь'] := 'Отвечать на вопросы и поддерживать беседу';
//  FExamples.Values['сколько времени'] := 'Я не могу определить текущее время';
  FExamples.Values['сколько времени'] := GetCurrentTimeString();
  FExamples.Values['который час'] := GetCurrentTimeString();
  FExamples.Values['время'] := GetCurrentTimeString();

  FExamples.Values['как тебя зовут'] := 'Моё имя - MindAI';
  FExamples.Values['кто тебя создал'] := 'Моя система разработана на Free Pascal';
  FExamples.Values['почему небо голубое'] := 'Из-за рассеяния солнечного света в атмосфере';
  FExamples.Values['что такое паскаль'] := 'Язык программирования высокого уровня';
  FExamples.Values['помоги'] := 'Чем я могу вам помочь?';
  FExamples.Values['расскажи о себе'] := 'Я - ИИ помощник, созданный для демонстрации технологий';
  FExamples.Values['как погода'] := 'У меня нет данных о погоде';
  FExamples.Values['что нового'] := 'Я постоянно учусь и развиваюсь';
  FExamples.Values['сколько будет 2+2'] := 'Два плюс два равно четыре';
  
  // Альтернативные формулировки
  FExamples.Values['твое имя'] := 'Меня зовут MindAI';
  FExamples.Values['как тебя звать'] := 'MindAI - моё имя';
  FExamples.Values['расскажи про себя'] := 'Я - программа искусственного интеллекта';

  FExamples.Values['как тебя зовут'] := 'Меня зовут MindAI';
  FExamples.Values['что ты умеешь'] := 
    'Я могу:'#13#10 +
    '- Отвечать на вопросы'#13#10 +
    '- Показывать текущее время'#13#10 +
    '- Решать математические выражения'#13#10 +
    '- Поддерживать беседу';
end;

procedure TMindEngine.TrainDecisionTree;
var
  X: TDoubleMatrix;
  Y: array of String;
  i: Integer;
begin
  SetLength(X, FExamples.Count);
  SetLength(Y, FExamples.Count);
  
  for i := 0 to FExamples.Count-1 do
  begin
    X[i] := GetFeatures(FExamples.Names[i]);
    Y[i] := FExamples.ValueFromIndex[i];
  end;
  
  FDecisionTree.Train(X, Y);
end;

function TMindEngine.GetContextualResponse(const Input: String): String;
var
  LowerInput: String;
begin
  LowerInput := LowerCase(Input);
  
  if ContainsAny(LowerInput, ['время', 'час', 'часы', 'сколько времени', 'который час']) then
    Exit(GetCurrentTimeString());
    
  if ContainsAny(LowerInput, ['погода', 'дождь', 'солнце', 'температура']) then
    Exit('У меня нет данных о погоде');
    
  if ContainsAny(LowerInput, ['имя', 'зовут', 'называешься', 'твое имя', 'как тебя звать']) then
    Exit('Меня зовут MindAI');
    
  // Добавляем обработку математических вопросов
  if ContainsAny(LowerInput, ['сколько будет', 'вычисли', 'посчитай', '+', '-', '*', '/', '^', 'корень']) then
  begin
    if TMathEvaluator.Evaluate(Input, Result) then
      Exit
    else
      Result := '';
  end;
    
  Result := '';
end;

function TMindEngine.FindBestMatch(const Input: String): String;
var
  i, j: Integer;
  LowerInput: String;
  BestMatch: String;
  BestScore: Integer;
  CurrentScore: Integer;
  InputWords: TStringArray;
  ExampleWords: TStringArray;
begin
  LowerInput := LowerCase(Trim(Input));
  BestMatch := '';
  BestScore := 0;
  
  // Для очень коротких вопросов (1-2 слова) ищем точное совпадение
  if WordCount(LowerInput, [' ']) <= 2 then
  begin
    for i := 0 to FExamples.Count-1 do
    begin
      if LowerInput = LowerCase(FExamples.Names[i]) then
        Exit(FExamples.ValueFromIndex[i]);
    end;
  end;
  
  // Разбиваем входной вопрос на слова
  InputWords := LowerInput.Split([' '], TStringSplitOptions.ExcludeEmpty);
  
  // Ищем максимальное пересечение слов
  for i := 0 to FExamples.Count-1 do
  begin
    ExampleWords := LowerCase(FExamples.Names[i]).Split([' '], TStringSplitOptions.ExcludeEmpty);
    CurrentScore := 0;
    
    // Подсчет совпадающих слов
    for j := 0 to High(InputWords) do
    begin
      if IndexStr(InputWords[j], ExampleWords) >= 0 then
        Inc(CurrentScore);
    end;
    
    // Если нашли лучшее совпадение
    if CurrentScore > BestScore then
    begin
      BestScore := CurrentScore;
      BestMatch := FExamples.ValueFromIndex[i];
    end;
  end;
  
  if BestScore > 0 then
    Result := BestMatch
  else
    Result := '';
end;

function TMindEngine.ProcessInput(const Input: String): TThought;
var
  LowerInput, Response, MathResult, DisplayExpr: String;
begin
  // Приоритетная обработка математики
  if TMathEvaluator.Evaluate(Input, MathResult) then
  begin
    if Pos('Ошибка', MathResult) = 0 then
    begin
      Result.Content := MathResult;
      Result.Confidence := 0.95;
      Result.Source := 'math_engine';
      Exit;
    end;
  end;
{
  if TMathEvaluator.Evaluate(Input, MathResult) then
  begin
    DisplayExpr := StringReplace(Input, 'сколько будет', '', [rfIgnoreCase]);
    DisplayExpr := StringReplace(DisplayExpr, '?', '', [rfReplaceAll]);
    DisplayExpr := Trim(DisplayExpr);

    Result.Content := DisplayExpr + ' = ' + MathResult;
    Result.Confidence := 0.95;
    Result.Source := 'math_engine';
    Exit;
  end;
}
  LowerInput := TStringConverter.RussianLowerCase(Input);

  // 1. Проверка пустого ввода
  if LowerInput = '' then
  begin
    Result.Content := 'Запрос пуст';
    Result.Confidence := 0;
    Result.Source := 'system';
    Exit;
  end;

  // 2. Приоритетная обработка математических выражений
  if TMathEvaluator.Evaluate(Input, MathResult) then
  begin
    if Pos('Ошибка', MathResult) = 0 then
    begin
      Result.Content := MathResult;
      Result.Confidence := 0.95;
      Result.Source := 'math_engine';
      Exit;
    end;
  end;

  // 3. Проверка точных соответствий в базе знаний
  if FKnowledgeBase.IndexOfName(LowerInput) >= 0 then
  begin
    Result.Content := FKnowledgeBase.Values[LowerInput];
    Result.Confidence := 0.95;
    Result.Source := 'knowledge_base';
    Exit;
  end;

  // 4. Контекстные ответы
  Response := GetContextualResponse(LowerInput);
  if Response <> '' then
  begin
    Result.Content := Response;
    Result.Confidence := 0.90;
    Result.Source := 'context';
    Exit;
  end;

  // 5. Поиск наилучшего соответствия
  Response := FindBestMatch(LowerInput);
  if Response <> '' then
  begin
    Result.Content := Response;
    Result.Confidence := 0.85;
    Result.Source := 'best_match';
    Exit;
  end;

  // 6. Фолбэк-ответ
  Result.Content := 'Я не совсем понял ваш вопрос. Можете переформулировать?';
  Result.Confidence := 0.3;
  Result.Source := 'fallback';
end;

procedure TMindEngine.Learn(const Input, Response: String);
var
  LowerInput: String;
begin
  if (Trim(Input) = '') or (Trim(Response) = '') then
    Exit;
    
  LowerInput := LowerCase(Trim(Input));
  
  // Добавляем в базу знаний и примеры
  FKnowledgeBase.Values[LowerInput] := Response;
  FExamples.Values[LowerInput] := Response;
  
  // Переобучаем дерево
  TrainDecisionTree;
end;

function TMindEngine.ProcessMathQuestion(const Input: String): TThought;
var
  MathResult: string;
begin
  if TryCalculateMath(Input, MathResult) then
  begin
    Result.Content := MathResult;
    Result.Confidence := 0.95;
    Result.Source := 'math_engine';
  end
  else
  begin
    Result := ProcessInput(Input); // Фолбэк на обычную обработку
  end;
end;

function TMindEngine.TryCalculateMath(const Question: string; out Answer: string): Boolean;
var
  Expr: string;
begin
  Result := False;
  Expr := Trim(Question);

  // Пропускаем явно не математические вопросы
  if (Pos('=', Expr) > 0) or (Pos('?', Expr) > 0) then
    Exit;

  // Добавляем обработку различных формулировок
  Expr := StringReplace(Expr, 'посчитай', '', [rfIgnoreCase]);
  Expr := StringReplace(Expr, 'вычисли', '', [rfIgnoreCase]);
  Expr := StringReplace(Expr, 'сколько будет', '', [rfIgnoreCase]);
  Expr := StringReplace(Expr, ' ', '', [rfReplaceAll]);

  // Проверяем, содержит ли выражение математические операторы
  if HasMathOperators(Expr) then
    Result := TMathProcessor.Calculate(Expr, Answer);
end;

function TMindEngine.ProcessMathExpression(const Input: String): TThought;
var
  MathResult, DisplayExpr: string;
begin
  if TryCalculateMath(Input, MathResult) then
  begin
    if Pos('Ошибка', MathResult) = 0 then
    begin
      DisplayExpr := ExtractMathExpression(Input);
      if DisplayExpr <> Input then
        Result.Content := DisplayExpr + ' = ' + MathResult
      else
        Result.Content := MathResult;
      Result.Confidence := 0.95;
      Result.Source := 'math_engine';
    end
    else
    begin
      Result.Content := MathResult;
      Result.Confidence := 0.5;
      Result.Source := 'math_error';
    end;
  end
  else
  begin
    Result := ProcessInput(Input);
  end;
end;

end.