unit NeuralChat;
{$MODE OBJFPC}{$H+}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


interface

uses
  SysUtils, Classes, NeuralCore;

type
  TChatModel = record
    BaseNetwork: TNeuralNetwork;
    Vocabulary: TStringList;
    ContextSize: Integer;
  end;

// Инициализация чат-модели
procedure InitChatModel(var Model: TChatModel; 
  const VocabFile: String; ContextSize: Integer = 10);

// Генерация ответа
function GenerateResponse(var Model: TChatModel; 
  const InputText: String): String;

// Обучение на диалогах
procedure TrainChatModel(var Model: TChatModel;
  const Dialogues: array of TStringArray;
  Epochs: Integer; LearningRate: Double);

implementation

uses
  StrUtils;

procedure InitChatModel(var Model: TChatModel; 
  const VocabFile: String; ContextSize: Integer);
begin
  // Загрузка словаря
  Model.Vocabulary := TStringList.Create;
  if FileExists(VocabFile) then
    Model.Vocabulary.LoadFromFile(VocabFile);

  Model.ContextSize := ContextSize;
  
  // Инициализация сети: входной слой учитывает контекст
  InitNetwork(Model.BaseNetwork, 
    [ContextSize * Model.Vocabulary.Count, 256, 128, Model.Vocabulary.Count],
    0.01, 0.001);
end;

function TextToVector(const Model: TChatModel; Text: String): TDoubleArray;
var
  I, Index: Integer;
  Words: TStringArray;
begin
  SetLength(Result, Model.ContextSize * Model.Vocabulary.Count);
  Words := SplitString(LowerCase(Text), ' ');
  
  for I := 0 to Min(High(Words), Model.ContextSize - 1) do
  begin
    Index := Model.Vocabulary.IndexOf(Words[I]);
    if Index >= 0 then
      Result[I * Model.Vocabulary.Count + Index] := 1.0;
  end;
end;

function GenerateResponse(var Model: TChatModel; 
  const InputText: String): String;
var
  InputVector: TDoubleArray;
  Output: TDoubleArray;
  I, MaxIndex: Integer;
  MaxValue: Double;
begin
  InputVector := TextToVector(Model, InputText);
  ForwardPropagation(Model.BaseNetwork, InputVector);
  Output := Model.BaseNetwork.Layers[High(Model.BaseNetwork.Layers)].Output;

  // Выбираем слово с максимальной вероятностью
  MaxValue := -1;
  MaxIndex := 0;
  for I := 0 to High(Output) do
    if Output[I] > MaxValue then
    begin
      MaxValue := Output[I];
      MaxIndex := I;
    end;

  if (MaxIndex >= 0) and (MaxIndex < Model.Vocabulary.Count) then
    Result := Model.Vocabulary[MaxIndex]
  else
    Result := '...';
end;

// ... (реализация TrainChatModel и других методов)

end.