unit NeuralChessCore;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


{$MODE OBJFPC}{$H+}

interface

uses SysUtils, classes, Math, DataUtils, NeuralNetwork, crt;

type
  TPieceType = (ptNone, ptPawn, ptKnight, ptBishop, ptRook, ptQueen, ptKing);
  TPieceColor = (pcWhite, pcBlack);
  TGameState = (gsPlaying, gsWhiteWon, gsBlackWon, gsDraw);
  
  TChessPiece = record
    PieceType: TPieceType;
    Color: TPieceColor;
    Moved: Boolean;
  end;

  TChessBoard = array[0..7, 0..7] of TChessPiece;
  TMove = record
    FromX, FromY, ToX, ToY: Integer;
    Promotion: TPieceType;
    IsEnPassant: Boolean;
    IsCastling: Boolean;
  end;

  TMoveArray = record
    Moves: array[0..255] of TMove;
    Count: Integer;
  end;

  TChessGame = class
  private
    FBoard: TChessBoard;
    FCurrentPlayer: TPieceColor;
    FGameState: TGameState;
    FNetwork: TNeuralNetwork;
    FMoveHistory: array of TMove;
    FEnPassantTarget: record X, Y: Integer end;
    
    procedure InitializeBoard;
    function PieceToChar(piece: TChessPiece): Char;
    function MoveToString(move: TMove): String;
    function IsMoveValid(var move: TMove): Boolean;
    procedure MakeMove(move: TMove);
    function BoardToInputVector: TDoubleArray;
    procedure GetPossibleMoves(var moves: TMoveArray);
    function IsInCheck(color: TPieceColor): Boolean;
    function IsCheckmate(color: TPieceColor): Boolean;
    function IsStalemate(color: TPieceColor): Boolean;
    function EvaluateBoard: Double;
    function Minimax(depth: Integer; alpha, beta: Double; maximizingPlayer: Boolean): Double;
    function FindBestMove: TMove;
    procedure HighlightMoves(x, y: Integer);
  public
//    constructor Create;
constructor Create(loadFromFile: Boolean = False; trainNetwork: Boolean = False);
    destructor Destroy; override;
    procedure PrintBoard;
    procedure HumanMove;
    procedure AIMove;
    procedure Play;
    procedure SaveGame(const filename: String);
    procedure LoadGame(const filename: String);
function TrainOnSingleGame(const gameText: String): Boolean;
procedure TrainOnPGN(const filename: String; epochs: Integer);
  end;

procedure InitBoard(var FBoard: TChessBoard);
procedure ApplyMove(var board: TChessBoard; const move: TMove);

implementation

uses NeuralChessTrain,PGNUtil;

constructor TChessGame.Create(loadFromFile: Boolean = False; trainNetwork: Boolean = False);
const
  WEIGHTS_FILE = 'chess_weights.dat';
  PGN_FILE = 'games.pgn';
begin
  // Инициализация доски
  InitializeBoard;
  FCurrentPlayer := pcWhite;
  FGameState := gsPlaying;
  FEnPassantTarget.X := -1;
  FEnPassantTarget.Y := -1;

WriteLn('Инициализация нейросети');

  // Инициализация нейросети (64 входа, 2 скрытых слоя по 128 нейронов, 64 выхода)
  InitializeNetwork(FNetwork, [64, 128, 128, 64], 0.01, 0.0001);

WriteLn('Загрузка сохранённых весов (если требуется)');

  // Загрузка сохранённых весов (если требуется)
  if loadFromFile and FileExists(WEIGHTS_FILE) then
  begin
    LoadWeights(FNetwork, WEIGHTS_FILE);
    Writeln('Network weights loaded from ', WEIGHTS_FILE);
  end;

WriteLn('Обучение нейросети (если требуется)');

  // Обучение нейросети (если требуется)
  if trainNetwork then
  begin
    if FileExists(PGN_FILE) then
    begin
      Writeln('Training network on ', PGN_FILE, '...');
      TrainOnPGN(PGN_FILE, 3); // 3 эпохи обучения
      Writeln('Training completed!');
      
      // Сохраняем обученные веса
      SaveWeights(FNetwork, WEIGHTS_FILE);
      Writeln('Trained weights saved to ', WEIGHTS_FILE);
    end
    else
    begin
      Writeln('Warning: PGN file not found (', PGN_FILE, '), skipping training');
    end;
  end;
end;

destructor TChessGame.Destroy;
begin
  FreeNetwork(FNetwork);
  inherited;
end;

procedure InitBoard(var FBoard: TChessBoard);
var x, y: byte;
begin
for x := 0 to 7 do begin
case x of
0,7:begin
FBoard[x, 0].PieceType := ptRook;
FBoard[x, 7].PieceType := ptRook;
end;
1,6:begin
FBoard[x, 0].PieceType := ptKnight;
FBoard[x, 7].PieceType := ptKnight;
end;
2,5:begin
FBoard[x, 0].PieceType := ptBishop;
FBoard[x, 7].PieceType := ptBishop;
end;
3:begin
FBoard[x, 0].PieceType := ptQueen;
FBoard[x, 7].PieceType := ptQueen;
end;
4:begin
FBoard[x, 0].PieceType := ptKing;
FBoard[x, 7].PieceType := ptKing;
end;
end; {end select}
FBoard[x, 1].PieceType := ptPawn;
FBoard[x, 6].PieceType := ptPawn;
for y := 0 to 7 do begin
FBoard[x, y].Moved := False;
if y > 5 then FBoard[x, y].Color := pcBlack else FBoard[x, y].Color := pcWhite;
case y of
2..5: FBoard[x, y].PieceType := ptNone;
end; {end select}
end; {next y}
end; {next x}
end;

procedure TChessGame.InitializeBoard;
begin
InitBoard(FBoard);
end;

function TChessGame.IsInCheck(color: TPieceColor): Boolean;
var
  x, y, kingX, kingY: Integer;
  move: TMove;
  opponentMoves: TMoveArray;
begin
  // Найдем позицию короля
  for x := 0 to 7 do
    for y := 0 to 7 do
      if (FBoard[x,y].PieceType = ptKing) and (FBoard[x,y].Color = color) then
      begin
        kingX := x;
        kingY := y;
        Break;
      end;

  // Проверим, есть ли ход противника, который бьет короля
  opponentMoves.Count := 0;
  FCurrentPlayer := TPieceColor(1 - Ord(color));
  GetPossibleMoves(opponentMoves);
  FCurrentPlayer := color;

  for x := 0 to opponentMoves.Count - 1 do
    if (opponentMoves.Moves[x].ToX = kingX) and (opponentMoves.Moves[x].ToY = kingY) then
      Exit(True);

  Result := False;
end;

function TChessGame.IsMoveValid(var move: TMove): Boolean;
var
  piece: TChessPiece;
  testBoard: TChessBoard;
begin
  // Базовые проверки
  if not ((move.FromX in [0..7]) and (move.FromY in [0..7]) and
          (move.ToX in [0..7]) and (move.ToY in [0..7])) then
    Exit(False);

  piece := FBoard[move.FromX, move.FromY];
  if piece.PieceType = ptNone then Exit(False);
  if piece.Color <> FCurrentPlayer then Exit(False);

  // Специальные ходы
  if piece.PieceType = ptPawn then
  begin
    // Взятие на проходе
    if (move.ToX <> move.FromX) and (FBoard[move.ToX, move.ToY].PieceType = ptNone) then
    begin
      if not ((move.FromY = 4) and (move.ToY = 5) and (FEnPassantTarget.X = move.ToX)) then
        Exit(False);
      move.IsEnPassant := True;
    end;
  end;

  // Проверка шаха после хода
  testBoard := FBoard;
  ApplyMove(testBoard, move);
  if IsInCheck(piece.Color) then
    Exit(False);

  Result := True;
end;

function TChessGame.EvaluateBoard: Double;
var
  x, y: Integer;
  piece: TChessPiece;
  score: Double;
begin
  score := 0;
  
  // Материальная оценка
  for x := 0 to 7 do
    for y := 0 to 7 do
    begin
      piece := FBoard[x,y];
      case piece.PieceType of
        ptPawn:   score := score + IfThen(piece.Color = pcWhite, 1, -1);
        ptKnight: score := score + IfThen(piece.Color = pcWhite, 3, -3);
        ptBishop: score := score + IfThen(piece.Color = pcWhite, 3, -3);
        ptRook:   score := score + IfThen(piece.Color = pcWhite, 5, -5);
        ptQueen:  score := score + IfThen(piece.Color = pcWhite, 9, -9);
        ptKing:   score := score + IfThen(piece.Color = pcWhite, 200, -200);
      end;
    end;

  // Позиционная оценка
  // ... (добавить оценку позиции фигур)

  // Оценка активности
  // ... (количество возможных ходов)

  Result := score;
end;

function TChessGame.Minimax(depth: Integer; alpha, beta: Double; maximizingPlayer: Boolean): Double;
var
  moves: TMoveArray;
  i: Integer;
  score: Double;
  oldBoard: TChessBoard;
begin
  if depth = 0 then
    Exit(EvaluateBoard);

  GetPossibleMoves(moves);
  if moves.Count = 0 then
  begin
    if IsInCheck(FCurrentPlayer) then
      Exit(IfThen(maximizingPlayer, -1000, 1000)) // Мат
    else
      Exit(0); // Пат
  end;

  if maximizingPlayer then
  begin
    score := -Infinity;
    for i := 0 to moves.Count - 1 do
    begin
      oldBoard := FBoard;
      MakeMove(moves.Moves[i]);
      score := Max(score, Minimax(depth - 1, alpha, beta, False));
      FBoard := oldBoard;
      
      alpha := Max(alpha, score);
      if beta <= alpha then
        Break;
    end;
    Result := score;
  end
  else
  begin
    score := Infinity;
    for i := 0 to moves.Count - 1 do
    begin
      oldBoard := FBoard;
      MakeMove(moves.Moves[i]);
      score := Min(score, Minimax(depth - 1, alpha, beta, True));
      FBoard := oldBoard;
      
      beta := Min(beta, score);
      if beta <= alpha then
        Break;
    end;
    Result := score;
  end;
end;

function TChessGame.FindBestMove: TMove;
var
  moves: TMoveArray;
  i, bestIndex: Integer;
  score, bestScore: Double;
  oldBoard: TChessBoard;
begin
  GetPossibleMoves(moves);
  if moves.Count = 0 then
    raise Exception.Create('No valid moves');

  bestIndex := 0;
  bestScore := -Infinity;
  
  for i := 0 to moves.Count - 1 do
  begin
    oldBoard := FBoard;
    MakeMove(moves.Moves[i]);
    score := Minimax(3, -Infinity, Infinity, False); // Глубина 3
    FBoard := oldBoard;
    
    if score > bestScore then
    begin
      bestScore := score;
      bestIndex := i;
    end;
  end;
  
  Result := moves.Moves[bestIndex];
end;

procedure TChessGame.AIMove;
var
  bestMove: TMove;
begin
  bestMove := FindBestMove;
  Writeln('AI moves: ', MoveToString(bestMove));
  MakeMove(bestMove);
end;

procedure TChessGame.HighlightMoves(x, y: Integer);
var
  moves: TMoveArray;
  i: Integer;
begin
  if (x < 0) or (x > 7) or (y < 0) or (y > 7) then Exit;
  if FBoard[x,y].PieceType = ptNone then Exit;
  if FBoard[x,y].Color <> FCurrentPlayer then Exit;

  moves.Count := 0;
  GetPossibleMoves(moves);
  
  TextBackground(Black);
  ClrScr;
  PrintBoard;
  
  TextBackground(Green);
  GotoXY(x*2 + 3, 8 - y + 1);
  Write(PieceToChar(FBoard[x,y]));
  
  for i := 0 to moves.Count - 1 do
    if (moves.Moves[i].FromX = x) and (moves.Moves[i].FromY = y) then
    begin
      GotoXY(moves.Moves[i].ToX*2 + 3, 8 - moves.Moves[i].ToY + 1);
      Write(' ');
    end;
end;

procedure TChessGame.Play;
var
  input: String;
  x, y: Integer;
begin
  while FGameState = gsPlaying do
  begin
    ClrScr;
    PrintBoard;
    
    if FCurrentPlayer = pcWhite then
    begin
      Writeln('White to move (or "save" to save game)');
      Write('Select piece (e.g. e2) or command: ');
      Readln(input);

      if input = 'exit' then
      begin
       Halt(0);
      end;
      
      if input = 'save' then
      begin
        SaveGame('chess_save.txt');
        Continue;
      end;
      
      if Length(input) = 2 then
      begin
        x := Ord(LowerCase(input[1])) - Ord('a');
        y := Ord(input[2]) - Ord('1');
        HighlightMoves(x, y);
        Continue;
      end;
      
      HumanMove;
    end
    else
    begin
      AIMove;
    end;
    
    // Проверка конечных состояний игры
    if IsCheckmate(pcBlack) then FGameState := gsWhiteWon
    else if IsCheckmate(pcWhite) then FGameState := gsBlackWon
    else if IsStalemate(FCurrentPlayer) then FGameState := gsDraw;
  end;
  
  case FGameState of
    gsWhiteWon: Writeln('White wins!');
    gsBlackWon: Writeln('Black wins!');
    gsDraw: Writeln('Draw!');
  end;
end;

function TChessGame.PieceToChar(piece: TChessPiece): Char;
begin
  case piece.PieceType of
    ptPawn:   if piece.Color = pcWhite then Result := 'P' else Result := 'p';
    ptKnight: if piece.Color = pcWhite then Result := 'N' else Result := 'n';
    ptBishop: if piece.Color = pcWhite then Result := 'B' else Result := 'b';
    ptRook:   if piece.Color = pcWhite then Result := 'R' else Result := 'r';
    ptQueen:  if piece.Color = pcWhite then Result := 'Q' else Result := 'q';
    ptKing:   if piece.Color = pcWhite then Result := 'K' else Result := 'k';
  else
    Result := '.';
  end;
end;

function TChessGame.MoveToString(move: TMove): String;
var
  promoPiece: TChessPiece;
begin
  Result := Chr(Ord('a') + move.FromX) + IntToStr(move.FromY + 1) + 
            Chr(Ord('a') + move.ToX) + IntToStr(move.ToY + 1);
  if move.Promotion <> ptNone then
  begin
    promoPiece.PieceType := move.Promotion;
    promoPiece.Color := FCurrentPlayer;
    Result := Result + LowerCase(PieceToChar(promoPiece));
  end;
end;

procedure TChessGame.MakeMove(move: TMove);
begin
  if not IsMoveValid(move) then
    raise Exception.Create('Invalid move');

  FBoard[move.ToX, move.ToY] := FBoard[move.FromX, move.FromY];
  FBoard[move.FromX, move.FromY].PieceType := ptNone;

  if (FBoard[move.ToX, move.ToY].PieceType = ptPawn) and 
     ((move.ToY = 0) or (move.ToY = 7)) then
  begin
    if move.Promotion = ptNone then
      FBoard[move.ToX, move.ToY].PieceType := ptQueen
    else
      FBoard[move.ToX, move.ToY].PieceType := move.Promotion;
  end;

  if FCurrentPlayer = pcWhite then
    FCurrentPlayer := pcBlack
  else
    FCurrentPlayer := pcWhite;
end;

function TChessGame.BoardToInputVector: TDoubleArray;
var
  x, y, idx: Integer;
  piece: TChessPiece;
begin
  SetLength(Result, 64);
  idx := 0;
  
  for y := 0 to 7 do
    for x := 0 to 7 do
    begin
      piece := FBoard[x, y];
      if piece.PieceType = ptNone then
        Result[idx] := 0.0
      else if piece.Color = pcWhite then
        Result[idx] := Ord(piece.PieceType)
      else
        Result[idx] := -Ord(piece.PieceType);
      
      Inc(idx);
    end;
end;

procedure TChessGame.GetPossibleMoves(var moves: TMoveArray);
var
  x1, y1, x2, y2: Integer;
  move: TMove;
begin
  moves.Count := 0;
  
  for x1 := 0 to 7 do
    for y1 := 0 to 7 do
    begin
      if (FBoard[x1, y1].PieceType <> ptNone) and (FBoard[x1, y1].Color = FCurrentPlayer) then
      begin
        for x2 := 0 to 7 do
          for y2 := 0 to 7 do
          begin
            move.FromX := x1;
            move.FromY := y1;
            move.ToX := x2;
            move.ToY := y2;
            move.Promotion := ptNone;
            
            if IsMoveValid(move) then
            begin
              if moves.Count >= High(moves.Moves) then
                Exit;
              
              moves.Moves[moves.Count] := move;
              Inc(moves.Count);
            end;
          end;
      end;
    end;
end;

procedure TChessGame.PrintBoard;
var
  x, y: Integer;
begin
  Writeln('  a b c d e f g h');
  for y := 7 downto 0 do
  begin
    Write(y + 1, ' ');
    for x := 0 to 7 do
    begin
      Write(PieceToChar(FBoard[x, y]), ' ');
    end;
    Writeln(y + 1);
  end;
  Writeln('  a b c d e f g h');
end;

procedure TChessGame.HumanMove;
var
  input: String;
  move: TMove;
  promotion: Char;
begin
  repeat
    Write('Your move (e.g. e2e4 or e7e8q): ');
    Readln(input);
    
    if Length(input) < 4 then
    begin
      Writeln('Invalid input. Use format like e2e4');
      Continue;
    end;
    
    move.FromX := Ord(LowerCase(input[1])) - Ord('a');
    move.FromY := Ord(input[2]) - Ord('1');
    move.ToX := Ord(LowerCase(input[3])) - Ord('a');
    move.ToY := Ord(input[4]) - Ord('1');
    move.Promotion := ptNone;
    
    if Length(input) >= 5 then
    begin
      promotion := LowerCase(input[5]);
      case promotion of
        'q': move.Promotion := ptQueen;
        'r': move.Promotion := ptRook;
        'b': move.Promotion := ptBishop;
        'n': move.Promotion := ptKnight;
      end;
    end;
    
    if IsMoveValid(move) then
      Break
    else
      Writeln('Invalid move. Try again.');
  until False;
  
  MakeMove(move);
end;

function TChessGame.IsCheckmate(color: TPieceColor): Boolean;
var
  moves: TMoveArray;
begin
  // Проверяем, находится ли король под шахом и нет ли возможных ходов
  Result := IsInCheck(color);
  if not Result then Exit(False);
  
  // Получаем все возможные ходы для цвета
  FCurrentPlayer := color;
  GetPossibleMoves(moves);
  FCurrentPlayer := TPieceColor(1 - Ord(color)); // Возвращаем текущего игрока
  
  // Если нет допустимых ходов - это мат
  Result := (moves.Count = 0);
end;

function TChessGame.IsStalemate(color: TPieceColor): Boolean;
var
  moves: TMoveArray;
begin
  // Проверяем, не находится ли король под шахом
  if IsInCheck(color) then Exit(False);
  
  // Получаем все возможные ходы для цвета
  FCurrentPlayer := color;
  GetPossibleMoves(moves);
  FCurrentPlayer := TPieceColor(1 - Ord(color)); // Возвращаем текущего игрока
  
  // Если нет допустимых ходов - это пат
  Result := (moves.Count = 0);
end;

procedure TChessGame.SaveGame(const filename: String);
var
  f: TextFile;
  x, y: Integer;
begin
  AssignFile(f, filename);
  Rewrite(f);
  try
    // Сохраняем текущего игрока
    WriteLn(f, Ord(FCurrentPlayer));
    
    // Сохраняем доску
    for y := 0 to 7 do
    begin
      for x := 0 to 7 do
      begin
        Write(f, Ord(FBoard[x,y].PieceType), ' ');
        Write(f, Ord(FBoard[x,y].Color), ' ');
        Write(f, Ord(FBoard[x,y].Moved), ' ');
      end;
      WriteLn(f);
    end;
    
    // Сохраняем историю ходов
    WriteLn(f, Length(FMoveHistory));
    for x := 0 to High(FMoveHistory) do
    begin
      with FMoveHistory[x] do
      begin
        WriteLn(f, FromX, ' ', FromY, ' ', ToX, ' ', ToY, ' ', 
                 Ord(Promotion), ' ', Ord(IsEnPassant), ' ', Ord(IsCastling));
      end;
    end;
  finally
    CloseFile(f);
  end;
  Writeln('Game saved to ', filename);
end;

procedure TChessGame.LoadGame(const filename: String);
var
  f: TextFile;
  x, y, i, count: Integer;
  val: Integer;
begin
  if not FileExists(filename) then
    raise Exception.Create('File not found: ' + filename);

  AssignFile(f, filename);
  Reset(f);
  try
    // Загружаем текущего игрока
    ReadLn(f, val);
    FCurrentPlayer := TPieceColor(val);
    
    // Загружаем доску
    for y := 0 to 7 do
    begin
      for x := 0 to 7 do
      begin
        Read(f, val); FBoard[x,y].PieceType := TPieceType(val);
        Read(f, val); FBoard[x,y].Color := TPieceColor(val);
        Read(f, val); FBoard[x,y].Moved := Boolean(val);
      end;
      ReadLn(f);
    end;
    
    // Загружаем историю ходов
    ReadLn(f, count);
    SetLength(FMoveHistory, count);
    for i := 0 to count - 1 do
    begin
      with FMoveHistory[i] do
      begin
        ReadLn(f, FromX, FromY, ToX, ToY, val); Promotion := TPieceType(val);
        ReadLn(f, val); IsEnPassant := Boolean(val);
        ReadLn(f, val); IsCastling := Boolean(val);
      end;
    end;
    
    // Обновляем состояние игры
    if IsCheckmate(pcWhite) then FGameState := gsBlackWon
    else if IsCheckmate(pcBlack) then FGameState := gsWhiteWon
    else if IsStalemate(FCurrentPlayer) then FGameState := gsDraw
    else FGameState := gsPlaying;
    
  finally
    CloseFile(f);
  end;
  Writeln('Game loaded from ', filename);
end;

procedure ApplyMove(var board: TChessBoard; const move: TMove);
var
  piece: TChessPiece;
begin
  piece := board[move.FromX, move.FromY];

  // Очищаем исходную клетку
  board[move.FromX, move.FromY].PieceType := ptNone;

  // Взятие на проходе
  if move.IsEnPassant then
    board[move.ToX, move.FromY].PieceType := ptNone;

  // Рокировка
  if move.IsCastling then
  begin
    if move.ToX > move.FromX then // Короткая рокировка
    begin
      board[5, move.FromY] := board[7, move.FromY];
      board[7, move.FromY].PieceType := ptNone;
    end
    else // Длинная рокировка
    begin
      board[3, move.FromY] := board[0, move.FromY];
      board[0, move.FromY].PieceType := ptNone;
    end;
  end;

  // Устанавливаем фигуру на новое место
  board[move.ToX, move.ToY] := piece;
  board[move.ToX, move.ToY].Moved := True;

  // Превращение пешки (исправленная строка)
  if (piece.PieceType = ptPawn) and (move.ToY in [0, 7]) then
  begin
    if move.Promotion <> ptNone then
      board[move.ToX, move.ToY].PieceType := move.Promotion
    else
      board[move.ToX, move.ToY].PieceType := ptQueen; // По умолчанию в ферзя
  end;
end;

procedure TChessGame.TrainOnPGN(const filename: String; epochs: Integer);
var
  f: TextFile;
  gameText, line: String;
  games: TStringList;
  i, e, errorCount: Integer;
  success: Boolean;
begin
  if not FileExists(filename) then
    raise Exception.Create('PGN file not found: ' + filename);

  games := TStringList.Create;
  try
    // Чтение PGN файла
    AssignFile(f, filename);
    try
      Reset(f);
      gameText := '';
      while not EOF(f) do
      begin
        ReadLn(f, line);
        if (line = '') or (Pos('[Event', line) = 1) then
        begin
          if gameText <> '' then games.Add(gameText);
          gameText := '';
        end
        else if line[1] <> '[' then
          gameText := gameText + line + ' ';
      end;
      if gameText <> '' then games.Add(gameText);
    finally
      CloseFile(f);
    end;

    // Обучение с защитой
    for e := 1 to epochs do
    begin
      errorCount := 0;
      Writeln('Starting epoch ', e, '/', epochs);
      
      for i := 0 to games.Count - 1 do
      begin
        try
          success := TrainOnSingleGame(games[i]);
          if not success then Inc(errorCount);
          
          if errorCount > 50 then
          begin
            Writeln('Too many errors, reinitializing network...');
            InitializeNetwork(FNetwork, [64, 128, 128, 64], 0.001, 0.0001);
            Break;
          end;
        except
          on E: Exception do
          begin
            Writeln('Error in game ', i, ': ', E.Message);
            Inc(errorCount);
          end;
        end;
      end;
      
      Writeln('Epoch ', e, ' completed with ', errorCount, ' errors');
      SaveWeights(FNetwork, 'weights_epoch_' + IntToStr(e) + '.dat');
    end;
  finally
    games.Free;
  end;
end;

function TChessGame.TrainOnSingleGame(const gameText: String): Boolean;
var
  moves: TStringList;
  board: TChessBoard;
  i: Integer;
  move: TMove;
  input, targetOutput: TDoubleArray;
begin
  Result := False;
  moves := TStringList.Create;
  try
    ExtractMovesFromPGN(gameText, moves);
    InitBoard(board);

    for i := 0 to moves.Count - 1 do
    begin
      if not TryPGNToMove(moves[i], board, move) then Continue;
      
      input := NeuralChessTrain.BoardToInputVector(board);
      targetOutput := GetTargetOutput(board, move);

      // Защита от невалидных данных
      if not ValidateVector(input) or not ValidateVector(targetOutput) then
      begin
        Writeln('Invalid vectors in game at move ', i);
        Exit;
      end;

      try
        ForwardPropagation(FNetwork, input);
        BackwardPropagationForChess(FNetwork, input, targetOutput);
        ApplyMove(board, move);
      except
        on E: Exception do
        begin
          Writeln('Training error: ', E.Message);
          Exit;
        end;
      end;
    end;
    
    Result := True;
  finally
    moves.Free;
  end;
end;

end.