unit NeuralChessTrain;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


{$MODE OBJFPC}{$H+}

interface

uses SysUtils, classes, DataUtils, NeuralNetwork, NeuralChessCore, PGNUtil;

procedure TrainOnPGN(var network: TNeuralNetwork; const filename: String; epochs: Integer);
procedure TrainOnGame(var network: TNeuralNetwork; const gameText: String; epochs: Integer);

procedure LoadWeights(var network: TNeuralNetwork; const filename: String);
procedure SaveWeights(const network: TNeuralNetwork; const filename: String);

procedure BackwardPropagationForChess(
  var network: TNeuralNetwork; 
  const input: TDoubleArray; 
  const targetOutput: TDoubleArray
);

function GetTargetOutput(const board: TChessBoard; const move: TMove): TDoubleArray;

function BoardToInputVector(var FBoard: TChessBoard): TDoubleArray;

implementation

function BoardToInputVector(var FBoard: TChessBoard): TDoubleArray;
var
  x, y, idx: Integer;
  piece: TChessPiece;
begin
  SetLength(Result, 64);
  idx := 0;
  
  for y := 0 to 7 do
    for x := 0 to 7 do
    begin
      piece := FBoard[x, y];
      if piece.PieceType = ptNone then
        Result[idx] := 0.0
      else if piece.Color = pcWhite then
        Result[idx] := Ord(piece.PieceType)
      else
        Result[idx] := -Ord(piece.PieceType);
      
      Inc(idx);
    end;
end;

function GetTargetOutput(const board: TChessBoard; const move: TMove): TDoubleArray;
var
  x, y, idx: Integer;
begin
  // Инициализируем выходной вектор нулями
  SetLength(Result, 64);
  for idx := 0 to 63 do
    Result[idx] := 0.0;

  // Преобразуем конечную позицию хода в индекс выходного вектора
  idx := move.ToY * 8 + move.ToX;
  
  // Устанавливаем целевую вероятность 1.0 для правильного хода
  if (idx >= 0) and (idx < 64) then
    Result[idx] := 1.0
  else
    raise Exception.Create('Invalid move target position');
end;

procedure TrainOnPGN(var network: TNeuralNetwork; const filename: String; epochs: Integer);
var
  pgnFile: TextFile;
  gameText, line: String;
  games: TStringList;
  i: Integer;
begin
  if not FileExists(filename) then
    raise Exception.Create('PGN file not found: ' + filename);

  games := TStringList.Create;
  try
    // Чтение PGN файла и разделение на отдельные игры
    AssignFile(pgnFile, filename);
    Reset(pgnFile);
    gameText := '';
    while not EOF(pgnFile) do
    begin
      ReadLn(pgnFile, line);
      if (line = '') or (Pos('[Event', line) = 1) then
      begin
        if gameText <> '' then
          games.Add(gameText);
        gameText := '';
      end
      else if (line[1] <> '[') then
        gameText := gameText + line + ' ';
    end;
    if gameText <> '' then
      games.Add(gameText);
    CloseFile(pgnFile);

    // Обучение на каждой игре
    for i := 0 to games.Count - 1 do
    begin
      if i mod 100 = 0 then
        WriteLn('Training on game ', i, ' of ', games.Count);
      TrainOnGame(network, games[i], epochs);
    end;
  finally
    games.Free;
  end;
end;

procedure BackwardPropagationForChess(
  var network: TNeuralNetwork; 
  const input: TDoubleArray; 
  const targetOutput: TDoubleArray
);
var
  i, j, k: Integer;
  error, derivative: Double;
  deltas: array of TDoubleArray;
begin
  // Проверка размеров
  if Length(targetOutput) <> Length(network.layers[High(network.layers)].output) then
    raise Exception.Create('Target output size mismatch');

  // Инициализация массива deltas
  SetLength(deltas, Length(network.layers));
  for i := 0 to High(deltas) do
    SetLength(deltas[i], Length(network.layers[i].output));

  // Вычисляем ошибку на выходном слое
  for i := 0 to High(network.layers[High(network.layers)].output) do
  begin
    error := network.layers[High(network.layers)].output[i] - targetOutput[i];
    deltas[High(deltas)][i] := error;
  end;

  // Распространяем ошибку назад
  for i := High(network.layers) downto 1 do
  begin
    for j := 0 to High(network.layers[i].weights) do
    begin
      derivative := 1.0; // Для ReLU/LeakyReLU
      if network.layers[i].output[j] <= 0 then
        derivative := 0.01; // Для LeakyReLU
      
      deltas[i-1][j] := 0.0;
      for k := 0 to High(network.layers[i-1].output) do
      begin
        deltas[i-1][j] := deltas[i-1][j] + 
          network.layers[i].weights[j][k] * deltas[i][k] * derivative;
      end;
    end;
  end;

  // Обновляем веса и смещения
  for i := 0 to High(network.layers) do
  begin
    for j := 0 to High(network.layers[i].weights) do
    begin
      for k := 0 to High(network.layers[i].weights[j]) do
      begin
        network.layers[i].weights[j][k] := network.layers[i].weights[j][k] - 
          network.learningRate * (deltas[i][j] * network.layers[i].output[k] + 
          network.lambda * network.layers[i].weights[j][k]);
      end;
      network.layers[i].biases[j] := network.layers[i].biases[j] - 
        network.learningRate * deltas[i][j];
    end;
  end;
end;

procedure NormalizeVector(var vec: TDoubleArray);
var
  i: Integer;
  maxVal, minVal, range: Double;
begin
  if Length(vec) = 0 then Exit;
  
  maxVal := vec[0];
  minVal := vec[0];
  
  for i := 1 to High(vec) do
  begin
    if vec[i] > maxVal then maxVal := vec[i];
    if vec[i] < minVal then minVal := vec[i];
  end;
  
  range := maxVal - minVal;
  if range = 0 then range := 1; // Защита от деления на ноль
  
  for i := 0 to High(vec) do
    vec[i] := (vec[i] - minVal) / range; // Нормализация к [0, 1]
end;

{
procedure BatchNormalization(var outputs: TDoubleArray);
var
  mean, variance, epsilon: Double;
  i: Integer;
begin
  epsilon := 1e-8;
  mean := Mean(outputs);
  variance := 0;
  
  for i := 0 to High(outputs) do
    variance := variance + Sqr(outputs[i] - mean);
  variance := variance / Length(outputs);
  
  for i := 0 to High(outputs) do
    outputs[i] := (outputs[i] - mean) / sqrt(variance + epsilon);
end;
}

procedure TrainOnGame(var network: TNeuralNetwork; const gameText: String; epochs: Integer);
var
  moves: TStringList;
  board: TChessBoard;
  i, e: Integer;
  input: TDoubleArray;
  move: TMove;
  errorCount: Integer;
begin
  moves := TStringList.Create;
  try
    ExtractMovesFromPGN(gameText, moves);
    if moves.Count = 0 then Exit;

    for e := 1 to epochs do
    begin
      InitBoard(board);
      errorCount := 0;

      for i := 0 to moves.Count - 1 do
      begin
        if TryPGNToMove(moves[i], board, move) then
        begin
          input := BoardToInputVector(board);
          try
            ForwardPropagation(network, input);
            BackwardPropagationForChess(network, input, GetTargetOutput(board, move));
            ApplyMove(board, move);
          except
            on E: Exception do
            begin
              Inc(errorCount);
              if errorCount > 10 then
              begin
                InitializeNetwork(network, [64, 128, 128, 64], 0.01, 0.0001);
                Break;
              end;
            end;
          end;
        end;
      end;
      
      if errorCount > 0 then
        Writeln('Epoch ', e, ': ', errorCount, ' errors');
    end;
  finally
    moves.Free;
  end;
end;

procedure SaveWeights(const network: TNeuralNetwork; const filename: String);
var
  f: File;
  i, j, k: Integer;
  numLayers: Integer;
begin
  AssignFile(f, filename);
  Rewrite(f, 1);
  try
    // Сохраняем количество слоев
    numLayers := Length(network.layers);
    BlockWrite(f, numLayers, SizeOf(numLayers));
    
    // Сохраняем параметры каждого слоя
    for i := 0 to High(network.layers) do
    begin
      // Сохраняем размеры весов
      j := Length(network.layers[i].weights);
      k := Length(network.layers[i].weights[0]);
      BlockWrite(f, j, SizeOf(j));
      BlockWrite(f, k, SizeOf(k));
      
      // Сохраняем веса
      for j := 0 to High(network.layers[i].weights) do
        BlockWrite(f, network.layers[i].weights[j][0], SizeOf(Double) * Length(network.layers[i].weights[j]));
      
      // Сохраняем смещения
      BlockWrite(f, network.layers[i].biases[0], SizeOf(Double) * Length(network.layers[i].biases));
    end;
    
    // Сохраняем параметры обучения
    BlockWrite(f, network.learningRate, SizeOf(network.learningRate));
    BlockWrite(f, network.lambda, SizeOf(network.lambda));
  finally
    CloseFile(f);
  end;
end;

procedure LoadWeights(var network: TNeuralNetwork; const filename: String);
var
  f: File;
  i, j, k, l: Integer;
  numLayers: Integer;
  sizeIn, sizeOut: Integer;
begin
  if not FileExists(filename) then
    raise Exception.Create('Weights file not found: ' + filename);

  AssignFile(f, filename);
  Reset(f, 1);
  try
    // Загружаем количество слоев
    BlockRead(f, numLayers, SizeOf(numLayers));
    if numLayers <> Length(network.layers) then
      raise Exception.Create('Network architecture mismatch');
    
    // Загружаем параметры каждого слоя
    for i := 0 to High(network.layers) do
    begin
      // Проверяем размеры весов
      BlockRead(f, sizeOut, SizeOf(sizeOut));
      BlockRead(f, sizeIn, SizeOf(sizeIn));
      
      if (sizeOut <> Length(network.layers[i].weights)) or
         (sizeIn <> Length(network.layers[i].weights[0])) then
        raise Exception.Create('Layer size mismatch at layer ' + IntToStr(i));
      
      // Загружаем веса
      for j := 0 to High(network.layers[i].weights) do
        BlockRead(f, network.layers[i].weights[j][0], SizeOf(Double) * Length(network.layers[i].weights[j]));
      
      // Загружаем смещения
      BlockRead(f, network.layers[i].biases[0], SizeOf(Double) * Length(network.layers[i].biases));
    end;
    
    // Загружаем параметры обучения
    BlockRead(f, network.learningRate, SizeOf(network.learningRate));
    BlockRead(f, network.lambda, SizeOf(network.lambda));
  finally
    CloseFile(f);
  end;
end;

end.