unit NeuralNetwork;
{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


interface

uses
  SysUtils, Math, DataUtils;

const
  MAX_EXPERIENCE = 1000;
  MIN_LEARNING_RATE = 0.0001;
  INIT_LEARNING_RATE = 0.1;

type
  TLayer = record
    weights: TDoubleMatrix; // Веса слоя
    biases: TDoubleArray;   // Смещения (bias)
    output: TDoubleArray;   // Выходные значения нейронов
  end;

  TNeuralNetwork = record
    layers: array of TLayer; // Слои сети
    learningRate: Double;    // Скорость обучения
    lambda: Double;          // Параметр L2-регуляризации
  end;

procedure InitializeNetwork(var network: TNeuralNetwork; const layerSizes: array of Integer; learningRate: Double; lambda: Double = 0.0);
procedure ForwardPropagation(var network: TNeuralNetwork; const input: TDoubleArray);
procedure BackwardPropagation(var network: TNeuralNetwork; const input: TDoubleArray; target: Double); // Изменён тип target на Double
procedure TrainNetwork(var network: TNeuralNetwork; 
                      x: TDoubleMatrix;
                      y: TDoubleArray;
                      epochs: Integer; 
                      experience: Integer);
function PredictNetwork(const network: TNeuralNetwork; const input: TDoubleArray): TDoubleArray;
procedure FreeNetwork(var network: TNeuralNetwork);
procedure SafeInitializeNetwork(var network: TNeuralNetwork;  const layers: array of Integer; learningRate, lambda: Double);
function ValidateVector(const vec: TDoubleArray): Boolean;

function ReLU(x: Double): Double;
function LeakyReLU(x: Double): Double;
function Softmax(const x: TDoubleArray): TDoubleArray;
function SafeSoftmax(const x: TDoubleArray): TDoubleArray;
function SafeExp(x: Double): Double;
function SafeSigmoid(x: Double): Double;

implementation

function ReLU(x: Double): Double;
begin
  if x > 0 then
    Result := x
  else
    Result := 0;
end;

function LeakyReLU(x: Double): Double;
begin
  if x > 0 then
    Result := x
  else
    Result := 0.01 * x; // Малый наклон для отрицательных значений
end;

// Функция для генерации массива случайных чисел
function RandomArray(size: Integer; minVal, maxVal: Double): TDoubleArray;
var
  i: Integer;
begin
  SetLength(Result, size); // Явная инициализация массива
  for i := 0 to High(Result) do
    Result[i] := minVal + (maxVal - minVal) * Random;
end;

// Softmax
function Softmax(const x: TDoubleArray): TDoubleArray;
var
  i: Integer;
  maxVal, sum: Double;
begin
  SetLength(Result, Length(x));
  
  // Находим максимальное значение для численной стабильности
  maxVal := x[0];
  for i := 1 to High(x) do
    if x[i] > maxVal then
      maxVal := x[i];
  
  // Вычисляем экспоненты и сумму
  sum := 0.0;
  for i := 0 to High(x) do
  begin
    Result[i] := Exp(x[i] - maxVal); // Сдвиг для стабильности
    sum := sum + Result[i];
  end;
  
  // Нормализуем
  if sum > 0 then
  begin
    for i := 0 to High(Result) do
      Result[i] := Result[i] / sum;
  end
  else
  begin
    // Защита от деления на ноль
    for i := 0 to High(Result) do
      Result[i] := 1.0 / Length(Result);
  end;
end;

// Инициализация сети
procedure InitializeNetwork(var network: TNeuralNetwork; const layerSizes: array of Integer; 
                          learningRate: Double; lambda: Double = 0.0);
var
  i, j: Integer;
  fan_in, fan_out: Integer;
  limit: Double;
begin
  network.learningRate := learningRate;
  network.lambda := lambda;
  SetLength(network.layers, Length(layerSizes) - 1);

  for i := 0 to High(network.layers) do
  begin
    // Инициализация Xavier/Glorot для лучшего обучения
    fan_in := layerSizes[i];
    fan_out := layerSizes[i+1];
    limit := sqrt(6.0 / (fan_in + fan_out));

    SetLength(network.layers[i].weights, layerSizes[i+1], layerSizes[i]);
    SetLength(network.layers[i].biases, layerSizes[i+1]);
    SetLength(network.layers[i].output, layerSizes[i+1]);

    for j := 0 to High(network.layers[i].weights) do
    begin
      network.layers[i].weights[j] := RandomArray(layerSizes[i], -limit, limit);
      network.layers[i].biases[j] := -limit + 2 * limit * Random;
    end;
  end;
end;

function SafeSoftmax(const x: TDoubleArray): TDoubleArray;
var
  i: Integer;
  maxVal, sum: Double;
begin
  SetLength(Result, Length(x));
  
  // Находим максимум для численной стабильности
  maxVal := MaxValue(x);
  
  // Вычисляем экспоненты с защитой
  sum := 0.0;
  for i := 0 to High(x) do
  begin
    Result[i] := Exp(x[i] - maxVal);
    if IsNan(Result[i]) then Result[i] := 0;
    sum := sum + Result[i];
  end;
  
  // Нормализация с защитой
  if sum <= 0 then sum := 1;
  for i := 0 to High(Result) do
    Result[i] := Result[i] / sum;
end;

// Прямое распространение (Forward Propagation)
procedure ForwardPropagation(var network: TNeuralNetwork; const input: TDoubleArray);
var
  i, j, k: Integer;
  sum: Double;
begin
  // Проверка входных данных
  if Length(input) = 0 then
    raise Exception.Create('Input array is empty');

  // Проверка соответствия размеров
  if Length(input) <> Length(network.layers[0].weights[0]) then
    raise Exception.Create(Format(
      'Input size mismatch. Expected %d, got %d',
      [Length(network.layers[0].weights[0]), Length(input)]));

  // Обработка слоёв с дополнительными проверками
  for i := 0 to High(network.layers) do
  begin
    for j := 0 to High(network.layers[i].weights) do
    begin
      sum := 0.0;
      
      // Проверка индексов перед доступом
      if (i > 0) and (Length(network.layers[i-1].output) <= High(network.layers[i].weights[j])) then
        raise Exception.Create(Format(
          'Index out of bounds in layer %d, neuron %d: weights length %d, previous layer output %d',
          [i, j, Length(network.layers[i].weights[j]), Length(network.layers[i-1].output)]));
      
      for k := 0 to High(network.layers[i].weights[j]) do
      begin
        // Проверка весов
        if IsNan(network.layers[i].weights[j][k]) then 
          network.layers[i].weights[j][k] := 0;
          
        // Безопасный доступ к входным данным
        if i = 0 then
        begin
          if k > High(input) then
            raise Exception.Create(Format(
              'Input index out of bounds: %d (input size %d)',
              [k, Length(input)]));
          sum := sum + network.layers[i].weights[j][k] * input[k];
        end
        else
        begin
          if k > High(network.layers[i-1].output) then
            raise Exception.Create(Format(
              'Previous layer output index out of bounds: %d (output size %d)',
              [k, Length(network.layers[i-1].output)]));
          sum := sum + network.layers[i].weights[j][k] * network.layers[i-1].output[k];
        end;
      end;
      
      sum := sum + network.layers[i].biases[j];
      
      // Активация
      if i < High(network.layers) then
        network.layers[i].output[j] := LeakyReLU(sum)
      else
        network.layers[i].output[j] := sum; // Для выходного слоя
    end;

    // Softmax только для выходного слоя
    if i = High(network.layers) then
      network.layers[i].output := SafeSoftmax(network.layers[i].output);
  end;
end;

// Обратное распространение (Backward Propagation)
procedure BackwardPropagation(var network: TNeuralNetwork; const input: TDoubleArray; target: Double);
var
  i, j, k: Integer;
  error, derivative: Double;
  deltas: array of TDoubleArray;
begin
  // Инициализация массива deltas
  SetLength(deltas, Length(network.layers));
  for i := 0 to High(deltas) do
    SetLength(deltas[i], Length(network.layers[i].output));

  // Вычисляем ошибку на выходном слое
  for i := 0 to High(network.layers[High(network.layers)].output) do
  begin
    error := network.layers[High(network.layers)].output[i] - target;
    deltas[High(deltas)][i] := error;
  end;

  // Распространяем ошибку назад
  for i := High(network.layers) downto 1 do
  begin
    for j := 0 to High(network.layers[i].weights) do
    begin
      derivative := 1.0 - Sqr(network.layers[i].output[j]); // Производная Tanh
      deltas[i - 1][j] := 0.0;
      for k := 0 to High(network.layers[i - 1].output) do
      begin
        // Проверка, чтобы не выйти за пределы массива
        if (k <= High(deltas[i])) and (k <= High(network.layers[i].weights[j])) then
          deltas[i - 1][j] := deltas[i - 1][j] + network.layers[i].weights[j][k] * deltas[i][k] * derivative;
      end;
    end;
  end;

  // Обновляем веса и смещения с учётом L2-регуляризации
  for i := 0 to High(network.layers) do
  begin
    for j := 0 to High(network.layers[i].weights) do
    begin
      for k := 0 to High(network.layers[i].weights[j]) do
      begin
        // Проверка, чтобы не выйти за пределы массива
        if (k <= High(network.layers[i].output)) then
        begin
          network.layers[i].weights[j][k] := network.layers[i].weights[j][k] - network.learningRate * (deltas[i][j] * network.layers[i].output[k] + network.lambda * network.layers[i].weights[j][k]);
        end;
      end;
      network.layers[i].biases[j] := network.layers[i].biases[j] - network.learningRate * deltas[i][j];
    end;
  end;
end;

// Обучение сети
// Модифицированная процедура обучения с защитой от переполнения
procedure TrainNetwork(var network: TNeuralNetwork; 
                      x: TDoubleMatrix; // Убрали const
                      y: TDoubleArray;  // Убрали const
                      epochs: Integer; 
                      experience: Integer);
var
  i, j, k: Integer;
  effectiveLR: Double;
  localX: TDoubleMatrix; // Локальная копия для обучения
  localY: TDoubleArray;  // Локальная копия меток
begin
  if Length(x) = 0 then Exit;
  
  // Создаем локальные копии данных для обучения
  localX := CopyMatrix(x);
  localY := CopyArray(y);
  
  // Ограничиваем размер обучающей выборки
  if Length(localX) > 50 then
  begin
    SetLength(localX, 50);
    SetLength(localY, 50);
  end;
  
  // Адаптивный learning rate
  effectiveLR := INIT_LEARNING_RATE * (1.0 - experience/MAX_EXPERIENCE);
  network.learningRate := Max(effectiveLR, MIN_LEARNING_RATE);
  
  // Нормализация входных данных
  for i := 0 to High(localX) do
    for j := 0 to High(localX[i]) do
      localX[i][j] := Max(-1.0, Min(1.0, localX[i][j]));

  // Остальной код обучения остается прежним, но работаем с localX и localY
  for i := 1 to Min(epochs, 10 + experience div 20) do
  begin
    for j := 0 to High(localX) do
    begin
      try
        ForwardPropagation(network, localX[j]);
        BackwardPropagation(network, localX[j], localY[j]);
        
        // Ограничение весов
        for k := 0 to High(network.layers) do
        begin
          network.layers[k].weights := 
            ClipMatrix(network.layers[k].weights, -5.0, 5.0);
          network.layers[k].biases := 
            ClipVector(network.layers[k].biases, -5.0, 5.0);
        end;
      except
        on E: Exception do
          WriteLn('Обучение пропущено: ', E.Message);
      end;
    end;
  end;
end;

// Предсказание
function PredictNetwork(const network: TNeuralNetwork; const input: TDoubleArray): TDoubleArray;
var
  networkCopy: TNeuralNetwork;
begin
  // Создаём копию сети, чтобы не изменять оригинальную
  networkCopy := network;
  ForwardPropagation(networkCopy, input);
  Result := networkCopy.layers[High(networkCopy.layers)].output;
end;

// Освобождение памяти
procedure FreeNetwork(var network: TNeuralNetwork);
var
  i: Integer;
begin
  for i := 0 to High(network.layers) do
  begin
    SetLength(network.layers[i].weights, 0);
    SetLength(network.layers[i].biases, 0);
    SetLength(network.layers[i].output, 0);
  end;
  SetLength(network.layers, 0);
end;

function ValidateVector(const vec: TDoubleArray): Boolean;
var
  i: Integer;
begin
  Result := False;
  if Length(vec) = 0 then Exit;
  
  for i := 0 to High(vec) do
  begin
    if IsNan(vec[i]) or IsInfinite(vec[i]) then Exit;
    if (vec[i] > 1e10) or (vec[i] < -1e10) then Exit;
  end;
  
  Result := True;
end;

procedure SafeInitializeNetwork(var network: TNeuralNetwork; 
  const layers: array of Integer; learningRate, lambda: Double);
var
  i: Integer;
begin
  if Length(layers) < 2 then
    raise Exception.Create('Network must have at least 2 layers');
    
  for i := 0 to High(layers) do
  begin
    if layers[i] <= 0 then
      raise Exception.Create(Format(
        'Invalid layer size %d at position %d', [layers[i], i]));
  end;
  
  InitializeNetwork(network, layers, learningRate, lambda);
end;

function SafeExp(x: Double): Double;
begin
  if x > 50 then Exit(Exp(50));
  if x < -50 then Exit(Exp(-50));
  Result := Exp(x);
end;

function SafeSigmoid(x: Double): Double;
begin
  if x > 30 then Exit(1.0);
  if x < -30 then Exit(0.0);
  try
    Result := 1.0 / (1.0 + SafeExp(-x));
  except
    Result := 0.5;
  end;
end;

end.