unit NeuralNetworkIntegration;
{$MODE OBJFPC}{$H+}{$RANGECHECKS ON}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


interface

uses
  SysUtils, Classes, PostProcessor, DataUtils, NeuralNetwork, LSTM, LazUTF8, 
  ucs4unit, ucs4opunit, ucs4functionsunit, Math, MatrixOps, TextDecoder;

type
  TTextProcessingResult = record
    Category: Integer;
    Confidence: Double;
    Features: TDoubleArray;
  end;

  TTokenMapEntry = record
    Token: string;
    Index: Integer;
  end;
  
  TTokenMap = array of TTokenMapEntry;

  TIntArray = array of Integer;
  TFloatArray = array of Single;

procedure InitializeTextClassifier(var network: TNeuralNetwork);
function ProcessTextToFeatures(const text: string): TDoubleArray;
function ClassifyText(var network: TNeuralNetwork; const text: string): TTextProcessingResult;
procedure TrainTextClassifier(var network: TNeuralNetwork; 
  const samples: array of string; const labels: array of Integer);
function CategoryToStr(category: Integer): string;
function ArgMax(const Values: TDoubleArray; ConfidenceThreshold: Double = 0.5): Integer;
function FindTokenIndex(const TokenMap: TTokenMap; const Token: string): Integer;
procedure AddTokenToMap(var TokenMap: TTokenMap; const Token: string; var NextIndex: Integer);
procedure BuildTokenMap(const Texts: TStringArray; var TokenMap: TTokenMap);
function TextToSequence(const Text: string; const TokenMap: TTokenMap): TIntArray;
function TextToSequence(const Text: ucs4; const TokenMap: TTokenMap): TIntArray;
function TextToMatrix(const text: string; SequenceLength: Integer): TDoubleMatrix;
function TextToMatrix(const text: ucs4; SequenceLength: Integer): TDoubleMatrix;

procedure InitializeLSTMClassifier(var network: TLSTMLayer; var denseLayers: TNeuralNetwork);
function LSTMClassifyText(var network: TLSTMLayer; var denseLayers: TNeuralNetwork; 
  const text: string): TTextProcessingResult;

function ProcessTransformerOutput(const output: TDoubleMatrix): string;

implementation

procedure InitializeTextClassifier(var network: TNeuralNetwork);
const
  LAYERS: array of Integer = (258, 64, 3);
  LEARNING_RATE = 0.01;
  LAMBDA = 0.0001;
begin
  SafeInitializeNetwork(network, LAYERS, LEARNING_RATE, LAMBDA);
end;

function ProcessTextToFeatures(const text: string): TDoubleArray;
var
  i, charCode: Integer;
  u8Char: string;
  charCounts: array[0..255] of Integer;
  totalChars: Integer;
  hasCyrillic, hasLatin: Boolean;
begin
  // Инициализация
  FillChar(charCounts, SizeOf(charCounts), 0);
  totalChars := 0;
  
  // Подсчет символов
  for i := 1 to UTF8Length(text) do
  begin
    u8Char := UTF8Copy(text, i, 1);
    charCode := Ord(u8Char[1]) mod 256;
    Inc(charCounts[charCode]);
    Inc(totalChars);
  end;
  
  // Создание вектора признаков
  SetLength(Result, 258); // 256 символов + 2 дополнительных признака
  if totalChars > 0 then
  begin
    for i := 0 to 255 do
      Result[i] := charCounts[i] / totalChars;
  end
  else
  begin
    for i := 0 to 255 do
      Result[i] := 0;
  end;

  // Дополнительные признаки
  hasCyrillic := False;
  hasLatin := False;
  for i := 1 to UTF8Length(text) do
  begin
    u8Char := UTF8Copy(text, i, 1);
    case UTF8CharToUCS4(u8Char) of
      $0400..$04FF: hasCyrillic := True;
      $0041..$007A: hasLatin := True;
    end;
  end;

  Result[256] := Ord(hasCyrillic);
  Result[257] := Ord(hasLatin);
  
  // Проверка размера результата
  if Length(Result) <> 258 then
    raise Exception.Create('Feature vector size must be 258');
end;

function ClassifyText(var network: TNeuralNetwork; const text: string): TTextProcessingResult;
var
  features: TDoubleArray;
  output: TDoubleArray;
  i, bestCategory: Integer;
  maxConfidence: Double;
begin
  // Преобразование текста в числовые признаки
  features := ProcessTextToFeatures(text);
  
  // Классификация
  output := PredictNetwork(network, features);
  
  // Находим категорию с наибольшей уверенностью
  bestCategory := 0;
  maxConfidence := output[0];
  for i := 1 to High(output) do
  begin
    if output[i] > maxConfidence then
    begin
      maxConfidence := output[i];
      bestCategory := i;
    end;
  end;
  
  // Возвращаем результат
  Result.Category := bestCategory;
  Result.Confidence := maxConfidence;
  Result.Features := features;
end;

procedure TrainTextClassifier(var network: TNeuralNetwork; 
  const samples: array of string; const labels: array of Integer);
var
  i: Integer;
  x: TDoubleMatrix;
  y: TDoubleArray;
begin
  if Length(samples) <> Length(labels) then
    raise Exception.Create('Samples and labels count mismatch');
  
  // Подготовка данных
  SetLength(x, Length(samples));
  SetLength(y, Length(samples));
  
  for i := 0 to High(samples) do
  begin
    x[i] := ProcessTextToFeatures(samples[i]);
    y[i] := labels[i];
  end;
  
  // Нормализация данных
  NormalizeData(x);

  // Отладочный вывод
  WriteLn('=== Training Info ===');
  WriteLn('Samples: ', Length(x));
  WriteLn('Features per sample: ', Length(x[0]));
  
  // Обучение сети с experience=0 (по умолчанию)
  TrainNetwork(network, x, y, 100, 0); // Добавлен параметр experience
  
  WriteLn('Training completed. Network architecture:');
  WriteLn('Input layer: ', Length(network.layers[0].weights[0]), ' neurons');
  for i := 0 to High(network.layers) do
    WriteLn('Layer ', i+1, ': ', Length(network.layers[i].weights), ' neurons');
end;

function CategoryToStr(category: Integer): string;
begin
  case category of
    0: Result := 'Русский текст';
    1: Result := 'Английский текст';
    2: Result := 'Другое/Символы';
    else Result := 'Неизвестно';
  end;
end;

procedure InitializeLSTMClassifier(var network: TLSTMLayer; var denseLayers: TNeuralNetwork);
begin
  // LSTM слой
  InitializeLSTMLayer(network, 256, 128); // 256 входов, 128 скрытых нейронов
  
  // Полносвязные слои
  SafeInitializeNetwork(denseLayers, [128, 64, 3], 0.01, 0.0001);
end;

function LSTMClassifyText(var network: TLSTMLayer; var denseLayers: TNeuralNetwork; 
  const text: string): TTextProcessingResult;
var
  features: TDoubleMatrix;
  lstmOutput: TDoubleArray;
begin
  // Преобразование текста в последовательность
  features := TextToMatrix(text, 50); // Фиксированная длина последовательности 50
  
  // Прямой проход через LSTM
  LSTMLayerForward(network, features);
  
  // Берем последний hidden state
  lstmOutput := network.Cells[High(network.Cells)].h;
  
  // Прямой проход через полносвязные слои
  ForwardPropagation(denseLayers, lstmOutput);
  
  // Обработка результата
  Result.Category := ArgMax(denseLayers.layers[High(denseLayers.layers)].output);
  Result.Confidence := MaxValue(denseLayers.layers[High(denseLayers.layers)].output);
end;

function FindTokenIndex(const TokenMap: TTokenMap; const Token: string): Integer;
var
  i: Integer;
begin
  for i := 0 to High(TokenMap) do
    if TokenMap[i].Token = Token then
      Exit(TokenMap[i].Index);
  Result := -1; // Not found
end;

procedure AddTokenToMap(var TokenMap: TTokenMap; const Token: string; var NextIndex: Integer);
begin
  if FindTokenIndex(TokenMap, Token) = -1 then
  begin
    SetLength(TokenMap, Length(TokenMap) + 1);
    TokenMap[High(TokenMap)].Token := Token;
    TokenMap[High(TokenMap)].Index := NextIndex;
    Inc(NextIndex);
  end;
end;

function ArgMax(const Values: TDoubleArray; ConfidenceThreshold: Double = 0.5): Integer;
var
  i: Integer;
  MaxValue: Double;
begin
  if Length(Values) = 0 then
    Exit(-1);

  Result := 0;
  MaxValue := Values[0];
  
  for i := 1 to High(Values) do
    if Values[i] > MaxValue then
    begin
      MaxValue := Values[i];
      Result := i;
    end;
    
  // Проверка порога уверенности
  if MaxValue < ConfidenceThreshold then
    Result := -1; // Низкая уверенность
end;


procedure BuildTokenMap(const Texts: TStringArray; var TokenMap: TTokenMap);
var
  Text, Token: string;
  Tokens: TStringArray;
  NextIndex: Integer;
  i: Integer;
begin
  TokenMap := nil;
  NextIndex := 1; // 0 обычно резервируется для неизвестных слов
  
  for Text in Texts do
  begin
    Tokens := Text.Split([' '], TStringSplitOptions.ExcludeEmpty);
    for Token in Tokens do
    begin
      AddTokenToMap(TokenMap, LowerCase(Token), NextIndex);
    end;
  end;
end;

// Преобразование текста в последовательность чисел (индексов токенов)

function TextToSequence(const Text: string; const TokenMap: TTokenMap): TIntArray;
var
  Tokens: TStringArray;
  i, Index: Integer;
begin
  Tokens := Text.Split([' '], TStringSplitOptions.ExcludeEmpty);
  SetLength(Result, Length(Tokens));
  
  for i := 0 to High(Tokens) do
  begin
    Index := FindTokenIndex(TokenMap, LowerCase(Tokens[i]));
    if Index = -1 then
      Result[i] := 0 // UNKNOWN_TOKEN индекс
    else
      Result[i] := Index;
  end;
end;

function TextToSequence(const Text: ucs4; const TokenMap: TTokenMap): TIntArray;
var
  Tokens: TUC4Array;
  i, Index: Integer;
  TokenStr: string;
begin
  Tokens := TokenizeForNLP(Text);
  SetLength(Result, Length(Tokens));
  
  for i := 0 to High(Tokens) do
  begin
    TokenStr := Tokens[i].ToUTF8();
    Index := FindTokenIndex(TokenMap, LowerCase(TokenStr));
    if Index = -1 then
      Result[i] := 0 // UNKNOWN_TOKEN
    else
      Result[i] := Index;
  end;
end;

function TextToMatrix(const text: string; SequenceLength: Integer): TDoubleMatrix;
var
  ucs4Text: ucs4;
begin
  ucs4Text.Init;
  ucs4Text.FromUTF8(text);
  Result := TextToMatrix(ucs4Text, SequenceLength);
  ucs4Text.Clear;
end;

function TextToMatrix(const text: ucs4; SequenceLength: Integer): TDoubleMatrix;
var
  i: Integer;
  normVal: Double;
begin
  // Инициализируем матрицу фиксированного размера
  SetLength(Result, SequenceLength);
  
  for i := 0 to SequenceLength - 1 do
  begin
    SetLength(Result[i], 1); // Один признак на символ
    
    if i < text.Length then
    begin
      // Нормализация Unicode кода к диапазону [0, 1]
      // Для лучшего представления можно использовать log(1 + code)
      normVal := Ln(1 + text[i]) / Ln(1 + $10FFFF); // Максимальный Unicode код
      Result[i][0] := normVal;
    end
    else
    begin
      // Padding нулями
      Result[i][0] := 0.0;
    end;
  end;
end;

function ProcessTransformerOutput(const output: TDoubleMatrix): string;
var
  i: Integer;
  maxIdx: Integer;
  maxVal: Double;
begin
  if Length(output) = 0 then
    Exit('Извините, не могу обработать ответ');

  // Простейшая реализация - берем первый вектор последовательности
  if Length(output[0]) = 0 then
    Exit('Пустой ответ');

  // Находим максимальное значение в выходном векторе
  maxVal := output[0][0];
  maxIdx := 0;
  for i := 1 to High(output[0]) do
  begin
    if output[0][i] > maxVal then
    begin
      maxVal := output[0][i];
      maxIdx := i;
    end;
  end;

  // Простейший шаблонный ответ
  case maxIdx of
    0: Result := 'Привет! Как я могу помочь?';
    1: Result := 'Здравствуйте! Чем могу быть полезен?';
    2: Result := 'Добрый день! Что вас интересует?';
    else
      Result := 'Я вас слушаю...';
  end;
end;

// Улучшенная версия с использованием декодера
{
function ProcessTransformerOutput(const output: TDoubleMatrix; var ResponseDecoder: TTextDecoder): string;
var
  probs: TDoubleArray;
  i: Integer;
begin
  if (Length(output) = 0) or (Length(output[0]) = 0) then
    Exit('Извините, возникла ошибка обработки');

  // Преобразуем выход трансформера в вероятности
  SetLength(probs, Length(output[0]));
  for i := 0 to High(probs) do
    probs[i] := Exp(output[0][i]);
  
  // Нормализуем
  MatrixOps.NormalizeVector(probs);

  // Используем декодер для генерации ответа
  Result := TextDecoder.GenerateResponse(ResponseDecoder, probs);
end;
}

end.