unit NeuralPoetry;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


{$MODE OBJFPC}{$H+}

interface

uses
  Classes, SysUtils, Math, contnrs;

type
  TNeuralLayer = array of array of Single;
  TVector = array of Single;

  TNeuralPoetryGenerator = class
  private
    // Параметры модели
    FEmbeddingSize: Integer;
    FHiddenSize: Integer;
    FVocabSize: Integer;
    
    // Веса модели
    FEmbeddings: TNeuralLayer;      // Матрица эмбеддингов
    FLSTMWeights: array[0..3] of TNeuralLayer; // Входные веса LSTM
    FLSTMUWeights: array[0..3] of TNeuralLayer; // Рекуррентные веса LSTM
    FDenseWeights: TNeuralLayer;    // Веса плотного слоя
    
    // Словарь
    FVocabulary: TStringList;
    FWordToIndex: TFPHashObjectList;
    
    procedure InitializeModel;
    procedure InitializeVocabulary;
    function LSTMStep(const x, h_prev, c_prev: TVector; 
                     var h_next, c_next: TVector): TVector;
    function Softmax(const x: TVector): TVector;
    function SampleWord(const probs: TVector): string;
    
  public
    constructor Create;
    destructor Destroy; override;
    function GeneratePoem(SeedText: string; Style: Integer; PoemLength: Integer = 8): string;
    procedure TrainOnText(const Text: string);
  end;

implementation

constructor TNeuralPoetryGenerator.Create;
begin
  inherited;
  
  // Параметры модели
  FEmbeddingSize := 32;
  FHiddenSize := 64;
  
  // Инициализация словаря
  FVocabulary := TStringList.Create;
  FWordToIndex := TFPHashObjectList.Create(True);
  InitializeVocabulary;
  
  FVocabSize := FVocabulary.Count;
  InitializeModel;
end;

destructor TNeuralPoetryGenerator.Destroy;
begin
  // Удаляем словарь перед освобождением списка
  FWordToIndex.OwnsObjects := False; // Добавьте эту строку
  FVocabulary.Free;
  FWordToIndex.Free;
  inherited;
end;

procedure TNeuralPoetryGenerator.InitializeVocabulary;
var
  i: Integer;
begin
  // Базовый словарь (можно расширить)
  FVocabulary.AddStrings([
    'любовь', 'море', 'весна', 'мечта', 'сердце', 'ветер', 'ночь',
    'студент', 'сессия', 'экзамен', 'универ', 'лаба', 'препод',
    'технологии', 'код', 'алгоритм', 'нейросеть', 'данные'
  ]);
  
  // Заполняем хеш-таблицу
  for i := 0 to FVocabulary.Count-1 do
    FWordToIndex.Add(FVocabulary[i], TObject(Pointer(i)));
end;

procedure TNeuralPoetryGenerator.InitializeModel;
var
  i, j, gate: Integer;
begin
  Randomize;
  
  // Инициализация эмбеддингов
  SetLength(FEmbeddings, FVocabSize, FEmbeddingSize);
  for i := 0 to High(FEmbeddings) do
    for j := 0 to High(FEmbeddings[i]) do
      FEmbeddings[i,j] := (Random - 0.5) * 0.1;
  
  // Инициализация LSTM (упрощенная версия)
  for gate := 0 to 3 do // 4 гейта LSTM
  begin
    SetLength(FLSTMWeights[gate], FEmbeddingSize, FHiddenSize);
    SetLength(FLSTMUWeights[gate], FHiddenSize, FHiddenSize);
    
    for i := 0 to High(FLSTMWeights[gate]) do
      for j := 0 to High(FLSTMWeights[gate,i]) do
        FLSTMWeights[gate,i,j] := (Random - 0.5) * 0.1;
        
    for i := 0 to High(FLSTMUWeights[gate]) do
      for j := 0 to High(FLSTMUWeights[gate,i]) do
        FLSTMUWeights[gate,i,j] := (Random - 0.5) * 0.1;
  end;
  
  // Инициализация плотного слоя
  SetLength(FDenseWeights, FHiddenSize, FVocabSize);
  for i := 0 to High(FDenseWeights) do
    for j := 0 to High(FDenseWeights[i]) do
      FDenseWeights[i,j] := (Random - 0.5) * 0.1;
end;

function TNeuralPoetryGenerator.LSTMStep(const x, h_prev, c_prev: TVector;
                                       var h_next, c_next: TVector): TVector;
var
  i, j, gate: Integer;
  gates: array[0..3] of TVector;
begin
  SetLength(h_next, FHiddenSize);
  SetLength(c_next, FHiddenSize);
  
  // Вычисляем гейты
  for gate := 0 to 3 do
  begin
    SetLength(gates[gate], FHiddenSize);
    // Упрощенное вычисление без смещений и сигмоид/танх
    for j := 0 to FHiddenSize-1 do
    begin
      gates[gate][j] := 0;
      for i := 0 to High(x) do
        gates[gate][j] := gates[gate][j] + x[i] * FLSTMWeights[gate,i,j];
      for i := 0 to High(h_prev) do
        gates[gate][j] := gates[gate][j] + h_prev[i] * FLSTMUWeights[gate,i,j];
    end;
  end;
  
  // Обновляем состояние
  for i := 0 to FHiddenSize-1 do
  begin
    // Применяем функции активации
    gates[0][i] := 1 / (1 + Exp(-gates[0][i])); // input gate
    gates[1][i] := 1 / (1 + Exp(-gates[1][i])); // forget gate
    gates[2][i] := Tanh(gates[2][i]);           // cell gate
    gates[3][i] := 1 / (1 + Exp(-gates[3][i])); // output gate
    
    c_next[i] := gates[1][i] * c_prev[i] + gates[0][i] * gates[2][i];
    h_next[i] := gates[3][i] * Tanh(c_next[i]);
  end;
  
  Result := h_next;
end;

function TNeuralPoetryGenerator.Softmax(const x: TVector): TVector;
var
  i: Integer;
  maxval, sum: Single;
begin
  SetLength(Result, Length(x));
  maxval := MaxValue(x);
  
  sum := 0;
  for i := 0 to High(x) do
  begin
    Result[i] := Exp(x[i] - maxval); // Для численной стабильности
    sum := sum + Result[i];
  end;
  
  for i := 0 to High(Result) do
    Result[i] := Result[i] / sum;
end;

function TNeuralPoetryGenerator.SampleWord(const probs: TVector): string;
var
  i: Integer;
  r, cumsum: Single;
begin
  r := Random;
  cumsum := 0;
  
  for i := 0 to High(probs) do
  begin
    cumsum := cumsum + probs[i];
    if r <= cumsum then
      Exit(FVocabulary[i]);
  end;
  
  Result := FVocabulary[Random(FVocabulary.Count)];
end;

function TNeuralPoetryGenerator.GeneratePoem(SeedText: string; Style: Integer; PoemLength: Integer): string;
var
  i, word_idx: Integer;
  word: string;
  h, c: TVector;
  x, h_prev, c_prev, output: TVector;
  words: TStringArray;
  j: Integer;
begin
  // Инициализация скрытого состояния
  SetLength(h_prev, FHiddenSize);
  SetLength(c_prev, FHiddenSize);
  for i := 0 to FHiddenSize-1 do
  begin
    h_prev[i] := 0;
    c_prev[i] := 0;
  end;
  
  // Преобразуем начальный текст в последовательность слов
  words := SeedText.Split([' ', ',', '.', '!', '?'], TStringSplitOptions.ExcludeEmpty);
  
  // Пропускаем начальные слова через LSTM
  for word in words do
  begin
    if FWordToIndex.Find(word) <> nil then
      word_idx := PtrInt(FWordToIndex.Find(word))
    else
      word_idx := Random(FVocabSize);
    
    // Получаем эмбеддинг слова
    SetLength(x, FEmbeddingSize);
    for j := 0 to FEmbeddingSize-1 do
      x[j] := FEmbeddings[word_idx, j];
    
    // LSTM шаг
    output := LSTMStep(x, h_prev, c_prev, h, c);
    h_prev := h;
    c_prev := c;
  end;
  
  // Генерация стиха
  Result := '';
  for i := 1 to PoemLength do
  begin
    // Плотный слой
    SetLength(output, FVocabSize);
    for word_idx := 0 to FVocabSize-1 do
    begin
      output[word_idx] := 0;
      for j := 0 to FHiddenSize-1 do
        output[word_idx] := output[word_idx] + h_prev[j] * FDenseWeights[j, word_idx];
    end;
    
    // Выбираем следующее слово
    word := SampleWord(Softmax(output));
    Result := Result + word + ' ';
    
    // Подготавливаем следующий шаг
    if FWordToIndex.Find(word) <> nil then
      word_idx := PtrInt(FWordToIndex.Find(word))
    else
      word_idx := Random(FVocabSize);
    
    SetLength(x, FEmbeddingSize);
    for j := 0 to FEmbeddingSize-1 do
      x[j] := FEmbeddings[word_idx, j];
    
    output := LSTMStep(x, h_prev, c_prev, h, c);
    h_prev := h;
    c_prev := c;
    
    // Добавляем переносы строки для форматирования
    if i mod 4 = 0 then
      Result := Result + #13#10;
  end;
end;

procedure TNeuralPoetryGenerator.TrainOnText(const Text: string);
begin
  // 1. Токенизация текста
  // 2. Обновление эмбеддингов
  // 3. Дообучение LSTM слоев
  // (Реализация зависит от вашей нейросетевой архитектуры)
end;

end.