program NeuralTicTacToe;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


{$mode objfpc}{$H+}

uses
  SysUtils, Math, StrUtils, Classes, NeuralNetwork, DataUtils;

type
  TBoard = array[0..2, 0..2] of Char; // Доска 3x3
  TMove = record
    Row, Col: Integer;
  end;

  TGameState = (gsPlaying, gsXWon, gsOWon, gsDraw);
  TPlayerType = (ptHuman, ptComputer);

const
  EMPTY = '.';
  PLAYER_X = 'X';
  PLAYER_O = 'O';

var
  Board: TBoard;
  CurrentPlayer: Char;
  GameState: TGameState;
  NeuralNet: TNeuralNetwork;
  TrainingData: TDoubleMatrix;
  TrainingLabels: TDoubleArray;
  MoveCount: Integer;

  Experience: Integer = 0; // Счётчик сыгранных игр

// Инициализация доски
procedure InitializeBoard;
var
  i, j: Integer;
begin
  for i := 0 to 2 do
    for j := 0 to 2 do
      Board[i][j] := EMPTY;
  CurrentPlayer := PLAYER_X;
  GameState := gsPlaying;
  MoveCount := 0;
end;

// Преобразование координат шахматного стиля (a1, b2 и т.д.) в индексы массива
function ChessNotationToIndices(const moveStr: string; out row, col: Integer): Boolean;
begin
  Result := False;
  if Length(moveStr) <> 2 then Exit;

  col := Ord(LowerCase(moveStr[1])) - Ord('a');
  row := Ord(moveStr[2]) - Ord('1');

  Result := (row >= 0) and (row <= 2) and (col >= 0) and (col <= 2);
end;

// Преобразование индексов массива в шахматную нотацию
function IndicesToChessNotation(row, col: Integer): string;
begin
  Result := Chr(Ord('a') + col) + Chr(Ord('1') + row);
end;

// Отображение доски
procedure DisplayBoard;
var
  i, j: Integer;
begin
  WriteLn('  a b c');
  for i := 0 to 2 do
  begin
    Write(i+1, ' ');
    for j := 0 to 2 do
    begin
      Write(Board[i][j], ' ');
    end;
    WriteLn;
  end;
end;

// Проверка на победу
function CheckWin: Boolean;
var
  i: Integer;
begin
  // Проверка строк и столбцов
  for i := 0 to 2 do
  begin
    if (Board[i][0] = CurrentPlayer) and (Board[i][1] = CurrentPlayer) and (Board[i][2] = CurrentPlayer) then
      Exit(True);
    if (Board[0][i] = CurrentPlayer) and (Board[1][i] = CurrentPlayer) and (Board[2][i] = CurrentPlayer) then
      Exit(True);
  end;

  // Проверка диагоналей
  if (Board[0][0] = CurrentPlayer) and (Board[1][1] = CurrentPlayer) and (Board[2][2] = CurrentPlayer) then
    Exit(True);
  if (Board[0][2] = CurrentPlayer) and (Board[1][1] = CurrentPlayer) and (Board[2][0] = CurrentPlayer) then
    Exit(True);

  Result := False;
end;

// Проверка на ничью
function CheckDraw: Boolean;
var
  i, j: Integer;
begin
  for i := 0 to 2 do
    for j := 0 to 2 do
      if Board[i][j] = EMPTY then
        Exit(False);
  Exit(True);
end;

// Преобразование доски в вектор для нейросети
function BoardToVector: TDoubleArray;
var
  i, j, idx: Integer;
begin
  SetLength(Result, 9);
  idx := 0;
  for i := 0 to 2 do
    for j := 0 to 2 do
    begin
      if Board[i][j] = PLAYER_X then
        Result[idx] := 1.0
      else if Board[i][j] = PLAYER_O then
        Result[idx] := -1.0
      else
        Result[idx] := 0.0;
      Inc(idx);
    end;
end;

// Защищенная сигмоида с проверкой на переполнение
function SafeSigmoid(x: Double): Double;
begin
  if x > 30 then Exit(1.0);
  if x < -30 then Exit(0.0);
  Result := 1.0 / (1.0 + Exp(-x));
end;

function CheckWinForBoard(const board: TBoard; player: Char): Boolean;
var
  i: Integer;
begin
  // Проверка строк и столбцов
  for i := 0 to 2 do
  begin
    if (board[i][0] = player) and (board[i][1] = player) and (board[i][2] = player) then
      Exit(True);
    if (board[0][i] = player) and (board[1][i] = player) and (board[2][i] = player) then
      Exit(True);
  end;

  // Проверка диагоналей
  if (board[0][0] = player) and (board[1][1] = player) and (board[2][2] = player) then
    Exit(True);
  if (board[0][2] = player) and (board[1][1] = player) and (board[2][0] = player) then
    Exit(True);

  Result := False;
end;

function FindWinningMove(player: Char): Integer;
var
  i, r, c: Integer;
  testBoard: TBoard;
begin
  Result := -1;
  for i := 0 to 8 do
  begin
    r := i div 3;
    c := i mod 3;
    if Board[r][c] = EMPTY then
    begin
      testBoard := Board;
      testBoard[r][c] := player;
      if CheckWinForBoard(testBoard, player) then
        Exit(i);
    end;
  end;
end;

procedure ApplyStrategicWeights(var output: TDoubleArray);
begin
  // Центр более важен
  output[4] := output[4] * 1.3;
  
  // Углы важнее при определённых условиях
  if (Board[0][0] = EMPTY) or (Board[0][2] = EMPTY) or 
     (Board[2][0] = EMPTY) or (Board[2][2] = EMPTY) then
  begin
    output[0] := output[0] * 1.2;
    output[2] := output[2] * 1.2;
    output[6] := output[6] * 1.2;
    output[8] := output[8] * 1.2;
  end;
end;

// Выбор хода нейросетью
function NeuralNetMove: TMove;
var
  inputVector, outputVector: TDoubleArray;
  i, bestMove: Integer;
  bestScore: Double;
  validMoves: array of Integer;
begin
  inputVector := BoardToVector;
  
  // Добавляем стратегическую логику
  if Experience > 30 then
  begin
    // Проверка на выигрышный ход
    bestMove := FindWinningMove(CurrentPlayer);
    if bestMove >= 0 then
    begin
      Result.Row := bestMove div 3;
      Result.Col := bestMove mod 3;
      Exit;
    end;
    
    // Блокировка выигрыша противника
    if CurrentPlayer = PLAYER_O then
    begin
      bestMove := FindWinningMove(PLAYER_X);
      if bestMove >= 0 then
      begin
        Result.Row := bestMove div 3;
        Result.Col := bestMove mod 3;
        Exit;
      end;
    end;
  end;

  // Основной интеллектуальный ход
  try
    outputVector := PredictNetwork(NeuralNet, inputVector);
    
    // Применяем стратегические приоритеты
    ApplyStrategicWeights(outputVector);
    
    bestMove := -1;
    bestScore := -Infinity;
    for i := 0 to 8 do
    begin
      if (Board[i div 3][i mod 3] = EMPTY) and (outputVector[i] > bestScore) then
      begin
        bestScore := outputVector[i];
        bestMove := i;
      end;
    end;
    
    if bestMove >= 0 then
    begin
      Result.Row := bestMove div 3;
      Result.Col := bestMove mod 3;
      Exit;
    end;
  except
    on E: Exception do
      WriteLn('Ошибка предсказания: ', E.Message);
  end;
  
  // Резервный случайный ход
  SetLength(validMoves, 0);
  for i := 0 to 8 do
    if Board[i div 3][i mod 3] = EMPTY then
    begin
      SetLength(validMoves, Length(validMoves)+1);
      validMoves[High(validMoves)] := i;
    end;
    
  if Length(validMoves) > 0 then
  begin
    bestMove := validMoves[Random(Length(validMoves))];
    Result.Row := bestMove div 3;
    Result.Col := bestMove mod 3;
  end
  else
  begin
    Result.Row := 0;
    Result.Col := 0;
  end;
end;

// Добавление хода в обучающую выборку
procedure AddToTrainingData(const move: TMove);
var
  inputVector: TDoubleArray;
  outputVector: TDoubleArray;
  moveIdx: Integer;
begin
  inputVector := BoardToVector;
  SetLength(outputVector, 9);
  FillChar(outputVector[0], Length(outputVector) * SizeOf(Double), 0);
  
  moveIdx := move.Row * 3 + move.Col;
  outputVector[moveIdx] := 1.0;
  
  // Добавляем в обучающую выборку
  SetLength(TrainingData, Length(TrainingData) + 1);
  TrainingData[High(TrainingData)] := inputVector;
  
  SetLength(TrainingLabels, Length(TrainingLabels) + 1);
  TrainingLabels[High(TrainingLabels)] := moveIdx;
end;

// Обучение нейросети на накопленных данных
procedure TrainNeuralNet;
var
  effectiveLearningRate: Double;
begin
  if Length(TrainingData) > 0 then
  begin
    try
      // Динамический learning rate (уменьшается с опытом)
      effectiveLearningRate := 0.1 / (1.0 + Experience * 0.01);
      NeuralNet.learningRate := Max(effectiveLearningRate, 0.001);
      
      // Увеличиваем количество эпох с опытом
      TrainNetwork(NeuralNet, TrainingData, TrainingLabels, 5 + Experience div 10, Experience);
      
      Inc(Experience);
      WriteLn('Опыт компьютера: ', Experience);
    except
      on E: Exception do
        WriteLn('Ошибка обучения: ', E.Message);
    end;
    
    SetLength(TrainingData, 0);
    SetLength(TrainingLabels, 0);
  end;
end;

// Ход игрока
procedure PlayerMove;
var
  moveStr: string;
  row, col: Integer;
  validMove: Boolean;
Move: TMove;
begin
  repeat
    Write('Ваш ход (например, a1, b2 и т.д.): ');
    ReadLn(moveStr);
    
    validMove := ChessNotationToIndices(moveStr, row, col);
    if not validMove then
    begin
      WriteLn('Некорректный ввод. Используйте букву от a до c и цифру от 1 до 3 (например, a1).');
      Continue;
    end;
    
    if Board[row][col] <> EMPTY then
    begin
      WriteLn('Эта клетка уже занята!');
      validMove := False;
    end;
  until validMove;
  
  Board[row][col] := CurrentPlayer;

Move.Row := row; Move.Col := col;
  AddToTrainingData(Move); // Запоминаем ход игрока для обучения
end;

// Ход компьютера
procedure ComputerMove;
var
  move: TMove;
begin
  WriteLn('Компьютер думает...');
  move := NeuralNetMove;
  Board[move.Row][move.Col] := CurrentPlayer;
  WriteLn('Компьютер сделал ход: ', IndicesToChessNotation(move.Row, move.Col));
end;

// Основной игровой цикл
procedure PlayGame;
var
  response: Char;
begin
  InitializeBoard;
  DisplayBoard;
  
  while GameState = gsPlaying do
  begin
    if CurrentPlayer = PLAYER_X then
      PlayerMove
    else
      ComputerMove;
    
    Inc(MoveCount);
    DisplayBoard;
    
    if CheckWin then
    begin
      if CurrentPlayer = PLAYER_X then
        GameState := gsXWon
      else
        GameState := gsOWon;
      WriteLn('Игрок ', CurrentPlayer, ' победил!');
    end
    else if CheckDraw then
    begin
      GameState := gsDraw;
      WriteLn('Ничья!');
    end;
    
    if CurrentPlayer = PLAYER_X then
      CurrentPlayer := PLAYER_O
    else
      CurrentPlayer := PLAYER_X;
  end;
  
  // После игры обучаем нейросеть на новых данных
  TrainNeuralNet;
end;

// Главная программа
var
  playAgain: Char;
begin
  Randomize;
  
  // Инициализация нейросети (3 слоя: 9 входов, 18 скрытых нейронов, 9 выходов)
  SafeInitializeNetwork(NeuralNet, [9, 18, 9], 0.01, 0.001);
  
  WriteLn('Нейросетевая игра "Крестики-нолики"');
  WriteLn('Используйте шахматную нотацию для ходов (a1, b2, c3 и т.д.)');
  WriteLn('Вы играете за X, компьютер за O');
  
  repeat
    PlayGame;
    Write('Хотите сыграть ещё раз? (y/n): ');
    ReadLn(playAgain);
  until LowerCase(playAgain) <> 'y';
  
  FreeNetwork(NeuralNet);
  WriteLn('До свидания!');
end.