unit PGNUtil;
{$MODE OBJFPC}{$H+}

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


interface

uses classes, sysutils, NeuralChessCore;

function PGNToMove(const pgnMove: String; var board: TChessBoard): TMove;
procedure ExtractMovesFromPGN(const gameText: String; moves: TStringList);

function FindPieceForMove(const board: TChessBoard; pieceType: TPieceType; color: TPieceColor; var move: TMove): Boolean;
function TryPGNToMove(const pgnMove: String; const board: TChessBoard; out move: TMove): Boolean;

implementation

function GetCurrentPlayerColor(const board: TChessBoard): TPieceColor;
// Определяет, чей сейчас ход (можно считать по количеству фигур или последнему ходу)
begin
  // Упрощённая реализация - нужно доработать
  Result := pcWhite;
end;

function IsMoveValid(const board: TChessBoard; const move: TMove): Boolean;
// Проверяет, разрешён ли такой ход правилами шахмат
begin
  // Реализация проверки правил для каждой фигуры
  Result := True;
end;

function PGNToMove(const pgnMove: String; var board: TChessBoard): TMove;
var
  s: String;
  promo: Char;
begin
  Result.FromX := -1;
  Result.FromY := -1;
  Result.IsEnPassant := False;
  Result.IsCastling := False;
  Result.Promotion := ptNone;

  s := StringReplace(pgnMove, '+', '', [rfReplaceAll]);
  s := StringReplace(s, '#', '', [rfReplaceAll]);
  s := StringReplace(s, 'x', '', [rfReplaceAll]);

  // Обработка рокировки
  if s = 'O-O' then
  begin
    Result.IsCastling := True;
    // Определяем цвет для рокировки
    if board[4, 0].PieceType = ptKing then // Белые
    begin
      Result.FromX := 4; Result.FromY := 0;
      Result.ToX := 6; Result.ToY := 0;
    end
    else // Черные
    begin
      Result.FromX := 4; Result.FromY := 7;
      Result.ToX := 6; Result.ToY := 7;
    end;
    Exit;
  end
  else if s = 'O-O-O' then
  begin
    Result.IsCastling := True;
    // Определяем цвет для рокировки
    if board[4, 0].PieceType = ptKing then // Белые
    begin
      Result.FromX := 4; Result.FromY := 0;
      Result.ToX := 2; Result.ToY := 0;
    end
    else // Черные
    begin
      Result.FromX := 4; Result.FromY := 7;
      Result.ToX := 2; Result.ToY := 7;
    end;
    Exit;
  end;

  // Обработка превращения пешки
  if Pos('=', s) > 0 then
  begin
    promo := s[Pos('=', s) + 1];
    case promo of
      'Q': Result.Promotion := ptQueen;
      'R': Result.Promotion := ptRook;
      'B': Result.Promotion := ptBishop;
      'N': Result.Promotion := ptKnight;
    end;
    s := Copy(s, 1, Pos('=', s) - 1);
  end;

  // Реализация парсинга обычных ходов...
  // (дополнительный код для обработки e4, Nf3, Qxd2 и т.д.)
end;

procedure ExtractMovesFromPGN(const gameText: String; moves: TStringList);
var
  i, start: Integer;
  s: String;
begin
  s := Trim(gameText);
  // Удаляем номера ходов (1. e4 e5 2. Nf3 Nc6 -> e4 e5 Nf3 Nc6)
  i := 1;
  while i <= Length(s) do
  begin
    if (s[i] in ['0'..'9']) and (i < Length(s)) and (s[i+1] = '.') then
    begin
      Delete(s, i, 2);
      while (i <= Length(s)) and (s[i] = ' ') do
        Delete(s, i, 1);
    end
    else
      Inc(i);
  end;
  
  // Разбиваем на отдельные ходы
  moves.Delimiter := ' ';
  moves.DelimitedText := s;
end;

function TryPGNToMove(const pgnMove: String; const board: TChessBoard; out move: TMove): Boolean;
var
  s: String;
  promoChar: Char;
  fromX, fromY, toX, toY: Integer;
  pieceType: TPieceType;
  color: TPieceColor;
begin
  Result := False;
  FillChar(move, SizeOf(move), 0);
  move.FromX := -1;
  move.FromY := -1;
  move.Promotion := ptNone;
  move.IsEnPassant := False;
  move.IsCastling := False;

  // Упрощаем запись хода
  s := StringReplace(pgnMove, '+', '', [rfReplaceAll]);
  s := StringReplace(s, '#', '', [rfReplaceAll]);
  s := StringReplace(s, 'x', '', [rfReplaceAll]);
  s := Trim(s);

  // Пропускаем комментарии и нумерацию
  if (s = '') or (s[1] = '{') or (Pos('.', s) = 1) then Exit(False);

  // Определяем цвет фигуры по текущему положению
  color := GetCurrentPlayerColor(board); // Нужно реализовать эту функцию

  // Обработка рокировки
  if s = 'O-O' then
  begin
    move.IsCastling := True;
    if color = pcWhite then
    begin
      move.FromX := 4; move.FromY := 0;
      move.ToX := 6; move.ToY := 0;
    end
    else
    begin
      move.FromX := 4; move.FromY := 7;
      move.ToX := 6; move.ToY := 7;
    end;
    Exit(True);
  end
  else if s = 'O-O-O' then
  begin
    move.IsCastling := True;
    if color = pcWhite then
    begin
      move.FromX := 4; move.FromY := 0;
      move.ToX := 2; move.ToY := 0;
    end
    else
    begin
      move.FromX := 4; move.FromY := 7;
      move.ToX := 2; move.ToY := 7;
    end;
    Exit(True);
  end;

  // Обработка превращения пешки
  if Pos('=', s) > 0 then
  begin
    promoChar := UpCase(s[Pos('=', s) + 1]);
    case promoChar of
      'Q': move.Promotion := ptQueen;
      'R': move.Promotion := ptRook;
      'B': move.Promotion := ptBishop;
      'N': move.Promotion := ptKnight;
    else
      Exit(False);
    end;
    s := Copy(s, 1, Pos('=', s) - 1);
  end;

  // Парсим обычные ходы (формат: e2e4, Nf3, Qa5xd2 и т.д.)
  if Length(s) >= 2 then
  begin
    // Определяем тип фигуры (по умолчанию пешка)
    pieceType := ptPawn;
    if s[1] in ['K','Q','R','B','N'] then
    begin
      case s[1] of
        'K': pieceType := ptKing;
        'Q': pieceType := ptQueen;
        'R': pieceType := ptRook;
        'B': pieceType := ptBishop;
        'N': pieceType := ptKnight;
      end;
      Delete(s, 1, 1); // Удаляем идентификатор фигуры
    end;

    // Парсим целевую позицию
    if Length(s) >= 2 then
    begin
      toX := Ord(LowerCase(s[Length(s)-1])) - Ord('a');
      toY := Ord(s[Length(s)]) - Ord('1');
      if not ((toX in [0..7]) and (toY in [0..7])) then Exit(False);
      move.ToX := toX;
      move.ToY := toY;
      Delete(s, Length(s)-1, 2); // Удаляем целевую позицию
    end
    else
      Exit(False);

    // Парсим исходную позицию (если указана)
    if Length(s) >= 2 then
    begin
      fromX := Ord(LowerCase(s[Length(s)-1])) - Ord('a');
      fromY := Ord(s[Length(s)]) - Ord('1');
      if not ((fromX in [0..7]) and (fromY in [0..7])) then Exit(False);
      move.FromX := fromX;
      move.FromY := fromY;
    end
    else
    begin
      // Если исходная позиция не указана, находим фигуру, которая может сделать этот ход
      if not FindPieceForMove(board, pieceType, color, move) then
        Exit(False);
    end;
  end
  else
    Exit(False);

  // Проверяем, соответствует ли фигура на доске ожидаемой
  if (board[move.FromX, move.FromY].PieceType <> pieceType) or
     (board[move.FromX, move.FromY].Color <> color) then
    Exit(False);

  // Проверяем валидность хода
  Result := True;
end;

function FindPieceForMove(const board: TChessBoard; pieceType: TPieceType; 
  color: TPieceColor; var move: TMove): Boolean;
var
  x, y: Integer;
  tmpMove: TMove;
begin
  Result := False;
  for x := 0 to 7 do
    for y := 0 to 7 do
      if (board[x,y].PieceType = pieceType) and (board[x,y].Color = color) then
      begin
        tmpMove := move;
        tmpMove.FromX := x;
        tmpMove.FromY := y;
        if IsMoveValid(board, tmpMove) then
        begin
          move.FromX := x;
          move.FromY := y;
          Exit(True);
        end;
      end;
end;

end.