unit PoetryPlugin;

{
    Part of AdvancedChatAI.
    For GNU/Linux 64 bit version.
    Version: 1.
    Written on FreePascal (https://freepascal.org/).
    Copyright (C) 2025-2026 Artyomov Alexander
    Used https://chat.deepseek.com/
    http://self-made-free.ru/
    aralni@mail.ru

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
}


{$MODE OBJFPC}{$H+}

interface

uses
  PluginInterface, SysUtils, Classes, MarkovChainUnit, NeuralPoetry,StrUtils;

type
  TPoemStyle = (psRomantic, psHumorous, psPhilosophical, psLively, psCozy);
  
  TPoetryPlugin = class(TInterfacedObject, IPlugin)
  private
    FInitialized: Boolean;
    FPoemTemplates: array[TPoemStyle] of TStringList;
    FThemes: TStringList;
    FMarkovChains: array[TPoemStyle] of TMarkovChain;
    FNeuralGenerator: TNeuralPoetryGenerator;

    procedure InitializeComponents;
    procedure InitializeTemplates;
    procedure InitializeMarkovChains;
    function DetectStyle(const Input: string): TPoemStyle;
    function ExtractTheme(const Input: string): string;
    function FixGrammar(const s: string): string;
    function GeneratePoem(Style: TPoemStyle; const Theme: string): string;
    function FormatPoem(const Text: string): string;
  public
    constructor Create;
    destructor Destroy; override;
    function CanHandle(const Input: string): Boolean;
    function HandleInput(const Input: string): string;
    function GetName: string;
    procedure LoadPoetryCorpus(const FileName: string);
    procedure TrainModelsOnPoem(const PoemText: string);
    procedure RetrainAllModels;
    procedure LoadDefaultPoems;
  end;

function DetectPoemStyle(const PoemText: string): TPoemStyle;

implementation

constructor TPoetryPlugin.Create;
begin
  inherited;
  FInitialized := False;
  try
    InitializeComponents;
    LoadDefaultPoems;
    FInitialized := True;
  except
    on E: Exception do
      WriteLn('[PoetryPlugin] Ошибка инициализации: ', E.Message);
  end;
end;

procedure TPoetryPlugin.InitializeComponents;
var
  style: TPoemStyle;
begin
  for style := Low(TPoemStyle) to High(TPoemStyle) do
  begin
    FPoemTemplates[style] := TStringList.Create;
    FMarkovChains[style] := TMarkovChain.Create(2);
  end;
  FThemes := TStringList.Create;
  FThemes.AddStrings(['любовь', 'природа', 'город', 'технологии']);
end;

{
procedure TPoetryPlugin.LoadDefaultPoems;
begin
  // Романтические стихи
  FPoemTemplates[psRomantic].Add(
    'Любовь - как свежий ветер мая,'#13#10 +
    'Как первый солнечный луч,'#13#10 +
    'Без неё жизнь - пустая,'#13#10 +
    'Лишь холодный скользкий ключ.');

  // Юмористические стихи
  FPoemTemplates[psHumorous].Add(
    'Любовь пришла внезапно,'#13#10 +
    'Как вирус в компьютере,'#13#10 +
    'Теперь сижу печальный,'#13#10 +
    'С антивирусом в квартире.');

  FPoemTemplates[psHumorous].Add(
    'Люблю тебя, как лабу любят студенты,'#13#10 +
    'Как препод любит пары по утрам,'#13#10 +
    'Как любят котики мять документы,'#13#10 +
    'Как я люблю писать код по ночам!');
    
  // Обучаем цепи Маркова
  FMarkovChains[psRomantic].Learn('любовь светла прекрасна');
  FMarkovChains[psRomantic].Learn('любовь горяча как пламя');
end;
}
procedure TPoetryPlugin.LoadDefaultPoems;
begin
  FPoemTemplates[psRomantic].Add(
    'Любовь - как свежий ветер мая,'#13#10 +
    'Как первый солнечный луч,'#13#10 +
    'Без неё жизнь - пустая,'#13#10 +
    'Лишь холодный скользкий ключ.');
    
  FPoemTemplates[psHumorous].Add(
    'Люблю тебя, как лабу любят студенты,'#13#10 +
    'Как препод любит пары по утрам,'#13#10 +
    'Как любят котики мять документы,'#13#10 +
    'Как я люблю писать код по ночам!');
end;

destructor TPoetryPlugin.Destroy;
var
  style: TPoemStyle;
begin
  for style := Low(TPoemStyle) to High(TPoemStyle) do
  begin
    if Assigned(FPoemTemplates[style]) then
      FPoemTemplates[style].Free;
    if Assigned(FMarkovChains[style]) then
      FMarkovChains[style].Free;
  end;
  
  if Assigned(FThemes) then
    FThemes.Free;
    
  inherited;
end;

procedure TPoetryPlugin.InitializeTemplates;
begin
  FPoemTemplates[psRomantic] := TStringList.Create;
  FPoemTemplates[psHumorous] := TStringList.Create;
  FPoemTemplates[psPhilosophical] := TStringList.Create;
  FPoemTemplates[psLively] := TStringList.Create;
  FPoemTemplates[psCozy] := TStringList.Create;

  with FPoemTemplates[psRomantic] do
  begin
    Add('О {theme}, ты так прекрасна,'#13#10'Как лунный свет в ночи безгласной,'#13#10'Твой образ в сердце навсегда,'#13#10'Моя любовь к тебе чиста!');
    Add('Как сладко пахнет {theme} весной,'#13#10'Когда весь мир наполнен теплотой,'#13#10'Так и в душе моей цветет,'#13#10'Любовь к {theme} меня зовет!');
  end;

  with FPoemTemplates[psHumorous] do
  begin
//    Add('Жил-был {theme}, вот чудак,'#13#10'Весь день лежал он на бочку,'#13#10'А когда спросили "Почему?"'#13#10'Ответил: "Так хочу!"');
    Add('Наши {theme} - просто клоуны,'#13#10'Вечно ходят с важным видом,'#13#10'Но стоит лишь присмотреться -'#13#10'Видно, что они несерьёзные!');
  end;

  with FPoemTemplates[psPhilosophical] do
  begin
    Add('Что есть {theme}? Не просто вещь,'#13#10'А целый мир, где есть и спесь,'#13#10'Где каждый сам себе кузнец,'#13#10'Своих желаний, наконец!');
    Add('{Theme} - это путь,'#13#10'Который каждый должен пройти,'#13#10'И в этом смысл бытия -'#13#10'Познать всю суть {theme}!');
  end;
end;

procedure TPoetryPlugin.InitializeMarkovChains;
var
  style: TPoemStyle;
begin
  for style := Low(TPoemStyle) to High(TPoemStyle) do
  begin
    FMarkovChains[style] := TMarkovChain.Create(2);
    
    // Добавляем базовые примеры для обучения
    case style of
      psRomantic:
        FMarkovChains[style].Learn('любовь прекрасна как рассвет');
      psHumorous:
        FMarkovChains[style].Learn('студенты вечно сдают экзамены');
      psPhilosophical:
        FMarkovChains[style].Learn('жизнь это путь без конца');
    end;
  end;
end;

{
function TPoetryPlugin.GeneratePoem(Style: TPoemStyle; const Theme: string): string;
var
  Words: TStringArray;
  TempList: TStringList;
  GenerationMethod: Integer;
  i: Integer;
  SeedText: string; // Добавлено для объединения слов в строку
begin
  // Выбираем метод генерации (0-шаблоны, 1-Марков, 2-нейросеть)
  GenerationMethod := Random(3);
  
  // Подготовка слов темы
  TempList := TStringList.Create;
  try
    TempList.Delimiter := ' ';
    TempList.DelimitedText := StringReplace(
      StringReplace(Theme, ',', ' ', [rfReplaceAll]),
      '.', ' ', [rfReplaceAll]);
      
    for i := TempList.Count - 1 downto 0 do
      if Trim(TempList[i]) = '' then TempList.Delete(i);
      
    SetLength(Words, TempList.Count);
    for i := 0 to TempList.Count - 1 do
      Words[i] := TempList[i];

    // Объединяем слова в строку для нейросетевого генератора
    SeedText := '';
    for i := 0 to High(Words) do
    begin
      if i > 0 then SeedText := SeedText + ' ';
      SeedText := SeedText + Words[i];
    end;
  finally
    TempList.Free;
  end;

  if Length(Words) = 0 then
    Words := FThemes.ToStringArray;

  case GenerationMethod of
    0: // Шаблоны
      begin
        if FPoemTemplates[Style].Count > 0 then
        begin
          Result := FPoemTemplates[Style][Random(FPoemTemplates[Style].Count)];
          Result := StringReplace(Result, '{theme}', Theme, [rfReplaceAll, rfIgnoreCase]);
        end
        else
          Result := '';
      end;
      
    1: // Цепи Маркова
      begin
        if (FMarkovChains[Style] <> nil) and (FMarkovChains[Style].StateCount > 0) then
        begin
          Result := '';
          for i := 0 to 3 do
          begin
            if Result <> '' then Result := Result + #13#10;
            Result := Result + FMarkovChains[Style].GenerateSentence(
              Words[Random(Length(Words))], 
              5 + Random(3)
            );
          end;
        end
        else
          Result := '';
      end;
      
    2: // Нейросеть
      begin
        Result := FNeuralGenerator.GeneratePoem(SeedText, Ord(Style)); // Исправлено: передаем строку вместо массива
      end;
  end;

  // Если генерация не удалась, используем шаблоны
  if Result = '' then
  begin
    if FPoemTemplates[Style].Count > 0 then
    begin
      Result := FPoemTemplates[Style][Random(FPoemTemplates[Style].Count)];
      Result := StringReplace(Result, '{theme}', Theme, [rfReplaceAll, rfIgnoreCase]);
    end
    else
      Result := 'Извините, не могу создать стихотворение';
  end;

  Result := FixGrammar(Result);
  
  // Добавляем стиль
  case Style of
    psRomantic: Result := 'Романтическое стихотворение:'#13#10 + Result;
    psHumorous: Result := 'Юмористическое стихотворение:'#13#10 + Result;
    psPhilosophical: Result := 'Философское размышление:'#13#10 + Result;
    psLively: Result := 'Энергичный стих:'#13#10 + Result;
    psCozy: Result := 'Уютное стихотворение:'#13#10 + Result;
  end;
end;
}
function TPoetryPlugin.GeneratePoem(Style: TPoemStyle; const Theme: string): string;
const
  MIN_WORDS = 6;  // Минимальное количество слов в стихе
  MAX_ATTEMPTS = 5; // Максимальное количество попыток
var
  Words: TStringArray;
  Attempts, WordCount: Integer;
begin
  // Разбиваем тему на слова
  Words := SplitString(Trim(Theme), ' ');
  if Length(Words) = 0 then
    Words := FThemes.ToStringArray;

  for Attempts := 1 to MAX_ATTEMPTS do
  begin
    // Генерируем через Марковскую цепь
    Result := Trim(FMarkovChains[Style].GenerateSentence(
      Words[Random(Length(Words))], 
      10 + Random(6) // Длина строки
    ));

    // Проверяем качество
    WordCount := Length(Result.Split([' ']));
    if (WordCount >= MIN_WORDS) and 
       (Pos('  ', Result) = 0) and
       (Length(Result) > 15) then
    begin
      // Форматируем с переносами строк
      Result := FormatPoem(Result);
      Exit;
    end;
  end;
  
  Result := 'Извините, не могу создать хороший стих прямо сейчас. ' +
            'Попробуйте задать более конкретную тему.';
end;

function TPoetryPlugin.FormatPoem(const Text: string): string;
var
  Words: TStringArray;
  i, LineLen: Integer;
begin
  Words := Text.Split([' ']);
  Result := '';
  LineLen := 0;
  
  for i := 0 to High(Words) do
  begin
    if LineLen + Length(Words[i]) > 15 then
    begin
      Result := Result + #13#10;
      LineLen := 0;
    end
    else if i > 0 then
    begin
      Result := Result + ' ';
      Inc(LineLen);
    end;
    
    Result := Result + Words[i];
    Inc(LineLen, Length(Words[i]));
  end;
end;

function TPoetryPlugin.CanHandle(const Input: string): Boolean;
begin
  Result := FInitialized and (
    ContainsText(LowerCase(Input), 'стих') or 
    ContainsText(LowerCase(Input), 'поэ') or
    ContainsText(LowerCase(Input), 'рифм')
  );
end;

{
function TPoetryPlugin.HandleInput(const Input: string): string;
var
  LowerInput, Theme: string;
  Style: TPoemStyle;
begin
  LowerInput := LowerCase(Trim(Input));
  
  // Определяем тему
  if ContainsText(LowerInput, 'о любви') then
    Theme := 'любовь'
  else if ContainsText(LowerInput, 'о природе') then
    Theme := 'природа'
  else
    Theme := ExtractTheme(LowerInput);

  // Определяем стиль
  if ContainsText(LowerInput, 'шутк') or ContainsText(LowerInput, 'юмор') then
    Style := psHumorous
  else
    Style := DetectStyle(LowerInput);

  // Генерируем стих
  Result := GeneratePoem(Style, Theme);
end;
}
function TPoetryPlugin.HandleInput(const Input: string): string;
begin
  if not FInitialized then
    Exit('Поэтический модуль не инициализирован');

  if ContainsText(LowerCase(Input), 'о любви') then
    Result := 'Любовь - как свежий ветер мая,'#13#10'Как первый солнечный луч'
  else
    Result := GeneratePoem(psRomantic, ExtractTheme(Input));
end;

function TPoetryPlugin.GetName: string;
begin
  Result := 'Universal Poetry Plugin v2.3';
end;

function TPoetryPlugin.DetectStyle(const Input: string): TPoemStyle;
var
  LowerInput: string;
begin
  LowerInput := LowerCase(Input);
  
  if ContainsText(LowerInput, 'романт') or ContainsText(LowerInput, 'любов') then
    Result := psRomantic
  else if ContainsText(LowerInput, 'юмор') or ContainsText(LowerInput, 'смешн') then
    Result := psHumorous
  else if ContainsText(LowerInput, 'философ') then
    Result := psPhilosophical
  else if ContainsText(LowerInput, 'энерг') then
    Result := psLively
  else
    Result := psCozy;
end;

function TPoetryPlugin.ExtractTheme(const Input: string): string;
var
  LowerInput: string;
  i: Integer;
begin
  Result := 'жизнь';
  LowerInput := LowerCase(Input);

  for i := 0 to FThemes.Count - 1 do
    if ContainsText(LowerInput, FThemes[i]) then
      Exit(FThemes[i]);

  if ContainsText(LowerInput, 'про ') then
    Result := Copy(LowerInput, Pos('про ', LowerInput) + 4, Length(LowerInput))
  else if ContainsText(LowerInput, 'о ') then
    Result := Copy(LowerInput, Pos('о ', LowerInput) + 2, Length(LowerInput));

  Result := Trim(Result);
end;

function TPoetryPlugin.FixGrammar(const s: string): string;
var
  LowerS: string;
begin
  Result := s;
  LowerS := LowerCase(Result);
  Result := StringReplace(Result, 'Наши о студентах', 'Наши студенты', [rfIgnoreCase]);
  Result := StringReplace(Result, ' о студентах', ' студенты', [rfIgnoreCase]);

  // Исправляем все формы слова "студенты"
  Result := StringReplace(Result, 'студентах', 'студенты', [rfReplaceAll, rfIgnoreCase]);
  Result := StringReplace(Result, 'студентов', 'студенты', [rfReplaceAll, rfIgnoreCase]);
  
  // Другие грамматические исправления
  Result := StringReplace(Result, 'к море', 'к морю', [rfReplaceAll, rfIgnoreCase]);
  Result := StringReplace(Result, 'технологиях?', 'технологии?', [rfReplaceAll, rfIgnoreCase]);

  // Исправляем "о студентов" -> "о студентах"
  if Pos('о студентов', LowerS) > 0 then
    Result := StringReplace(Result, 'о студентов', 'о студентах', [rfIgnoreCase]);

  // Исправляем "технологиях?" -> "технологии?"
  if Pos('технологиях?', LowerS) > 0 then
    Result := StringReplace(Result, 'технологиях?', 'технологии?', [rfIgnoreCase]);

  // Исправляем "наши о студентах" -> "наши студенты"
  if Pos('наши о студентах', LowerS) > 0 then
    Result := StringReplace(Result, 'наши о студентах', 'наши студенты', [rfIgnoreCase]);

  // Исправляем "к морю" -> "к морю" (уже правильно)
  if Pos('к море', LowerS) > 0 then
    Result := StringReplace(Result, 'к море', 'к морю', [rfIgnoreCase]);
end;

procedure TPoetryPlugin.LoadPoetryCorpus(const FileName: string);
var
  PoetryFile: TStringList;
  i: Integer;
  PoemText: string;
begin
  if not FileExists(FileName) then Exit;

  PoetryFile := TStringList.Create;
  try
    PoetryFile.LoadFromFile(FileName);
    
    // Обработка каждого стихотворения в файле
    PoemText := '';
    for i := 0 to PoetryFile.Count - 1 do
    begin
      if Trim(PoetryFile[i]) = '' then
      begin
        if PoemText <> '' then
        begin
          // Обучаем все модели на новом стихе
          TrainModelsOnPoem(PoemText);
          PoemText := '';
        end;
      end
      else
      begin
        if PoemText <> '' then PoemText := PoemText + #13#10;
        PoemText := PoemText + PoetryFile[i];
      end;
    end;
    
    // Обучаем на последнем стихе в файле
    if PoemText <> '' then
      TrainModelsOnPoem(PoemText);
  finally
    PoetryFile.Free;
  end;
end;

procedure TPoetryPlugin.TrainModelsOnPoem(const PoemText: string);
var
  style: TPoemStyle;
begin
  // Обучаем цепи Маркова для каждого стиля
  for style := Low(TPoemStyle) to High(TPoemStyle) do
    if Assigned(FMarkovChains[style]) then
      FMarkovChains[style].Learn(PoemText);
  
  // Добавляем в нейросетевой генератор
  if Assigned(FNeuralGenerator) then
    FNeuralGenerator.TrainOnText(PoemText);
end;

procedure TPoetryPlugin.RetrainAllModels;
var
  style: TPoemStyle;
begin
  for style := Low(TPoemStyle) to High(TPoemStyle) do
    if Assigned(FMarkovChains[style]) then
      FMarkovChains[style].Retrain;
end;

function DetectPoemStyle(const PoemText: string): TPoemStyle;
var
  Keywords: array[TPoemStyle] of TStringArray;
  i: Integer;
  LowerText: string;
begin
  LowerText := LowerCase(PoemText);
  
  // Ключевые слова для каждого стиля
  Keywords[psRomantic] := ['любовь', 'мечта', 'сердце'];
  Keywords[psHumorous] := ['смех', 'шутк', 'прикол'];
  // ... и т.д.
  
  // Анализ текста
  for i := 0 to High(Keywords[psHumorous]) do
    if ContainsText(LowerText, Keywords[psHumorous][i]) then
      Exit(psHumorous);
  
  // По умолчанию
  Result := psRomantic;
end;

end.